/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.util.Random;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.core.AnimatableBoolean;
import ch.kuramo.javie.core.AnimatableDouble;
import ch.kuramo.javie.core.AnimatableInteger;
import ch.kuramo.javie.core.CoreContext;
import ch.kuramo.javie.core.ExpressionScope;
import ch.kuramo.javie.core.TARangeSelector;
import ch.kuramo.javie.core.annotations.ProjectElement;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;

@ProjectElement("TARangeSelector")
public class TARangeSelectorImpl extends AbstractTASelector implements TARangeSelector {

	private AnimatableDouble start = new AnimatableDouble(0d, 0d, 100d);

	private AnimatableDouble end = new AnimatableDouble(100d, 0d, 100d);

	private AnimatableDouble offset = new AnimatableDouble(0d, -100d, 100d);

	private AnimatableMode mode = new AnimatableMode(Mode.ADD);

	private AnimatableDouble amount = new AnimatableDouble(100d, -100d, 100d);

	private Shape shape = Shape.SQUARE;

	private AnimatableDouble squareSmoothness = new AnimatableDouble(100d, 0d, 100d);

	private AnimatableDouble easeHigh = new AnimatableDouble(0d, -100d, 100d);

	private AnimatableDouble easeLow = new AnimatableDouble(0d, -100d, 100d);

	private AnimatableBoolean randomizeOrder = new AnimatableBoolean(Boolean.FALSE);

	private AnimatableInteger randomSeed = new AnimatableInteger(0, 0, 10000);


	private final RenderContext context;

	@Inject
	public TARangeSelectorImpl(RenderContext context) {
		super("範囲セレクタ");
		this.context = context;
	}

	public AnimatableDouble getStart() {
		return start;
	}

	public void setStart(AnimatableDouble start) {
		start.copyConfigurationFrom(this.start);
		this.start = start;
	}

	public AnimatableDouble getEnd() {
		return end;
	}

	public void setEnd(AnimatableDouble end) {
		end.copyConfigurationFrom(this.end);
		this.end = end;
	}

	public AnimatableDouble getOffset() {
		return offset;
	}

	public void setOffset(AnimatableDouble offset) {
		offset.copyConfigurationFrom(this.offset);
		this.offset = offset;
	}

	public AnimatableMode getMode() {
		return mode;
	}

	public void setMode(AnimatableMode mode) {
		mode.copyConfigurationFrom(this.mode);
		this.mode = mode;
	}

	public AnimatableDouble getAmount() {
		return amount;
	}

	public void setAmount(AnimatableDouble amount) {
		amount.copyConfigurationFrom(this.amount);
		this.amount = amount;
	}

	public Shape getShape() {
		return shape;
	}

	public void setShape(Shape shape) {
		this.shape = shape;
	}

	public AnimatableDouble getSquareSmoothness() {
		return squareSmoothness;
	}

	public void setSquareSmoothness(AnimatableDouble squareSmoothness) {
		squareSmoothness.copyConfigurationFrom(this.squareSmoothness);
		this.squareSmoothness = squareSmoothness;
	}

	public AnimatableDouble getEaseHigh() {
		return easeHigh;
	}

	public void setEaseHigh(AnimatableDouble easeHigh) {
		easeHigh.copyConfigurationFrom(this.easeHigh);
		this.easeHigh = easeHigh;
	}

	public AnimatableDouble getEaseLow() {
		return easeLow;
	}

	public void setEaseLow(AnimatableDouble easeLow) {
		easeLow.copyConfigurationFrom(this.easeLow);
		this.easeLow = easeLow;
	}

	public AnimatableBoolean getRandomizeOrder() {
		return randomizeOrder;
	}

	public void setRandomizeOrder(AnimatableBoolean randomizeOrder) {
		randomizeOrder.copyConfigurationFrom(this.randomizeOrder);
		this.randomizeOrder = randomizeOrder;
	}

	public AnimatableInteger getRandomSeed() {
		return randomSeed;
	}

	public void setRandomSeed(AnimatableInteger randomSeed) {
		randomSeed.copyConfigurationFrom(this.randomSeed);
		this.randomSeed = randomSeed;
	}

	public void prepareExpression(ExpressionScope scope) {
		scope.assignTo(start);
		scope.assignTo(end);
		scope.assignTo(offset);
		scope.assignTo(mode);
		scope.assignTo(amount);
		scope.assignTo(squareSmoothness);
		scope.assignTo(easeHigh);
		scope.assignTo(easeLow);
		scope.assignTo(randomizeOrder);
		scope.assignTo(randomSeed);
	}

	public Object createExpressionElement(CoreContext context) {
		return new TARangeSelectorExpressionElement(context);
	}

	public class TARangeSelectorExpressionElement extends TASelectorExpressionElement {

		public TARangeSelectorExpressionElement(CoreContext context) {
			super(context);
		}

		public Object getStart()			{ return elem(start); }
		public Object getEnd()				{ return elem(end); }
		public Object getOffset()			{ return elem(offset); }
		public Object getMode()				{ return elem(mode); }
		public Object getAmount()			{ return elem(amount); }
		public String getShape()			{ return shape.name(); }
		public Object getSquareSmoothness()	{ return elem(squareSmoothness); }
		public Object getEaseHigh()			{ return elem(easeHigh); }
		public Object getEaseLow()			{ return elem(easeLow); }
		public Object getRandomizeOrder()	{ return elem(randomizeOrder); }
		public Object getRandomSeed()		{ return elem(randomSeed); }
	}

	public Evaluator createEvaluator(int[] totals, Time mediaTime) {
		double start0 = this.start.value(context) * 0.01;
		double end0 = this.end.value(context) * 0.01;
		double offset = this.offset.value(context) * 0.01;
		final double start, end;
		if (start0 < end0) {
			start = start0 + offset;
			end = end0 + offset;
		} else {
			start = end0 + offset;
			end = start0 + offset;
		}

		final Mode mode = this.mode.value(context);
		final double amount = this.amount.value(context) * 0.01;
		final double squareSmoothness = (shape == Shape.SQUARE)
				? this.squareSmoothness.value(context) * 0.01 : 0;
		final double easeHigh = this.easeHigh.value(context) * 0.01;
		final double easeLow = this.easeLow.value(context) * 0.01;

		Random rndOrder = randomizeOrder.value(context)
				? new Random(randomSeed.value(context)) : null;

		return new AbstractEvaluator(totals, rndOrder) {
			@Override
			protected double[] evaluate(double[] combine, int index0, int index1, int total) {
				if (index0 >= index1 || index0 < 0 || index1 > total) {
					throw new IllegalArgumentException();
				}

				double t = 0;
				if (shape == Shape.SQUARE && squareSmoothness > 0) {
					double p0 = (double)index0 / total;
					double p1 = (double)index1 / total;
					double p1_or_end = Math.min(p1, end);
					t = (p1 <= start || p0 >= end) ? 0
					  : (p0 <= start && p1 > start) ? (p1_or_end-start) / (p1-p0)
					  : (p1 >= end && p0 < end) ? (end-p0) / (p1-p0)
					  : 1;
					t = Math.min(Math.max((t-0.5)/squareSmoothness+0.5, 0), 1);

				} else {
					double p = (index0 + index1) * 0.5 / total;
					t = (p - start) / (end - start);
					switch (shape) {
						case SQUARE:
							// AEは0,1ともに等号が入っている動作をする
							t = (t <= 0 || t >= 1) ? 0 : 1;
							break;
						case RAMP_UP:
							t = Math.min(Math.max(t, 0), 1);
							break;
						case RAMP_DOWN:
							t = 1 - Math.min(Math.max(t, 0), 1);
							break;
						case TRIANGLE:
							t = Math.max(1-Math.abs(t*2-1), 0);
							break;
						case ROUND:
							t = Math.min(Math.max(t, 0), 1);
							t = Math.sqrt(4*t*(1-t));
							break;
						case SMOOTH:
							t = Math.max(1-Math.abs(t*2-1), 0);
							t = 0.5*(t*t*(3-2*t)+t);  // ←これを入れるとAEと似た感じになる。
							t = t*t*(3-2*t);
							break;
						default:
							throw new RuntimeException("unknown Shape: " + shape);
					}
				}

				t = ease(t) * amount;

				return applyMode(mode, combine, new double[] { t, t, t });
			}

			private double ease(double t) {
				if (t == 0 || t == 1) {
					return t;

				} else if (easeHigh > 0 && easeLow > 0) {
					double p;
					if (easeHigh < easeLow) {
						p = 0.5*(1+(easeLow-easeHigh)/easeLow);
					} else {
						p = 0.5*(1-(easeHigh-easeLow)/easeHigh);
					}

					double mix = Math.max(easeHigh, easeLow);

					if (t < p) {
						double u = t/p;
						return (p*u*u*u)*mix + t*(1-mix);
					} else {
						double q = 1/(1-p);
						double u = q*(t-1);
						return (u*u*u/q+1)*mix + t*(1-mix);
					}

				} else if (easeHigh < 0 && easeLow < 0) {
					double p;
					if (easeHigh > easeLow) {
						p = 0.5*(1+(easeLow-easeHigh)/easeLow);
					} else {
						p = 0.5*(1-(easeHigh-easeLow)/easeHigh);
					}

					double mix = Math.abs(Math.min(easeHigh, easeLow));

					if (t < p) {
						double u = t/p;
						return (p*Math.cbrt(u))*mix + t*(1-mix);
					} else {
						double q = 1/(1-p);
						double u = q*(t-1);
						return (Math.cbrt(u)/q+1)*mix + t*(1-mix);
					}

				} else if (easeHigh <= 0 && easeLow >= 0) {
					double eh = -easeHigh;
					double u = t-1;
					double t2 = (Math.cbrt(u)+1)*eh + t*(1-eh);
					double t3 = t*t*t*easeLow + t*(1-easeLow);
					return t2*t3/t;

				} else if (easeHigh >= 0 && easeLow <= 0) {
					double el = -easeLow;
					double u = t-1;
					double t2 = (u*u*u+1)*easeHigh + t*(1-easeHigh);
					double t3 = Math.cbrt(t)*el + t*(1-el);
					return 1-(1-t2)*(1-t3)/(1-t);

				} else {
					return t;
				}
			}
		};
	}

}
