/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.lang.ref.WeakReference;
import java.util.Map;
import java.util.Set;
import java.util.WeakHashMap;

import org.mozilla.javascript.Context;
import org.mozilla.javascript.ContextAction;
import org.mozilla.javascript.ContextFactory;
import org.mozilla.javascript.NativeJavaObject;
import org.mozilla.javascript.Script;
import org.mozilla.javascript.Scriptable;
import org.mozilla.javascript.ScriptableObject;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CoreContext;
import ch.kuramo.javie.core.ExpressionScope;
import ch.kuramo.javie.core.Expressionee;
import ch.kuramo.javie.core.Expressioner;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.expression.ExpressionUtils;
import ch.kuramo.javie.core.expression.Interpolation;
import ch.kuramo.javie.core.expression.RandomNumbers;

import com.google.inject.Inject;

public abstract class AbstractCoreContext implements CoreContext {

	@Inject
	private ContextFactory _jsContextFactory;


	private Time _time;

	private Composition _comp;

	private Scriptable _initialExprScope;

	private final Set<Composition> _scopeCreatedComps = Util.newSet();

	private final Map<Expressioner<?>, ExpressionScopeImpl> _scopeMap = Util.newMap();

	private final WeakHashMap<Expressionee, WeakReference<?>> _exprElemPool = Util.newWeakHashMap();

	private final Set<Expressioner<?>> _cyclicCheck = Util.newSet();

	private final WeakHashMap<String, Script> _exprScriptCache = Util.newWeakHashMap();


	public void activate() {
		if (_initialExprScope != null) {
			throw new IllegalStateException("already activated");
		}

		Context jsContext = _jsContextFactory.enterContext();
		ScriptableObject scope = jsContext.initStandardObjects(null, true);
		scope.defineFunctionProperties(Interpolation.METHOD_NAMES, Interpolation.class, ScriptableObject.DONTENUM);
		scope.defineFunctionProperties(RandomNumbers.METHOD_NAMES, RandomNumbers.class, ScriptableObject.DONTENUM);
		scope.sealObject();
		_initialExprScope = scope;
	}

	public void deactivate() {
		_time = null;
		_comp = null;

		if (_initialExprScope != null) {
			_initialExprScope = null;
			Context.exit();
		}
	}

	public boolean isActive() {
		return (_initialExprScope != null);
	}

	private Scriptable createNewExprScope(final Scriptable sharedScope) {
		return (Scriptable) _jsContextFactory.call(new ContextAction() {
			public Object run(Context context) {
				Scriptable newScope = context.newObject(sharedScope);
				newScope.setPrototype(sharedScope);
				newScope.setParentScope(null);
				return newScope;
			}
		});
	}

	public void reset() {
		_time = null;
		_comp = null;
	}

	public <T> T saveAndExecute(WrappedOperation<T> operation) {
		Time time = _time;
		Composition comp = _comp;
		try {
			return operation.execute();
		} finally {
			_time = time;
			_comp = comp;
		}
	}

	public Time getTime() {
		return _time;
	}

	public void setTime(Time time) {
		_time = time;
	}

	public Composition getComposition() {
		return _comp;
	}

	public void setComposition(Composition comp) {
		_comp = comp;
	}

	public ExpressionScope createInitialExpressionScope(Composition comp) {
		_scopeCreatedComps.clear();
		_scopeMap.clear();

		_scopeCreatedComps.add(comp);
		return new ExpressionScopeImpl(createNewExprScope(_initialExprScope));
	}

	public <T> T getExpressionElement(Expressionee exprnee) {
		@SuppressWarnings("unchecked")
		WeakReference<T> ref = (WeakReference<T>) _exprElemPool.get(exprnee);
		if (ref != null) {
			T elem = ref.get();
			if (elem != null) {
				return elem;
			}
		}

		@SuppressWarnings("unchecked")
		T elem = (T) exprnee.createExpressionElement(this);
		_exprElemPool.put(exprnee, new WeakReference<Object>(elem));
		return elem;
	}

	public boolean checkCyclicEvaluation(Expressioner<?> expr) {
		return _cyclicCheck.contains(expr);
	}

	public <T> T evaluateExpression(final Expressioner<T> expr) {
		final ExpressionScopeImpl scope = _scopeMap.get(expr);
		if (scope == null) {
			throw new IllegalArgumentException("no expression scope is assigned.");
		}

		if (checkCyclicEvaluation(expr)) {
			throw new IllegalArgumentException("cyclic evaluation: " + expr.getExpression());
		}

		_cyclicCheck.add(expr);
		try {
			return expr.jsToJava(_jsContextFactory.call(new ContextAction() {
				public Object run(Context context) {
					ExpressionUtils.pushThreadLocals(context, /*Time.class,*/ RandomNumbers.class);
					try {
						//context.putThreadLocal(Time.class, _time);
						context.putThreadLocal(RandomNumbers.class,
								new RandomNumbers(31 * scope.seed + expr.randomSeed(), _time));

						Scriptable localScope = createNewExprScope(scope.scope);
						ScriptableObject.putConstProperty(localScope, "time", Context.javaToJS(_time.toSecond(), localScope));
						return getScript(context, expr).exec(context, localScope);

					} finally {
						ExpressionUtils.popThreadLocals(context);
					}
				}
			}));
		} finally {
			_cyclicCheck.remove(expr);
		}
	}

	public Scriptable toNativeJavaObject(Object javaObject, final Class<?> defaultHint) {
		@SuppressWarnings("serial")
		NativeJavaObject njo = new NativeJavaObject(createNewExprScope(_initialExprScope), javaObject, null) {
			@Override
			public String getClassName() {
				return "JavieJavaObject";
			}

			@Override
			public Object getDefaultValue(Class<?> hint) {
				return super.getDefaultValue((hint != null) ? hint : defaultHint);
			}
		};
		return njo;
	}

	private Script getScript(Context context, Expressioner<?> expr) {
		String exprStr = expr.getExpression();

		Script script = _exprScriptCache.get(exprStr);
		if (script != null) {
			return script;
		}

		String sourceName = "";	// TODO Expressionerにソースネームを返すメソッドを追加して、そこから取得するようにする。
		script = context.compileString(exprStr, sourceName, 1, null);

		_exprScriptCache.put(exprStr, script);

		return script;
	}

	protected Set<Expressioner<?>> getExpressioners() {
		return _scopeMap.keySet();
	}

	private class ExpressionScopeImpl implements ExpressionScope {

		private Scriptable scope;

		private int seed = 1;

		private ExpressionScopeImpl(Scriptable scope) {
			this.scope = scope;
		}

		public ExpressionScopeImpl clone() {
			try {
				ExpressionScopeImpl clone = (ExpressionScopeImpl) super.clone();
				clone.scope = createNewExprScope(scope);
				return clone;
			} catch (CloneNotSupportedException e) {
				// never reach here
				throw new Error(e);
			}
		}

		public ExpressionScope createPrecompositionScope(Composition preComp) {
			if (_scopeCreatedComps.contains(preComp)) return null;

			_scopeCreatedComps.add(preComp);
			return new ExpressionScopeImpl(createNewExprScope(_initialExprScope));
		}

		public void setThis(Expressionee exprnee) {
			Scriptable newScope = (Scriptable) Context.javaToJS(getExpressionElement(exprnee), scope);
			newScope.setPrototype(scope);
			newScope.setParentScope(null);
			scope = newScope;
		}

		public void setRandomSeed(int seed) {
			this.seed = seed;
		}

		public void putExpressionElement(String name, Expressionee exprnee) {
			ScriptableObject.putConstProperty(scope, name, Context.javaToJS(getExpressionElement(exprnee), scope));
		}

		public void assignTo(Expressioner<?> expr) {
			if (_scopeMap.containsKey(expr)) {
				throw new IllegalArgumentException("already assigned");
			}
			_scopeMap.put(expr, this);
		}

	}

}
