/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.io.File;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.core.MediaSource;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.internal.JavaImageSource;
import ch.kuramo.javie.core.internal.JavaSoundSource;
import ch.kuramo.javie.core.services.MediaSourceFactory;

import com.google.inject.Inject;
import com.google.inject.Injector;

public class MediaSourceFactoryImpl implements MediaSourceFactory {

	private static final Logger _logger = LoggerFactory.getLogger(MediaSourceFactoryImpl.class);


	private static class Entry {
		int counter = 1;
		final MediaSource source;
		Entry(MediaSource source) { this.source = source; }
	}


	private final List<Class<? extends MediaSource>> _sourceClasses = Util.newList();

	private final Map<String, Entry> _entries = Util.newMap();

	@Inject
	private Injector _injector;


	public MediaSourceFactoryImpl() {
		super();

		Class<MediaSource> clazz;

		if ((clazz = classForName("ch.kuramo.javie.core.internal.MacOSXQTCoreVideoSource")) != null) {
			_sourceClasses.add(JavaImageSource.class);
			_sourceClasses.add(JavaSoundSource.class);	// サンプルレートの変換が行われるときはQTの方だとノイズが乗るので、
														// JavaSoundSourceをMacOSXQTCoreVideoSourceより前に入れておく。
														// (QT Playerでサンプルレートを変換して書き出しても問題ないようなので、Javieの実装が何かまずいと思われる)

			// Quartz Composer のアプリはデベロッパーツールを入れる必要があるが
			// QuartzComposer.framework はデベロッパーツール無しでもインストールされているのだろうか？
			// インストールされていない場合は QCCompositionSource がロード出来ないので、チェックする必要がある。
			Class<MediaSource> qccsClass = classForName("ch.kuramo.javie.core.internal.QCCompositionSource");
			if (qccsClass != null) {
				_sourceClasses.add(qccsClass);
			}

			_sourceClasses.add(clazz);

		} else if ((clazz = classForName("ch.kuramo.javie.core.internal.WindowsDirectShowSource")) != null) {
			_sourceClasses.add(JavaImageSource.class);
			_sourceClasses.add(JavaSoundSource.class);	// WindowsDirectShowSourceでwavファイルを読み込むと
			_sourceClasses.add(clazz);					// パフォーマンスに問題があるようなので、JavaSoundSourceを前に入れておく。

		} else {
			_logger.warn("only JavaImageSource and JavaSoundSource are available");
			_sourceClasses.add(JavaImageSource.class);
			_sourceClasses.add(JavaSoundSource.class);
		}
	}

	private Class<MediaSource> classForName(String className) {
		try {
			@SuppressWarnings("unchecked")
			Class<MediaSource> clazz = (Class<MediaSource>) Class.forName(className);
			return clazz;
		} catch (Throwable t) {
			return null;
		}
	}

	public synchronized MediaSource get(String key, File file) {
		Entry entry = _entries.get(key);
		if (entry != null) {
			++entry.counter;
			return entry.source;
		}

		// TODO 複数の MediaSource 実装の優先順位をユーザーが決められるようにして、
		//		その順番で試行するようにする。

		Map<Class<? extends MediaSource>, Throwable> throwables = Util.newLinkedHashMap();

		for (Class<? extends MediaSource> clazz : _sourceClasses) {
			MediaSource source;
			try {
				source = _injector.getInstance(clazz);
				if (!source.initialize(file)) {
					source.dispose();
					continue;
				}
			} catch (Throwable t) {
				throwables.put(clazz, t);
				continue;
			}

			_entries.put(key, new Entry(source));
			return source;
		}

		for (Map.Entry<Class<? extends MediaSource>, Throwable> e : throwables.entrySet()) {
			_logger.info(String.format("failed to create MediaSource (%s): %s: %s",
					e.getKey().getName(), file.getAbsolutePath(), e.getValue().toString()));
		}

		return null;
	}

	public synchronized <T extends MediaSource> T get(Class<T> sourceClass, String key, File file) {
		Entry entry = _entries.get(key);
		if (entry != null) {
			++entry.counter;
			return sourceClass.cast(entry.source);
		}

		try {
			T source = _injector.getInstance(sourceClass);
			if (source.initialize(file)) {
				_entries.put(key, new Entry(source));
				return source;
			} else {
				source.dispose();
			}
		} catch (Throwable t) {
			_logger.info(String.format("failed to create MediaSource (%s): %s: %s",
					sourceClass.getName(), file.getAbsolutePath(), t.toString()));
		}

		return null;
	}

	public synchronized void release(String key) {
		Entry entry = _entries.get(key);
		if (--entry.counter == 0) {
			_entries.remove(key);
			entry.source.dispose();
		}
	}

}
