/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "stdafx.h"
#include "DirectShowOutput.h"
#include <MMReg.h>


#define RELEASE(p)						do { if (p) { p->Release(); p = NULL; } } while (false)
#define RETURN_HRESULT_IF_FAILED(hr)	do { HRESULT _hr = (hr); if (FAILED(_hr)) return _hr; } while (false)
#define BAIL_IF_FAILED(label, hr)		do { if (FAILED((hr))) goto label; } while (false)


// {0C324FBB-6C2D-4c5c-BE29-0CCAEA40F4E0}
DEFINE_GUID(CLSID_VideoSource, 
0xc324fbb, 0x6c2d, 0x4c5c, 0xbe, 0x29, 0xc, 0xca, 0xea, 0x40, 0xf4, 0xe0);

// {652597D7-74BA-4caa-904A-16A80F9ABE1C}
DEFINE_GUID(CLSID_AudioSource, 
0x652597d7, 0x74ba, 0x4caa, 0x90, 0x4a, 0x16, 0xa8, 0xf, 0x9a, 0xbe, 0x1c);


/**********************************************
 *
 *  CVideoPushPin
 *
 **********************************************/

CVideoPushPin::CVideoPushPin(HRESULT* hr, CSource* filter,
							 long width, long height, LONGLONG frameDuration, BOOL hasAlpha)
	: CSourceStream(NAME("CVideoPushPin"), hr, filter, L"Out"),
	  mWidth(width), mHeight(height), mFrameDuration(frameDuration), mHasAlpha(hasAlpha),
	  mTimeStart(0), mTimeEnd(0), mBuffer(NULL),
	  mWaitEvent1(CreateEvent(NULL, FALSE, FALSE, NULL)),
	  mWaitEvent2(CreateEvent(NULL, FALSE, FALSE, NULL))
{
}

CVideoPushPin::~CVideoPushPin()
{
	CloseHandle(mWaitEvent1);
	CloseHandle(mWaitEvent2);
}

HRESULT CVideoPushPin::GetMediaType(CMediaType* mediaType)
{
	CheckPointer(mediaType, E_POINTER);
	CAutoLock autoLock(m_pFilter->pStateLock());

	VIDEOINFO* vi = (VIDEOINFO*) mediaType->AllocFormatBuffer(sizeof(VIDEOINFO));
	if (NULL == vi) return E_OUTOFMEMORY;

	ZeroMemory(vi, sizeof(VIDEOINFO));

	vi->AvgTimePerFrame = mFrameDuration;

	BITMAPINFOHEADER* bmi = &(vi->bmiHeader);
	bmi->biSize = sizeof(BITMAPINFOHEADER);
	bmi->biWidth = mWidth;
	bmi->biHeight = -mHeight;
	bmi->biPlanes = 1;
	bmi->biBitCount = 32;
	bmi->biCompression = BI_RGB;
	bmi->biSizeImage = DIBSIZE(vi->bmiHeader);

	mediaType->SetType(&MEDIATYPE_Video);
	mediaType->SetFormatType(&FORMAT_VideoInfo);
	mediaType->SetTemporalCompression(FALSE);
	mediaType->SetSampleSize(bmi->biSizeImage);
	mediaType->SetSubtype(mHasAlpha ? &MEDIASUBTYPE_ARGB32 : &MEDIASUBTYPE_RGB32);

	return S_OK;
}

HRESULT CVideoPushPin::DecideBufferSize(IMemAllocator* alloc, ALLOCATOR_PROPERTIES* request)
{
	CheckPointer(alloc, E_POINTER);
	CheckPointer(request, E_POINTER);
	CAutoLock autoLock(m_pFilter->pStateLock());

	VIDEOINFO* vi = (VIDEOINFO*) m_mt.Format();

	request->cBuffers = 1;
	request->cbBuffer = vi->bmiHeader.biSizeImage;

	ALLOCATOR_PROPERTIES actual;
	HRESULT hr = alloc->SetProperties(request, &actual);
	if (FAILED(hr)) return hr;
	if (actual.cbBuffer < request->cbBuffer) return E_FAIL;

	return S_OK;
}

HRESULT CVideoPushPin::FillBuffer(IMediaSample* sample)
{
	CheckPointer(sample, E_POINTER);

	WaitForSingleObject(mWaitEvent1, INFINITE);
	if (mBuffer == NULL) return S_FALSE;

	ASSERT(m_mt.formattype == FORMAT_VideoInfo);

	VIDEOINFO* vi = (VIDEOINFO*) m_mt.Format();
	long size = min(vi->bmiHeader.biSizeImage, (DWORD)sample->GetSize());
	LPBYTE data = NULL;
	sample->GetPointer(&data);

	memcpy(data, mBuffer, size);

	sample->SetTime(&mTimeStart, &mTimeEnd);
	sample->SetActualDataLength(size);
	sample->SetSyncPoint(TRUE);

	SetEvent(mWaitEvent2);

	return S_OK;
}

void CVideoPushPin::WriteBuffer(LONGLONG timeStart, LONGLONG timeEnd, void* buffer)
{
	mTimeStart = timeStart;
	mTimeEnd = timeEnd;
	mBuffer = buffer;

	if (mBuffer != NULL) {
		SignalObjectAndWait(mWaitEvent1, mWaitEvent2, INFINITE, FALSE);

	} else {
		SetEvent(mWaitEvent1);

		Stop();
		Exit();
		Close();
	}
}

/**********************************************
 *
 *  CVideoSource
 *
 **********************************************/

CVideoSource::CVideoSource(LPUNKNOWN unk, HRESULT* hr,
						   long width, long height, LONGLONG frameDuration, BOOL hasAlpha)
	: CSource(NAME("VideoSource"), unk, CLSID_VideoSource),
	  mPushPin(NULL)
{
	mPushPin = new CVideoPushPin(hr, this, width, height, frameDuration, hasAlpha);
	if (mPushPin == NULL) {
		*hr = E_OUTOFMEMORY;
	}
}

CVideoSource::~CVideoSource()
{
}


/**********************************************
 *
 *  CAudioPushPin
 *
 **********************************************/

CAudioPushPin::CAudioPushPin(HRESULT* hr, CSource* filter,
							 long channels, long sampleRate, long sampleSize, BOOL float_)
	: CSourceStream(NAME("CAudioPushPin"), hr, filter, L"Out"),
	  mChannels(channels), mSampleRate(sampleRate), mSampleSize(sampleSize), mFloat(float_),
	  mTime(0), mAudioFrameCount(0), mBuffer(NULL),
	  mWaitEvent1(CreateEvent(NULL, FALSE, FALSE, NULL)),
	  mWaitEvent2(CreateEvent(NULL, FALSE, FALSE, NULL))
{
}

CAudioPushPin::~CAudioPushPin()
{
	CloseHandle(mWaitEvent1);
	CloseHandle(mWaitEvent2);
}

HRESULT CAudioPushPin::GetMediaType(CMediaType* mediaType)
{
	CheckPointer(mediaType, E_POINTER);
	CAutoLock autoLock(m_pFilter->pStateLock());

	WAVEFORMATEX* wfx = (WAVEFORMATEX*) mediaType->AllocFormatBuffer(sizeof(WAVEFORMATEX));
	if (NULL == wfx) return E_OUTOFMEMORY;

	ZeroMemory(wfx, sizeof(WAVEFORMATEX));

	wfx->cbSize = 0;
	wfx->nAvgBytesPerSec = mChannels * mSampleSize * mSampleRate;
	wfx->nBlockAlign = (WORD) (mChannels * mSampleSize);
	wfx->nChannels = (WORD) mChannels;
	wfx->nSamplesPerSec = mSampleRate;
	wfx->wBitsPerSample = (WORD) (mSampleSize * 8);
	wfx->wFormatTag = mFloat ? WAVE_FORMAT_IEEE_FLOAT : WAVE_FORMAT_PCM;

	mediaType->SetType(&MEDIATYPE_Audio);
	mediaType->SetFormatType(&FORMAT_WaveFormatEx);
	mediaType->SetTemporalCompression(FALSE);
	mediaType->SetSampleSize(wfx->nBlockAlign);
	mediaType->SetSubtype(mFloat ? &MEDIASUBTYPE_IEEE_FLOAT : &MEDIASUBTYPE_PCM);

	return S_OK;
}

HRESULT CAudioPushPin::DecideBufferSize(IMemAllocator* alloc, ALLOCATOR_PROPERTIES* request)
{
	CheckPointer(alloc, E_POINTER);
	CheckPointer(request, E_POINTER);
	CAutoLock autoLock(m_pFilter->pStateLock());

	WAVEFORMATEX* wfx = (WAVEFORMATEX*) m_mt.Format();

	request->cBuffers = 1;
	request->cbBuffer = wfx->nAvgBytesPerSec;	// Pb̃obt@mۂ

	ALLOCATOR_PROPERTIES actual;
	HRESULT hr = alloc->SetProperties(request, &actual);
	if (FAILED(hr)) return hr;
	if (actual.cbBuffer < request->cbBuffer) return E_FAIL;

	return S_OK;
}

HRESULT CAudioPushPin::FillBuffer(IMediaSample* sample)
{
	CheckPointer(sample, E_POINTER);

	WaitForSingleObject(mWaitEvent1, INFINITE);
	if (mBuffer == NULL) return S_FALSE;

	ASSERT(m_mt.formattype == FORMAT_WaveFormatEx);


	WAVEFORMATEX* wfx = (WAVEFORMATEX*) m_mt.Format();
	long size = min(wfx->nBlockAlign * mAudioFrameCount, sample->GetSize());
	LPBYTE data = NULL;
	sample->GetPointer(&data);

	memcpy(data, mBuffer, size);

	LONGLONG timeEnd = mTime + (LONGLONG)10000000 * mAudioFrameCount / wfx->nSamplesPerSec;
	sample->SetTime(&mTime, &timeEnd);
	sample->SetActualDataLength(size);
	sample->SetSyncPoint(TRUE);

	SetEvent(mWaitEvent2);

	return S_OK;
}

void CAudioPushPin::WriteBuffer(LONGLONG time, long audioFrameCount, void* buffer)
{
	mTime = time;
	mAudioFrameCount = audioFrameCount;
	mBuffer = buffer;

	if (mBuffer != NULL) {
		SignalObjectAndWait(mWaitEvent1, mWaitEvent2, INFINITE, FALSE);

	} else {
		SetEvent(mWaitEvent1);

		Stop();
		Exit();
		Close();
	}
}


/**********************************************
 *
 *  CAudioSource
 *
 **********************************************/

CAudioSource::CAudioSource(LPUNKNOWN unk, HRESULT* hr,
						   long channels, long sampleRate, long sampleSize, BOOL float_)
	: CSource(NAME("AudioSource"), unk, CLSID_AudioSource),
	  mPushPin(NULL)
{
	mPushPin = new CAudioPushPin(hr, this, channels, sampleRate, sampleSize, float_);
	if (mPushPin == NULL) {
		*hr = E_OUTOFMEMORY;
	}
}

CAudioSource::~CAudioSource()
{
}


/**********************************************
 *
 *  DirectShowOutput
 *
 **********************************************/

DirectShowOutput::DirectShowOutput()
	: mCapBuilder2(NULL), mAVIMux(NULL),
	  mGraphBuilder(NULL), mMediaControl(NULL),
	  mVideoSource(NULL), mAudioSource(NULL),
	  mRotRegister(0)
{
}

DirectShowOutput::~DirectShowOutput()
{
	RemoveFromRot();

	if (mAudioSource != NULL) {
		mAudioSource->GetPushPin()->WriteBuffer(0, 0, NULL);
	}

	if (mVideoSource != NULL) {
		mVideoSource->GetPushPin()->WriteBuffer(0, 0, NULL);
	}

	if (mMediaControl != NULL) {
		mMediaControl->Stop();
	}

	RELEASE(mAudioSource);
	RELEASE(mVideoSource);
	RELEASE(mMediaControl);
	RELEASE(mGraphBuilder);
	RELEASE(mAVIMux);
	RELEASE(mCapBuilder2);
}

HRESULT DirectShowOutput::Initialize(LPCWSTR file)
{
	HRESULT hr;

	BAIL_IF_FAILED(bail,
		hr = CoCreateInstance(CLSID_CaptureGraphBuilder2,
			NULL, CLSCTX_INPROC_SERVER, IID_ICaptureGraphBuilder2, (LPVOID*)&mCapBuilder2));

	BAIL_IF_FAILED(bail,
		hr = mCapBuilder2->SetOutputFileName(&MEDIASUBTYPE_Avi, file, &mAVIMux, NULL));

	BAIL_IF_FAILED(bail,
		hr = mCapBuilder2->GetFiltergraph(&mGraphBuilder));

	AddToRot();

	hr = S_OK;

bail:
	return hr;
}

HRESULT DirectShowOutput::AddVideoSource(
	long width, long height, LONGLONG frameDuration, BOOL hasAlpha,
	LPCWSTR vcName, double vcQuality, long vcKeyFrameRate, long vcPFramesPerKey,
	LONGLONG vcBitRate, LONGLONG vcWindowSize, void* vcConfig, size_t vcConfigSize)
{
	HRESULT hr;
	IBindCtx* bindCtx = NULL;
	IMoniker* moniker = NULL;
	IBaseFilter* vcompFilter = NULL;
	IPin* outPin = NULL;
	IAMVideoCompression* vcomp = NULL;
	IAMVfwCompressDialogs* dialogs = NULL;

	mVideoSource = new CVideoSource(NULL, &hr, width, height, frameDuration, hasAlpha);
	if (mVideoSource == NULL) {
		hr = E_OUTOFMEMORY;
	} else {
		mVideoSource->AddRef();
	}
	BAIL_IF_FAILED(bail, hr);

	BAIL_IF_FAILED(bail,
		hr = mGraphBuilder->AddFilter(mVideoSource, L"Vidoe Source"));

	if (vcName != NULL) {
		BAIL_IF_FAILED(bail,
			hr = CreateBindCtx(0, &bindCtx));

		ULONG eaten;
		BAIL_IF_FAILED(bail,
			hr = MkParseDisplayName(bindCtx, vcName, &eaten, &moniker));

		BAIL_IF_FAILED(bail,
			hr = moniker->BindToObject(bindCtx, NULL, IID_IBaseFilter, (LPVOID*)&vcompFilter));

		BAIL_IF_FAILED(bail,
			hr = mGraphBuilder->AddFilter(vcompFilter, L"Video Compressor"));
	}

	BAIL_IF_FAILED(bail,
		hr = mCapBuilder2->RenderStream(NULL, NULL, (IBaseFilter*)mVideoSource, vcompFilter, mAVIMux));

	if (vcompFilter != NULL) {
		BAIL_IF_FAILED(bail,
			hr = VideoCompressorUtil::GetPin(vcompFilter, PINDIR_OUTPUT, &outPin));

		hr = outPin->QueryInterface(IID_IAMVideoCompression, (LPVOID*)&vcomp);
		if (hr == S_OK) {
			int versSize, descSize;
			long m, n, capabilities;
			double d;
			hr = vcomp->GetInfo(NULL, &versSize, NULL, &descSize, &m, &n, &d, &capabilities);
			if (hr == S_OK) {
				if ((capabilities & 0x01) != 0) BAIL_IF_FAILED(bail, hr = vcomp->put_Quality(vcQuality));
				if ((capabilities & 0x04) != 0) BAIL_IF_FAILED(bail, hr = vcomp->put_KeyFrameRate(vcKeyFrameRate));
				if ((capabilities & 0x08) != 0) BAIL_IF_FAILED(bail, hr = vcomp->put_PFramesPerKeyFrame(vcPFramesPerKey));
				if ((capabilities & 0x10) != 0) BAIL_IF_FAILED(bail, hr = vcomp->put_WindowSize(vcWindowSize));
				if ((capabilities & 0x02) != 0) BAIL_IF_FAILED(bail, hr = VideoCompressorUtil::SetBitRate(outPin, vcBitRate));
			}
		}

		if (vcConfig != NULL) {
			BAIL_IF_FAILED(bail,
				hr = vcompFilter->QueryInterface(IID_IAMVfwCompressDialogs, (LPVOID*)&dialogs));

			// MSDÑhLgɂƁu߂ĺAehCo̎ɉĈقȂBv
			// ƂƂȂ̂Ŗ߂l̃`FbN͂ȂB
			/*hr =*/ dialogs->SendDriverMessage(ICM_SETSTATE, (long)vcConfig, vcConfigSize);
		}
	}

	hr = S_OK;

bail:
	RELEASE(dialogs);
	RELEASE(vcomp);
	RELEASE(outPin);
	RELEASE(vcompFilter);
	RELEASE(moniker);
	RELEASE(bindCtx);
	return hr;
}

HRESULT DirectShowOutput::AddAudioSource(long channels, long sampleRate, long sampleSize, BOOL float_)
{
	HRESULT hr;

	mAudioSource = new CAudioSource(NULL, &hr, channels, sampleRate, sampleSize, float_);
	if (mAudioSource == NULL) {
		hr = E_OUTOFMEMORY;
	} else {
		mAudioSource->AddRef();
	}
	BAIL_IF_FAILED(bail, hr);

	BAIL_IF_FAILED(bail,
		hr = mGraphBuilder->AddFilter(mAudioSource, L"Audio Source"));

	BAIL_IF_FAILED(bail,
		hr = mCapBuilder2->RenderStream(NULL, NULL, (IBaseFilter*)mAudioSource, NULL, mAVIMux));

	hr = S_OK;

bail:
	return hr;
}

HRESULT DirectShowOutput::GetAudioStreamIndex(int* index)
{
	HRESULT hr;
	CMediaType audioType(&MEDIATYPE_Audio);
	IEnumPins* enumPins = NULL;
	IPin* pin = NULL;

	BAIL_IF_FAILED(bail,
		hr = mAVIMux->EnumPins(&enumPins));

	for (int i = 0; (hr = enumPins->Next(1, &pin, NULL)) == S_OK; ) {
		PIN_DIRECTION pd;

		BAIL_IF_FAILED(bail,
			hr = pin->QueryDirection(&pd));

		if (pd == PINDIR_INPUT) {
			CMediaType mediaType;

			BAIL_IF_FAILED(bail,
				hr = pin->ConnectionMediaType(&mediaType));

			if (mediaType.MatchesPartial(&audioType)) {
				*index = i;
				hr = S_OK;
				goto bail;
			}

			++i;
		}

		RELEASE(pin);
	}

bail:
	RELEASE(pin);
	RELEASE(enumPins);
	return hr;
}

HRESULT DirectShowOutput::SetMasterStreamAndInterleaving(REFERENCE_TIME interleave)
{
	if (mVideoSource == NULL || mAudioSource == NULL) return S_FALSE;

	HRESULT hr;
	int audioStreamIndex;
	IConfigAviMux* configAviMux = NULL;
	IConfigInterleaving* configIntlv = NULL;

	BAIL_IF_FAILED(bail,
		hr = GetAudioStreamIndex(&audioStreamIndex));

	BAIL_IF_FAILED(bail,
		hr = mAVIMux->QueryInterface(IID_IConfigAviMux, (LPVOID*)&configAviMux));

	BAIL_IF_FAILED(bail,
		hr = configAviMux->SetMasterStream(audioStreamIndex));

	BAIL_IF_FAILED(bail,
		hr = mAVIMux->QueryInterface(IID_IConfigInterleaving, (LPVOID*)&configIntlv));

	BAIL_IF_FAILED(bail,
		hr = configIntlv->put_Mode(INTERLEAVE_FULL));

	REFERENCE_TIME preroll = 0;
	BAIL_IF_FAILED(bail,
		hr = configIntlv->put_Interleaving(&interleave, &preroll));

	hr = S_OK;

bail:
	RELEASE(configIntlv);
	RELEASE(configAviMux);
	return hr;
}

HRESULT DirectShowOutput::RunGraph()
{
	HRESULT hr;

	BAIL_IF_FAILED(bail,
		hr = mGraphBuilder->QueryInterface(IID_IMediaControl, (LPVOID*)&mMediaControl));

	BAIL_IF_FAILED(bail,
		hr = mMediaControl->Run());

	if (hr == S_FALSE) {
		// TODO  State_Running ɂȂ̂܂ő҂Kv邩H
	}

	hr = S_OK;

bail:
	return hr;
}

void DirectShowOutput::AddToRot()
{
	IRunningObjectTable* rot;
	if (FAILED(GetRunningObjectTable(0, &rot))) {
		return;
	}

	WCHAR name[256];
	wsprintfW(name, L"FilterGraph %08x pid %08x", (DWORD_PTR)mGraphBuilder, GetCurrentProcessId());

	IMoniker* moniker;
	HRESULT hr = CreateItemMoniker(L"!", name, &moniker);
	if (SUCCEEDED(hr)) {
		hr = rot->Register(ROTFLAGS_REGISTRATIONKEEPSALIVE, mGraphBuilder, moniker, &mRotRegister);
		moniker->Release();
	}
	rot->Release();
}

void DirectShowOutput::RemoveFromRot()
{
	if (mRotRegister != 0) {
		IRunningObjectTable* rot;
		if (SUCCEEDED(GetRunningObjectTable(0, &rot))) {
			rot->Revoke(mRotRegister);
			rot->Release();
			mRotRegister = 0;
		}
	}
}

/**********************************************
 *
 *  VideoCompressorEnumerator
 *
 **********************************************/

VideoCompressorEnumerator::VideoCompressorEnumerator(
	long width, long height, LONGLONG frameDuration, BOOL hasAlpha)
	: mWidth(width), mHeight(height), mFrameDuration(frameDuration), mHasAlpha(hasAlpha),
	  mEnumMoniker(NULL), mMalloc(NULL),
	  mDisplayName(NULL), mCapabilities(0), mDefaultQuality(-1),
	  mDefaultKeyFrameRate(-1), mDefaultPFramesPerKey(-1), mDefaultBitRate(0),
	  mCanConfigDialog(FALSE), mCanAboutDialog(FALSE)
{
	VariantInit(&mFriendlyName);
}

VideoCompressorEnumerator::~VideoCompressorEnumerator()
{
	ReleaseCurrent();

	RELEASE(mMalloc);
	RELEASE(mEnumMoniker);
}

HRESULT VideoCompressorEnumerator::ReleaseCurrent()
{
	mCapabilities = 0;
	mDefaultQuality = -1;
	mDefaultKeyFrameRate = -1;
	mDefaultPFramesPerKey = -1;
	mDefaultBitRate = 0;
	mCanConfigDialog = FALSE;
	mCanAboutDialog = FALSE;


	HRESULT hr;

	if (mMalloc == NULL) {
		hr = CoGetMalloc(1, &mMalloc);
		if (FAILED(hr)) return hr;
	}

	mMalloc->Free(mDisplayName);
	mDisplayName = NULL;

	hr = VariantClear(&mFriendlyName);
	if (FAILED(hr)) return hr;

	return S_OK;
}

HRESULT VideoCompressorEnumerator::Next()
{
	HRESULT hr;

	ICreateDevEnum*	sysDevEnum = NULL;
	IMoniker* moniker = NULL;
	IPropertyBag* propBag = NULL;
	IBaseFilter* vcompFilter = NULL;
	IGraphBuilder* dummyGraph = NULL;
	IPin* outPin = NULL;
	IAMVideoCompression* vcomp = NULL;
	IAMVfwCompressDialogs* dialogs = NULL;
	BOOL next = FALSE;

	BAIL_IF_FAILED(bail,
		hr = ReleaseCurrent());

	if (mEnumMoniker == NULL) {
		BAIL_IF_FAILED(bail,
			hr = CoCreateInstance(CLSID_SystemDeviceEnum, NULL, CLSCTX_INPROC_SERVER,
									IID_ICreateDevEnum, (LPVOID*)&sysDevEnum));

		hr = sysDevEnum->CreateClassEnumerator(CLSID_VideoCompressorCategory, &mEnumMoniker, 0);
		if (hr != S_OK) goto bail;
	}

	hr = mEnumMoniker->Next(1, &moniker, NULL);
	if (hr != S_OK) goto bail;

	hr = moniker->GetDisplayName(NULL, NULL, &mDisplayName);
	if (hr != S_OK) { next = TRUE; goto bail; }

	hr = moniker->BindToStorage(0, 0, IID_IPropertyBag, (LPVOID*)&propBag);
	if (hr != S_OK) { next = TRUE; goto bail; }

	hr = propBag->Read(L"FriendlyName", &mFriendlyName, 0);
	if (hr != S_OK) { next = TRUE; goto bail; }

	hr = moniker->BindToObject(NULL, NULL, IID_IBaseFilter, (LPVOID*)&vcompFilter);
	if (hr != S_OK) { next = TRUE; goto bail; }

	hr = VideoCompressorUtil::BuildDummyGraph(mWidth, mHeight, mFrameDuration, mHasAlpha, vcompFilter, &dummyGraph);
	if (hr != S_OK) { next = TRUE; goto bail; }

	hr = VideoCompressorUtil::GetPin(vcompFilter, PINDIR_OUTPUT, &outPin);
	if (hr != S_OK) { next = TRUE; goto bail; }

	hr = outPin->QueryInterface(IID_IAMVideoCompression, (LPVOID*)&vcomp);
	if (hr == S_OK) {
		int versSize, descSize;
		hr = vcomp->GetInfo(NULL, &versSize, NULL, &descSize,
				&mDefaultKeyFrameRate, &mDefaultPFramesPerKey, &mDefaultQuality, &mCapabilities);
	}
	if (hr != S_OK) {
		mCapabilities = 0;
		mDefaultQuality = -1;
		mDefaultKeyFrameRate = -1;
		mDefaultPFramesPerKey = -1;
	}

	if ((mCapabilities & CompressionCaps_CanCrunch) != 0) {
		VideoCompressorUtil::GetBitRate(outPin, &mDefaultBitRate);
	} else {
		mDefaultBitRate = 0;
	}

	hr = vcompFilter->QueryInterface(IID_IAMVfwCompressDialogs, (LPVOID*)&dialogs);
	if (hr == S_OK) {
		mCanConfigDialog = (dialogs->ShowDialog(VfwCompressDialog_QueryConfig, NULL) == S_OK);
		mCanAboutDialog = (dialogs->ShowDialog(VfwCompressDialog_QueryAbout, NULL) == S_OK);
	} else {
		mCanConfigDialog = FALSE;
		mCanAboutDialog = FALSE;
	}

	hr = S_OK;

bail:
	RELEASE(dialogs);
	RELEASE(vcomp);
	RELEASE(outPin);
	RELEASE(dummyGraph);
	RELEASE(vcompFilter);
	RELEASE(propBag);
	RELEASE(moniker);
	RELEASE(sysDevEnum);

	return next ? Next() : hr;
}

/**********************************************
 *
 *  VideoCompressorUtil
 *
 **********************************************/

HRESULT VideoCompressorUtil::BuildDummyGraph(
	long width, long height, LONGLONG frameDuration, BOOL hasAlpha,
	IBaseFilter* vcompFilter, IGraphBuilder** dummyGraph)
{
	HRESULT hr;

	ICaptureGraphBuilder2* builder = NULL;
	IGraphBuilder* graph = NULL;
	IBaseFilter* nullRenderer = NULL;
	CVideoSource* videoSource = NULL;

	BAIL_IF_FAILED(bail,
		hr = CoCreateInstance(CLSID_CaptureGraphBuilder2,
			NULL, CLSCTX_INPROC_SERVER, IID_ICaptureGraphBuilder2, (LPVOID*)&builder));

	BAIL_IF_FAILED(bail,
		hr = CoCreateInstance(CLSID_FilterGraph,
			NULL, CLSCTX_INPROC, IID_IGraphBuilder, (LPVOID*)&graph));

	BAIL_IF_FAILED(bail,
		hr = builder->SetFiltergraph(graph));

	BAIL_IF_FAILED(bail,
		hr = CoCreateInstance(CLSID_NullRenderer,
			NULL, CLSCTX_INPROC, IID_IBaseFilter, (LPVOID*)&nullRenderer));

	videoSource = new CVideoSource(NULL, &hr, width, height, frameDuration, hasAlpha);
	if (videoSource == NULL) {
		hr = E_OUTOFMEMORY;
	} else {
		videoSource->AddRef();
	}
	BAIL_IF_FAILED(bail, hr);

	BAIL_IF_FAILED(bail,
		hr = graph->AddFilter(nullRenderer, L"Null Renderer"));

	BAIL_IF_FAILED(bail,
		hr = graph->AddFilter(videoSource, L"Vidoe Source"));

	BAIL_IF_FAILED(bail,
		hr = graph->AddFilter(vcompFilter, L"Video Compressor"));

	BAIL_IF_FAILED(bail,
		hr = builder->RenderStream(NULL, NULL, (IBaseFilter*)videoSource, vcompFilter, nullRenderer));

	*dummyGraph = graph;
	graph->AddRef();
	hr = S_OK;

bail:
	RELEASE(videoSource);
	RELEASE(nullRenderer);
	RELEASE(graph);
	RELEASE(builder);
	return hr;
}

HRESULT VideoCompressorUtil::GetPin(IBaseFilter* filter, PIN_DIRECTION pindir, IPin** pin)
{
	HRESULT hr;
	IEnumPins* enumPins = NULL;
	IPin* tmpPin = NULL;

	*pin = NULL;

	BAIL_IF_FAILED(bail,
		hr = filter->EnumPins(&enumPins));

	while ((hr = enumPins->Next(1, &tmpPin, NULL)) == S_OK) {
		PIN_DIRECTION pd;

		BAIL_IF_FAILED(bail,
			hr = tmpPin->QueryDirection(&pd));

		if (pd == pindir) {
			*pin = tmpPin;
			tmpPin = NULL;
			hr = S_OK;
			goto bail;
		}

		RELEASE(tmpPin);
	}

bail:
	RELEASE(tmpPin);
	RELEASE(enumPins);
	return hr;
}

HRESULT VideoCompressorUtil::GetBitRate(IPin* pin, LONGLONG* bitRate)
{
	HRESULT hr;
	IAMStreamConfig* streamConfig = NULL;
	AM_MEDIA_TYPE* mediaType = NULL;

	*bitRate = 0;

	BAIL_IF_FAILED(bail,
		hr = pin->QueryInterface(IID_IAMStreamConfig, (LPVOID*)&streamConfig));

	BAIL_IF_FAILED(bail,
		hr = streamConfig->GetFormat(&mediaType));

	if (mediaType->formattype == FORMAT_VideoInfo) {
		VIDEOINFOHEADER* vih = (VIDEOINFOHEADER*)mediaType->pbFormat;
		*bitRate = vih->dwBitRate;
		hr = S_OK;
	} else {
		hr = E_FAIL;
	}

bail:
	if (mediaType != NULL) DeleteMediaType(mediaType);
	RELEASE(streamConfig);
	return hr;
}

HRESULT VideoCompressorUtil::SetBitRate(IPin* pin, LONGLONG bitRate)
{
	HRESULT hr;
	IAMStreamConfig* streamConfig = NULL;
	AM_MEDIA_TYPE* mediaType = NULL;

	BAIL_IF_FAILED(bail,
		hr = pin->QueryInterface(IID_IAMStreamConfig, (LPVOID*)&streamConfig));

	BAIL_IF_FAILED(bail,
		hr = streamConfig->GetFormat(&mediaType));

	if (mediaType->formattype == FORMAT_VideoInfo) {
		VIDEOINFOHEADER* vih = (VIDEOINFOHEADER*)mediaType->pbFormat;
		vih->dwBitRate = (DWORD)bitRate;
		hr = streamConfig->SetFormat(mediaType);
	} else {
		hr = E_FAIL;
	}

bail:
	if (mediaType != NULL) DeleteMediaType(mediaType);
	RELEASE(streamConfig);
	return hr;
}

HRESULT VideoCompressorUtil::OpenConfigDialog(
	long width, long height, LONGLONG frameDuration, BOOL hasAlpha,
	LPCWSTR displayName, void** ioConfig, size_t* ioConfigSize, HWND hwnd)
{
	HRESULT hr;

	IBindCtx* bindCtx = NULL;
	IMoniker* moniker = NULL;
	IBaseFilter* vcompFilter = NULL;
	IGraphBuilder* dummyGraph = NULL;
	IAMVfwCompressDialogs* dialogs = NULL;

	void* buffer = NULL;
	int bufferSize = 0;

	void* config = *ioConfig;
	int configSize = *ioConfigSize;

	*ioConfig = NULL;
	*ioConfigSize = 0;


	BAIL_IF_FAILED(bail,
		hr = CreateBindCtx(0, &bindCtx));

	ULONG eaten;
	BAIL_IF_FAILED(bail,
		hr = MkParseDisplayName(bindCtx, displayName, &eaten, &moniker));

	BAIL_IF_FAILED(bail,
		hr = moniker->BindToObject(bindCtx, NULL, IID_IBaseFilter, (LPVOID*)&vcompFilter));

	BAIL_IF_FAILED(bail,
		hr = VideoCompressorUtil::BuildDummyGraph(
			width, height, frameDuration, hasAlpha, vcompFilter, &dummyGraph));

	BAIL_IF_FAILED(bail,
		hr = vcompFilter->QueryInterface(IID_IAMVfwCompressDialogs, (LPVOID*)&dialogs));

	if (config != NULL) {
		BAIL_IF_FAILED(bail,
			hr = dialogs->SendDriverMessage(ICM_SETSTATE, (long)config, configSize));
	}

	BAIL_IF_FAILED(bail,
		hr = dialogs->ShowDialog(VfwCompressDialog_Config, (HWND)hwnd));

	BAIL_IF_FAILED(bail,
		hr = dialogs->GetState(NULL, &bufferSize));

	buffer = malloc(bufferSize);
	BAIL_IF_FAILED(bail,
		hr = (buffer == NULL) ? E_OUTOFMEMORY : S_OK);

	BAIL_IF_FAILED(bail,
		hr = dialogs->GetState(buffer, &bufferSize));

	*ioConfig = buffer;
	*ioConfigSize = bufferSize;
	buffer = NULL;

	hr = S_OK;

bail:
	if (buffer != NULL) free(buffer);
	RELEASE(dialogs);
	RELEASE(dummyGraph);
	RELEASE(vcompFilter);
	RELEASE(moniker);
	RELEASE(bindCtx);
	return hr;
}

HRESULT VideoCompressorUtil::OpenAboutDialog(LPCWSTR displayName, HWND hwnd)
{
	HRESULT hr;

	IBindCtx* bindCtx = NULL;
	IMoniker* moniker = NULL;
	IBaseFilter* vcompFilter = NULL;
	IAMVfwCompressDialogs* dialogs = NULL;

	BAIL_IF_FAILED(bail,
		hr = CreateBindCtx(0, &bindCtx));

	ULONG eaten;
	BAIL_IF_FAILED(bail,
		hr = MkParseDisplayName(bindCtx, displayName, &eaten, &moniker));

	BAIL_IF_FAILED(bail,
		hr = moniker->BindToObject(bindCtx, NULL, IID_IBaseFilter, (LPVOID*)&vcompFilter));

	BAIL_IF_FAILED(bail,
		hr = vcompFilter->QueryInterface(IID_IAMVfwCompressDialogs, (LPVOID*)&dialogs));

	BAIL_IF_FAILED(bail,
		hr = dialogs->ShowDialog(VfwCompressDialog_About, (HWND)hwnd));

bail:
	RELEASE(dialogs);
	RELEASE(vcompFilter);
	RELEASE(moniker);
	RELEASE(bindCtx);
	return hr;
}
