/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.actions;

import java.util.Map;
import java.util.Set;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.ui.IWorkbenchWindow;

import ch.kuramo.javie.app.project.AddEffectsOperation;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.views.layercomp.EffectElement;
import ch.kuramo.javie.app.views.layercomp.EffectsElement;
import ch.kuramo.javie.app.views.layercomp.Element;
import ch.kuramo.javie.app.views.layercomp.LayerElement;
import ch.kuramo.javie.core.EffectDescriptor;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.Util;

public class AddEffectAction extends Action implements ISelectionChangedListener {

	private final IWorkbenchWindow window;

	private final EffectDescriptor descriptor;

	private final Set<Element> elements = Util.newLinkedHashSet();


	public AddEffectAction(IWorkbenchWindow window, Viewer viewer, EffectDescriptor descriptor) {
		super(descriptor.getLabel().replaceAll("&", "&&"));
		this.window = window;
		this.descriptor = descriptor;

		//setId("EFFECT." + descriptor.getId());
		//setActionDefinitionId("EFFECT." + descriptor.getId());
		//setImageDescriptor(Activator.getImageDescriptor("/icons/add_effect.png"));

		updateState(viewer.getSelection());
		viewer.addSelectionChangedListener(this);
	}

	private void updateState(ISelection selection) {
		elements.clear();

		if (selection instanceof IStructuredSelection) {
			for (Object o : ((IStructuredSelection) selection).toList()) {
				if (o instanceof LayerElement) {
					LayerElement element = (LayerElement) o;
					if (element.layer instanceof EffectableLayer) {
						elements.add(element);
						continue;
					}
				} else if (o instanceof EffectsElement || o instanceof EffectElement) {
					elements.add((Element) o);
					continue;
				}

				elements.clear();
				break;
			}
		}

		setEnabled(!elements.isEmpty());
	}

	public EffectDescriptor getEffectDescriptor() {
		return descriptor;
	}

	public void selectionChanged(SelectionChangedEvent event) {
		updateState(event.getSelection());
	}

	@Override
	public void run() {
		Map<EffectableLayer, Integer> effectIndices = Util.newLinkedHashMap();
		for (Element e : elements) {
			EffectableLayer layer;
			int index;

			if (e instanceof LayerElement) {
				layer = (EffectableLayer) ((LayerElement) e).layer;
				index = layer.getEffects().size();

			} else if (e instanceof EffectsElement) {
				layer = (EffectableLayer) ((LayerElement) e.parent).layer;
				index = layer.getEffects().size();

			} else if (e instanceof EffectElement) {
				layer = (EffectableLayer) ((LayerElement) e.parent.parent).layer;
				index = layer.getEffects().indexOf(((EffectElement) e).effect);

			} else {
				continue;
			}

			if (!effectIndices.containsKey(layer) || effectIndices.get(layer) > index) {
				effectIndices.put(layer, index);
			}
		}

		if (!effectIndices.isEmpty()) {
			ProjectManager pm = ProjectManager.forWorkbenchWindow(window);
			pm.postOperation(new AddEffectsOperation(pm, effectIndices, descriptor));
		}
	}

}
