/*
 * Copyright (c) 2010,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core;

import java.util.Arrays;
import java.util.EnumSet;

import net.arnx.jsonic.JSONHint;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;

public class MediaOptions implements Cloneable {

	public enum Option {
		IGNORE_ALPHA,
		STRAIGHT_ALPHA,
		PREMULTIPLIED_ALPHA,

		VIDEO_FRAME_DURATION,
		DURATION,

		SIZE,

		FLIP_VERTICAL
	}

	public enum AlphaType {
		IGNORE,
		STRAIGHT,
		PREMULTIPLIED
	}


	private EnumSet<Option> availableOptions;

	private AlphaType alphaType;

	private Color colorMatte;

	private Time videoFrameDuration;

	private Time duration;

	private Size2i size;

	private boolean flipVertical;


	public MediaOptions() {
		clearAvailableOptions();
	}

	public MediaOptions(Option... availableOptions) {
		setAvailableOptions(availableOptions);
	}

	public void clearAvailableOptions() {
		setAvailableOptions();
	}

	@JSONHint(ignore=true)
	public void setAvailableOptions(Option... availableOptions) {
		EnumSet<Option> set = (availableOptions.length > 0)
				? EnumSet.copyOf(Arrays.asList(availableOptions))
				: EnumSet.noneOf(Option.class);

		if (set.contains(Option.VIDEO_FRAME_DURATION) && set.contains(Option.DURATION)) {
			throw new IllegalArgumentException(
					"VIDEO_FRAME_DURATION and DURATION cannot be used together.");
		}

		this.availableOptions = set;
		validate();
	}

	private void validate() {
		AlphaType defaultAlphaType;
		if (isAvailable(Option.STRAIGHT_ALPHA)) {
			defaultAlphaType = AlphaType.STRAIGHT;
		} else if (isAvailable(Option.PREMULTIPLIED_ALPHA)) {
			defaultAlphaType = AlphaType.PREMULTIPLIED;
		} else if (isAvailable(Option.IGNORE_ALPHA)) {
			defaultAlphaType = AlphaType.IGNORE;
		} else {
			defaultAlphaType = null;
		}

		if (alphaType == null) {
			alphaType = defaultAlphaType;

		} else {
			switch (alphaType) {
				case IGNORE:
					if (!isAvailable(Option.IGNORE_ALPHA)) alphaType = defaultAlphaType;
					break;
				case STRAIGHT:
					if (!isAvailable(Option.STRAIGHT_ALPHA)) alphaType = defaultAlphaType;
					break;
				case PREMULTIPLIED:
					if (!isAvailable(Option.PREMULTIPLIED_ALPHA)) alphaType = defaultAlphaType;
					break;
			}
		}

		if (colorMatte == null && isAvailable(Option.PREMULTIPLIED_ALPHA)) {
			colorMatte = Color.BLACK;
		}
	}

	@JSONHint(ignore=true)
	public boolean isAvailable() {
		return !availableOptions.isEmpty();
	}

	public boolean isAvailable(Option option) {
		return availableOptions.contains(option);
	}

	public AlphaType getAlphaType() {
		return alphaType;
	}

	public void setAlphaType(AlphaType alphaType) {
		this.alphaType = alphaType;
	}

	public Color getColorMatte() {
		return colorMatte;
	}

	public void setColorMatte(Color colorMatte) {
		this.colorMatte = colorMatte;
	}

	public Time getVideoFrameDuration() {
		return videoFrameDuration;
	}

	public void setVideoFrameDuration(Time videoFrameDuration) {
		this.videoFrameDuration = videoFrameDuration;
	}

	public Time getDuration() {
		return duration;
	}

	public void setDuration(Time duration) {
		this.duration = duration;
	}

	public Size2i getSize() {
		return size;
	}

	public void setSize(Size2i size) {
		this.size = size;
	}

	public boolean isFlipVertical() {
		return flipVertical;
	}

	public void setFlipVertical(boolean flipVertical) {
		this.flipVertical = flipVertical;
	}

	public MediaOptions clone() {
		try {
			MediaOptions clone = (MediaOptions) super.clone();
			clone.availableOptions = this.availableOptions.clone();
			return clone;
		} catch (CloneNotSupportedException e) {
			throw new RuntimeException(e);
		}
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((alphaType == null) ? 0 : alphaType.hashCode());
		result = prime
				* result
				+ ((availableOptions == null) ? 0 : availableOptions.hashCode());
		result = prime * result
				+ ((colorMatte == null) ? 0 : colorMatte.hashCode());
		result = prime * result
				+ ((duration == null) ? 0 : duration.hashCode());
		result = prime * result + (flipVertical ? 1231 : 1237);
		result = prime * result + ((size == null) ? 0 : size.hashCode());
		result = prime
				* result
				+ ((videoFrameDuration == null) ? 0 : videoFrameDuration
						.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		MediaOptions other = (MediaOptions) obj;
		if (alphaType == null) {
			if (other.alphaType != null)
				return false;
		} else if (!alphaType.equals(other.alphaType))
			return false;
		if (availableOptions == null) {
			if (other.availableOptions != null)
				return false;
		} else if (!availableOptions.equals(other.availableOptions))
			return false;
		if (colorMatte == null) {
			if (other.colorMatte != null)
				return false;
		} else if (!colorMatte.equals(other.colorMatte))
			return false;
		if (duration == null) {
			if (other.duration != null)
				return false;
		} else if (!duration.equals(other.duration))
			return false;
		if (flipVertical != other.flipVertical)
			return false;
		if (size == null) {
			if (other.size != null)
				return false;
		} else if (!size.equals(other.size))
			return false;
		if (videoFrameDuration == null) {
			if (other.videoFrameDuration != null)
				return false;
		} else if (!videoFrameDuration.equals(other.videoFrameDuration))
			return false;
		return true;
	}

}
