/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.core.CameraLayer;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.LightLayer;
import ch.kuramo.javie.core.TransformableLayer;
import ch.kuramo.javie.core.services.RenderContext;
import ch.kuramo.javie.core.services.VideoRenderSupport;

class LayerMatrixUtil {

	static void multModelViewMatrix(
			TransformableLayer layer, RenderContext context, VideoRenderSupport vrSupport) {

		LayerMatrixUtil util = new LayerMatrixUtil(context, vrSupport);
		util.multModelViewMatrix(layer, true);
	}

	static void multModelViewMatrix(
			LightLayer layer, RenderContext context, VideoRenderSupport vrSupport) {

		LayerMatrixUtil util = new LayerMatrixUtil(context, vrSupport);
		util.multModelViewMatrix(layer, true);
	}

	static void multParentMatrix(
			CameraLayer cameraLayer, RenderContext context, VideoRenderSupport vrSupport) {

		Layer parent = ((LayerComposition) context.getComposition()).getParentLayer(cameraLayer);
		if (parent != null) {
			new LayerMatrixUtil(context, vrSupport)
					.multModelViewMatrixInverse(parent, true);
		}
	}



	private final RenderContext _context;

	private final VideoRenderSupport _vrSupport;


	private LayerMatrixUtil(RenderContext context, VideoRenderSupport vrSupport) {
		super();
		_context = context;
		_vrSupport = vrSupport;
	}

	private void multModelViewMatrix(Layer layer, boolean threeD) {
		threeD &= LayerNature.isThreeD(layer);

		Layer parent = ((LayerComposition) _context.getComposition()).getParentLayer(layer);
		if (parent != null) {
			multModelViewMatrix(parent, threeD);
		}

		if (layer instanceof TransformableLayer) {
			multTransformableLayerMatrix((TransformableLayer) layer, threeD, false);
		} else if (layer instanceof LightLayer) {
			multLightLayerMatrix((LightLayer) layer, threeD, false);
		} else if (layer instanceof CameraLayer) {
			multCameraLayerMatrix((CameraLayer) layer, threeD, false);
		}
	}

	private void multModelViewMatrixInverse(Layer layer, boolean threeD) {
		threeD &= LayerNature.isThreeD(layer);

		if (layer instanceof TransformableLayer) {
			multTransformableLayerMatrix((TransformableLayer) layer, threeD, true);
		} else if (layer instanceof LightLayer) {
			multLightLayerMatrix((LightLayer) layer, threeD, true);
		} else if (layer instanceof CameraLayer) {
			multCameraLayerMatrix((CameraLayer) layer, threeD, true);
		}

		Layer parent = ((LayerComposition) _context.getComposition()).getParentLayer(layer);
		if (parent != null) {
			multModelViewMatrixInverse(parent, threeD);
		}
	}

	private void multTransformableLayerMatrix(TransformableLayer tlayer, boolean threeD, boolean inverse) {
		// ビデオをサポートしていない場合は計算しても無駄なので何もせずに抜ける。
		// (抜けずに行列計算をしたとしても恒等変換になるので結果に影響は無いはずだが)
		if (!LayerNature.isVideoNature(tlayer)) {
			return;
		}

		Vec3d anchorPoint = tlayer.getAnchorPoint().value(_context);
		Vec3d scale = tlayer.getScale().value(_context);
		Vec3d orientation;
		Vec3d rotation;
		Vec3d position = tlayer.getPosition().value(_context);

		if (threeD) {
			orientation = tlayer.getOrientation().value(_context);
			rotation = new Vec3d(
					tlayer.getRotationX().value(_context),
					tlayer.getRotationY().value(_context),
					tlayer.getRotationZ().value(_context));

		} else {
			anchorPoint = new Vec3d(anchorPoint.x, anchorPoint.y, 0);
			scale = new Vec3d(scale.x, scale.y, 100);
			orientation = new Vec3d(0, 0, 0);
			rotation = new Vec3d(0, 0, tlayer.getRotationZ().value(_context));
			position = new Vec3d(position.x, position.y, 0);
		}

		Resolution resolution = _context.getVideoResolution();

		if (inverse) {
			_vrSupport.multModelViewMatrixInverse(
					resolution.scale(anchorPoint), scale, rotation, orientation, null, resolution.scale(position));
		} else {
			_vrSupport.multModelViewMatrix(
					resolution.scale(anchorPoint), scale, rotation, orientation, null, resolution.scale(position));
		}
	}

	private void multLightLayerMatrix(LightLayer layer, boolean threeD, boolean inverse) {
		Vec3d anchorPoint = Vec3d.ZERO;
		Vec3d scale = new Vec3d(100, 100, 100);
		Vec3d orientation;
		Vec3d rotation;
		Vec3d interest = null;
		Vec3d position = layer.getPosition().value(_context);

		switch (layer.getLightType()) {
			case PARALLEL:
			case SPOT:
				interest = layer.getPointOfInterest().value(_context);
				break;
		}

		if (threeD) {
			orientation = layer.getOrientation().value(_context);
			rotation = new Vec3d(
					layer.getRotationX().value(_context),
					layer.getRotationY().value(_context),
					layer.getRotationZ().value(_context));

		} else {
			// AEでは、2Dレイヤーの親にライトを指定したとき、目標点の値がアンカーポイントとして解釈されていると思われる動作をする。
			// 実装上便宜的にそうなってるだけなのか、何らかの意味があってそうなっているのか不明だが、
			// とりあえずその動作に合わせるためにこのように以下のようにしている。

			if (interest != null) {
				anchorPoint = new Vec3d(interest.x, interest.y, 0);
				interest = null;
			}
			orientation = Vec3d.ZERO;
			rotation = new Vec3d(0, 0, layer.getRotationZ().value(_context));
			position = new Vec3d(position.x, position.y, 0);
		}

		Resolution resolution = _context.getVideoResolution();

		if (inverse) {
			_vrSupport.multModelViewMatrixInverse(
					resolution.scale(anchorPoint), scale, rotation, orientation,
					interest != null ? resolution.scale(interest) : null, resolution.scale(position));
		} else {
			_vrSupport.multModelViewMatrix(
					resolution.scale(anchorPoint), scale, rotation, orientation,
					interest != null ? resolution.scale(interest) : null, resolution.scale(position));
		}
	}

	private void multCameraLayerMatrix(CameraLayer clayer, boolean threeD, boolean inverse) {
		Resolution resolution = _context.getVideoResolution();

		Vec3d position = clayer.getPosition().value(_context);
		Vec3d interest = clayer.getPointOfInterest().value(_context);

		if (threeD) {
			Vec3d orientation = clayer.getOrientation().value(_context);
			Vec3d rotation = new Vec3d(
					clayer.getRotationX().value(_context),
					clayer.getRotationY().value(_context),
					clayer.getRotationZ().value(_context));

			// This is OK: カメラレイヤーを親にしたときの変換は、カメラ自身の変換の逆変換となる。
			if (inverse) {
				_vrSupport.multCameraMatrix(
						rotation, orientation, resolution.scale(interest), resolution.scale(position), false);
			} else {
				_vrSupport.multCameraMatrixInverse(
						rotation, orientation, resolution.scale(interest), resolution.scale(position));
			}
		} else {
			// AEでは、2Dレイヤーの親にカメラを指定したとき、目標点の値がアンカーポイントとして解釈されていると思われる動作をする。
			// 実装上便宜的にそうなってるだけなのか、何らかの意味があってそうなっているのか不明だが、
			// とりあえずその動作に合わせるためにこのように以下のようにしている。

			Vec3d anchorPoint = new Vec3d(interest.x, interest.y, 0);
			Vec3d scale = new Vec3d(100, 100, 100);
			Vec3d orientation = new Vec3d(0, 0, 0);
			Vec3d rotation = new Vec3d(0, 0, clayer.getRotationZ().value(_context));
			position = new Vec3d(position.x, position.y, 0);

			if (inverse) {
				_vrSupport.multModelViewMatrixInverse(
						resolution.scale(anchorPoint), scale, rotation, orientation, null, resolution.scale(position));
			} else {
				_vrSupport.multModelViewMatrix(
						resolution.scale(anchorPoint), scale, rotation, orientation, null, resolution.scale(position));
			}
		}
	}

}
