/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.stylize;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.BlendMode;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableInteger;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.ShaderType;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IBlendSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.LightBurst", category=Categories.STYLIZE)
public class LightBurst {

	public enum RayComposite { ON_TOP, BEHIND, RAY_ONLY }


	@Property
	private IAnimatableVec2d center;

	@Property(value="100", min="0"/*, max="1000"*/)
	private IAnimatableDouble intensity;

	@Property(value="0", min="0", max="200")
	private IAnimatableDouble rayLength;

//	@Property
//	private IAnimatableEnum<Burst> burst;
//
//	@Property
//	private IAnimatableBoolean haloAlpha;

	@Property("false")
	private IAnimatableBoolean setColor;

	@Property("1.0,0.8,0.4")
	private IAnimatableColor color;

	@Property("ON_TOP")
	private IAnimatableEnum<RayComposite> rayComposite;

	@Property("ADD")
	private IAnimatableEnum<BlendMode> blendMode;

	@Property(value="200", min="0")
	private IAnimatableInteger maxSamples;

	@Property("true")
	private IAnimatableBoolean fast;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IBlendSupport blendSupport;

	private final IShaderRegistry shaders;

	@Inject
	public LightBurst(
			IVideoEffectContext context, IVideoRenderSupport support,
			IBlendSupport blendSupport, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		this.blendSupport = blendSupport;
		this.shaders = shaders;
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();
		VideoBounds bounds = source.getBounds();
		if (bounds.isEmpty()) {
			return source;
		}

		Resolution resolution = context.getVideoResolution();
		Vec2d center = resolution.scale(context.value(this.center));
		double intensity = context.value(this.intensity) / 100;
		double rayLength = resolution.scale(context.value(this.rayLength));
		Color color = context.value(this.setColor) ? context.value(this.color) : null;
		RayComposite rayComposite = context.value(this.rayComposite);
		BlendMode blendMode = context.value(this.blendMode);
		int maxSamples = context.value(this.maxSamples);
		boolean fast = context.value(this.fast);

		Set<IVideoBuffer> tmpBuffers = new HashSet<IVideoBuffer>();
		tmpBuffers.add(source);

		try {
			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			IVideoBuffer buffer = source;

			if (fast) {
				uniforms.add(new GLUniformData("amount", (float)(rayLength/resolution.scale(200.0) / maxSamples * 5)));
				uniforms.add(new GLUniformData("source", 0));
				uniforms.add(new GLUniformData("center", 2, toFloatBuffer(center.x-bounds.x, center.y-bounds.y)));
				uniforms.add(new GLUniformData("size", 2, toFloatBuffer(bounds.width, bounds.height)));
				uniforms.add(new GLUniformData("maxSamples", (float)5));

				for (int i = 0; i < 2; ++i) {
					buffer.setTextureFilter(TextureFilter.LINEAR);
					buffer = support.useShaderProgram(getPreprocessProgram(), uniforms, null, buffer);
					tmpBuffers.add(buffer);
				}

				uniforms.clear();
				maxSamples = (int) Math.ceil(maxSamples / 5.0);
			}

			IShaderProgram program;
			if (color == null) {
				program = getSourceColorProgram();
			} else {
				program = getSetColorProgram();
				uniforms.add(new GLUniformData("color", 3, toFloatBuffer(color.r, color.g, color.b)));
			}

			uniforms.add(new GLUniformData("amount", (float)(rayLength/resolution.scale(200.0))));
			uniforms.add(new GLUniformData("source", 0));
			uniforms.add(new GLUniformData("center", 2, toFloatBuffer(center.x-bounds.x, center.y-bounds.y)));
			uniforms.add(new GLUniformData("size", 2, toFloatBuffer(bounds.width, bounds.height)));
			uniforms.add(new GLUniformData("intensity", (float)intensity));
			uniforms.add(new GLUniformData("maxSamples", (float)maxSamples));

			buffer.setTextureFilter(TextureFilter.LINEAR);

			buffer = support.useShaderProgram(program, uniforms, null, buffer);
			tmpBuffers.add(buffer);

			source.setTextureFilter(TextureFilter.NEAREST);

			switch (rayComposite) {
				case ON_TOP:
					return blendSupport.blend(buffer, source, null, blendMode, 1.0, context);
				case BEHIND:
					return blendSupport.blend(source, buffer, null, blendMode, 1.0, context);
				//case RAY_ONLY:
				default:
					tmpBuffers.remove(buffer);
					return buffer;
			}

		} finally {
			for (IVideoBuffer vb : tmpBuffers) {
				vb.dispose();
			}
		}
	}

	private FloatBuffer toFloatBuffer(double... values) {
		float[] array = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			array[i] = (float)values[i];
		}
		return FloatBuffer.wrap(array);
	}

	private IShaderProgram getPreprocessProgram() {
		return getProgram(true, false);
	}

	private IShaderProgram getSourceColorProgram() {
		return getProgram(false, false);
	}

	private IShaderProgram getSetColorProgram() {
		return getProgram(false, true);
	}

	private IShaderProgram getProgram(boolean preprocess, boolean setColor) {
		setColor &= !preprocess;
		String programName = LightBurst.class.getName()
							+ (preprocess ? ".PREPROCESS" : "")
							+ (setColor ? ".SET_COLOR" : "");
		IShaderProgram program = shaders.getProgram(programName);
		if (program == null) {
			String[] source = createProgramSource(preprocess, setColor);
			program = shaders.registerProgram(programName, ShaderType.FRAGMENT_SHADER, null, source);
		}
		return program;
	}

	private static String[] createProgramSource(boolean preprocess, boolean setColor) {
		boolean p = preprocess;
		boolean c = !preprocess && setColor;

		return new String[] {
			p ? "#define PREPROCESS" : "",
			c ? "#define SET_COLOR" : "",
				"",
				"uniform sampler2D source;",
				"uniform float amount;",
				"uniform vec2 center;",
				"uniform vec2 size;",
				"uniform float maxSamples;",
				"",
				"#ifndef PREPROCESS",
				"	uniform float intensity;",
				"#endif",
				"",
				"#ifdef SET_COLOR",
				"	uniform vec3 color;",
				"#endif",
				"",
				"const float PI_0_5 = 3.14159265358979323846264 * 0.5;",
				"",
				"void main(void)",
				"{",
				"	vec2 coord = gl_TexCoord[0].st;",
				"	vec2 v = center - coord * size;",
				"	float d = length(v);",
				"",
				"	vec4 srcSum = texture2D(source, coord);",
				"",
				"	if (d > 0.0) {",
				"		v /= size;",
				"		int n = 0;",
				"",
				"		int   n1 = int(min(amount*d+1.0, maxSamples));",
				"		float rn1 = 1.0/float(n1);",
				"		vec2  v1 = v*amount*rn1;",
				"		for (int i = 1; i <= n1; ++i) {",
				"#ifdef PREPROCESS",
				"			srcSum += texture2D(source, coord + v1*float(i));",
				"#else",
				"			float fi = float(i);",
				"			float t = 1.0 - fi*rn1;",
				"			srcSum += texture2D(source, coord + v1*fi) * t;",
				"#endif",
				"		}",
				"		n += n1;",
				"",
				"#ifdef PREPROCESS",
				"		int   n2 = int(min(amount*d+1.0, maxSamples));",
				"		vec2  v2 = v*amount/float(n2);",
				"		for (int i = 1; i <= n2; ++i) {",
				"			srcSum += texture2D(source, coord - v2*float(i));",
				"		}",
				"		n += n2;",
				"#endif",
				"",
				"		srcSum /= float(n+1);",
				"	}",
				"",
				"#ifdef PREPROCESS",
				"	gl_FragColor = srcSum;",
				"#else",
				"",
				"#ifdef SET_COLOR",
				"	float lum = dot(srcSum.rgb, vec3(0.299, 0.587, 0.114));",
				"	vec4 fc = vec4(color*lum, lum);",
				"#else",
				"	vec4 fc = srcSum;",
				"#endif",
				"",
//				"	gl_FragColor = vec4(1.0) - pow(vec4(1.0) - fc*2.0, vec4(intensity));",	// こっちの式だと、何故か同心円状の変な模様が出てしまう。
				"	gl_FragColor = sin(PI_0_5 * min(fc*2.0*intensity, 1.0));",
				"#endif",
				"}"
		};
	}

}
