/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.api;

/**
 * <p><code>Vec2d</code>クラスは、2次元空間上のベクトルまたは座標を表します。
 * 値は2つのdouble値で保持します。</p>
 * 
 * <p><code>Vec2d</code>オブジェクトは不変です。
 * <code>Vec2d</code>オブジェクトは作成したあとに変更できないため、共用することができます。</p>
 */
public final class Vec2d {

	/**
	 * <code>x</code>, <code>y</code> が <code>0</code>
	 * の <code>Vec2d</code> オブジェクトです。
	 */
	public static final Vec2d ZERO = new Vec2d(0, 0);

	/**
	 * <code>x</code>, <code>y</code> が <code>Double.POSITIVE_INFINITY</code>
	 * の <code>Vec2d</code> オブジェクトです。
	 */
	public static final Vec2d POSITIVE_INFINITY = new Vec2d(Double.POSITIVE_INFINITY, Double.POSITIVE_INFINITY);

	/**
	 * <code>x</code>, <code>y</code> が <code>Double.NEGATIVE_INFINITY</code>
	 * の <code>Vec2d</code> オブジェクトです。
	 */
	public static final Vec2d NEGATIVE_INFINITY = new Vec2d(Double.NEGATIVE_INFINITY, Double.NEGATIVE_INFINITY);


	/**
	 * ベクトルまたは座標の <code>x</code> の値です。
	 */
	public final double x;

	/**
	 * ベクトルまたは座標の <code>y</code> の値です。
	 */
	public final double y;


	/**
	 * <code>x</code>, <code>y</code> の値を指定して
	 * <code>Vec2d</code> オブジェクトを生成します。
	 * 
	 * @param x ベクトルまたは座標の <code>x</code> の値
	 * @param y ベクトルまたは座標の <code>y</code> の値
	 */
	public Vec2d(double x, double y) {
		super();
		this.x = x;
		this.y = y;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		long temp;
		temp = Double.doubleToLongBits(x);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		temp = Double.doubleToLongBits(y);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		return result;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Vec2d other = (Vec2d) obj;
		if (Double.doubleToLongBits(x) != Double.doubleToLongBits(other.x))
			return false;
		if (Double.doubleToLongBits(y) != Double.doubleToLongBits(other.y))
			return false;
		return true;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return "Vec2d [x=" + x + ", y=" + y + "]";
	}

	/**
	 * 引数に指定した2つの <code>Vec2d</code> オブジェクトの <code>x</code>, <code>y</code>
	 * をそれぞれ比較し、小さい方の値を用いて新たな <code>Vec2d</code> オブジェクトを生成します。
	 * 
	 * @param a 比較対象の1つ目の <code>Vec2d</code> オブジェクト
	 * @param b 比較対象の2つ目の <code>Vec2d</code> オブジェクト
	 * @return	<code>x</code>, <code>y</code>
	 * 			それぞれの小さい方の値を用いて作成した <code>Vec2d</code> オブジェクト
	 */
	public static Vec2d min(Vec2d a, Vec2d b) {
		return new Vec2d(Math.min(a.x, b.x), Math.min(a.y, b.y));
	}

	/**
	 * 引数に指定した2つの <code>Vec2d</code> オブジェクトの <code>x</code>, <code>y</code>
	 * をそれぞれ比較し、大きい方の値を用いて新たな <code>Vec2d</code> オブジェクトを生成します。
	 * 
	 * @param a 比較対象の1つ目の <code>Vec2d</code> オブジェクト
	 * @param b 比較対象の2つ目の <code>Vec2d</code> オブジェクト
	 * @return	<code>x</code>, <code>y</code>
	 * 			それぞれの大きい方の値を用いて作成した <code>Vec2d</code> オブジェクト
	 */
	public static Vec2d max(Vec2d a, Vec2d b) {
		return new Vec2d(Math.max(a.x, b.x), Math.max(a.y, b.y));
	}

}
