/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Arrays;
import java.util.Set;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.Platform;
import org.eclipse.swt.SWT;
import org.scannotation.AnnotationDB;
import org.scannotation.ClasspathUrlFinder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.kuramo.javie.api.plugin.JaviePlugin;
import ch.kuramo.javie.core.JavieRuntimeException;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.EffectRegistry;
import ch.kuramo.javie.core.services.ProjectElementFactory;
import ch.kuramo.javie.core.services.ShaderRegistry;
import ch.kuramo.javie.core.services.ShaderRegistryException;

import com.google.inject.Inject;

public class Initializer {

	public static void initialize() {
		Initializer initializer = InjectorHolder.getInjector().getInstance(Initializer.class);

		SplashProgressMonitor.subTask("Scanning Core Classes");
		initializer.scanCoreClasses();
		SplashProgressMonitor.worked(1);

		SplashProgressMonitor.subTask("Scanning Extensions");
		initializer.scanExtensions();
		SplashProgressMonitor.worked(1);

		SplashProgressMonitor.subTask("Scanning EzPlugins");
		initializer.scanEasyPlugins();
		SplashProgressMonitor.worked(1);

		SplashProgressMonitor.subTask("");
	}


	private static final Logger _logger = LoggerFactory.getLogger(Initializer.class);

	@Inject
	private EffectRegistry _effectRegistry;

	@Inject
	private ProjectElementFactory _elementFactory;

	@Inject
	private ShaderRegistry _shaderRegistry;


	Initializer() {
		super();
	}

	private void scanCoreClasses() {
		try {
			URL url = FileLocator.resolve(ClasspathUrlFinder.findClassBase(Project.class));
			AnnotationDB db = new AnnotationDB();
			db.scanArchives(url);

			ClassLoader cl = Project.class.getClassLoader();
			_effectRegistry.searchClasses(db, cl);
			_elementFactory.searchClasses(db, cl);
			_shaderRegistry.scanShaders(db, cl);

		} catch (IOException e) {
			throw new JavieRuntimeException(e);
		} catch (ShaderRegistryException e) {
			throw new JavieRuntimeException(e);
		}
	}

	private void scanExtensions() {
		IExtensionRegistry registry = Platform.getExtensionRegistry();
		IExtensionPoint point = registry.getExtensionPoint("ch.kuramo.javie.api.plugin");
		if (point == null) {
			return;
		}

		for (IExtension ext : point.getExtensions()) {
			for (IConfigurationElement cfgElem : ext.getConfigurationElements()) {
				if (!"javie-plugin".equals(cfgElem.getName())) {
					continue;
				}

				String id = ext.getUniqueIdentifier();
				String name = ext.getLabel();
				_logger.info(String.format("loading JaviePlugin: %s [%s]", name, id));
				SplashProgressMonitor.subTask("Scanning Extension: " + name);

				JaviePlugin plugin;
				try {
					plugin = (JaviePlugin) cfgElem.createExecutableExtension("class");
				} catch (CoreException e) {
					_logger.warn("can't load JaviePlugin: " + id, e);
					continue;
				} catch (ClassCastException e) {
					_logger.warn("can't load JaviePlugin: " + id, e);
					continue;
				}

				try {
					Class<?> clazz = plugin.getClass();
					URL url = FileLocator.resolve(ClasspathUrlFinder.findClassBase(clazz));
					AnnotationDB db = new AnnotationDB();
					db.scanArchives(url);

					ClassLoader cl = clazz.getClassLoader();
					_effectRegistry.searchClasses(db, cl);
					_elementFactory.searchClasses(db, cl);
					_shaderRegistry.scanShaders(db, cl);

				} catch (Exception e) {
					_logger.warn("can't load JaviePlugin: " + id, e);
				}
			}
		}
	}

	private void scanEasyPlugins() {
		String path = System.getProperty("javie.ezplugins.dir");
		if (path == null) {
			if (!new File("Javie.ini").isFile()) {
				// Javie.ini が無い場合は開発環境から実行してるとみなして、以降の処理は行わない。
				return;
			}
			if (SWT.getPlatform().equals("cocoa")) {
				path = "../../../ezplugins";
			} else {
				path = "ezplugins";
			}
		}

		File folder = new File(path);
		if (!folder.exists()) {
			_logger.warn("no ezplugins directory found: " + folder.getAbsolutePath());
			return;
		} else if (!folder.isDirectory()) {
			_logger.warn("path of ezplugins is not directory: " + folder.getAbsolutePath());
			return;
		}

		File[] jars = folder.listFiles(new FilenameFilter() {
			public boolean accept(File dir, String name) {
				return name.toLowerCase().endsWith(".jar");
			}
		});

		// ファイル名順に読み込むためにソートする。
		Arrays.sort(jars);

		// ファイル名順を維持するために LinkedHashSet を使う。
		Set<URL> urls = Util.newLinkedHashSet();
		for (File jar : jars) {
			try {
				urls.add(jar.toURI().toURL());
			} catch (MalformedURLException e) {
				_logger.warn("can't determine URL of jar file: " + jar.getAbsolutePath(), e);
			}
		}

		if (urls.isEmpty()) {
			return;
		}

		// TODO 親ローダはこれでいいのか？
		ClassLoader parent = this.getClass().getClassLoader();
		URLClassLoader cl = new URLClassLoader(urls.toArray(new URL[urls.size()]), parent);

		for (URL url : cl.getURLs()) {
			String name = url.getFile();
			int lastSlash = name.lastIndexOf('/');
			if (lastSlash != -1) {
				name = name.substring(lastSlash+1);
			}
			_logger.info(String.format("loading ezplugin: %s", name));
			SplashProgressMonitor.subTask("Scanning EzPlugin: " + name);

			try {
				AnnotationDB db = new AnnotationDB();
				db.scanArchives(url);

				_effectRegistry.searchClasses(db, cl);
				_elementFactory.searchClasses(db, cl);
				_shaderRegistry.scanShaders(db, cl);

			} catch (Exception e) {
				_logger.warn("can't load ezplugin: " + name, e);
			}
		}
	}

}
