/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Collection;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.app.PropertyUtil;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class DuplicateLayersOperation extends ProjectOperation {

	private final String _compId;

	private final List<Object[]> _data;

	private final Time _splitTime;

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;


	public DuplicateLayersOperation(ProjectManager pm, Collection<Layer> layers, int insertIndex) {
		this(pm, layers, insertIndex, null);
	}

	public DuplicateLayersOperation(ProjectManager pm, Collection<Layer> layers, Time splitTime) {
		this(pm, layers, -1, splitTime);
		setLabel("レイヤーの分割");
	}

	public DuplicateLayersOperation(ProjectManager pm, Collection<Layer> layers, int insertIndex, Time splitTime) {
		super(pm, "レイヤーの複製");
		InjectorHolder.getInjector().injectMembers(this);

		if (layers.size() == 0) {
			_noEffect = true;
			_compId = null;
			_data = null;
			_splitTime = null;
			return;
		}

		Set<Layer> layerSet = Util.newSet(layers);
		if (layerSet.size() != layers.size()) {
			throw new IllegalArgumentException("duplicated elements exist in layers argument");
		}

		LayerComposition comp = pm.checkLayer(layerSet.iterator().next());
		_compId = comp.getId();

		if (insertIndex < -1 || insertIndex > comp.getLayers().size()) {
			throw new IllegalArgumentException("insertIndex is out of range");
		}
		if (splitTime != null && insertIndex != -1) {
			throw new IllegalArgumentException("insertIndex must be -1 when splitTime is specified");
		}

		_data = Util.newList();
		_splitTime = splitTime;

		for (ListIterator<Layer> it = comp.getLayers().listIterator(); it.hasNext(); ) {
			int index = it.nextIndex();
			Layer layer = it.next();
			if (layerSet.remove(layer)) {
				if (splitTime != null && !(splitTime.after(layer.getInPoint()) && splitTime.before(layer.getOutPoint()))) {
					continue;
				}
				if (splitTime != null) {
					_data.add(new Object[] { index+1, _encoder.encodeElement(layer), Util.randomId(), layer.getId(), layer.getOutPoint() });
				} else if (insertIndex == -1) {
					_data.add(new Object[] { index+1, _encoder.encodeElement(layer), Util.randomId() });
				} else {
					_data.add(new Object[] { insertIndex, _encoder.encodeElement(layer), Util.randomId() });
				}
			}
		}
		if (!layerSet.isEmpty()) {
			throw new IllegalArgumentException("all layers must belong to same composition");
		}

		_noEffect = _data.isEmpty();
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);
		List<Layer> layers = comp.getLayers();
		List<Layer> newLayers = Util.newList();
		List<Layer> origLayers = null;

		if (_splitTime != null) {
			origLayers = Util.newList();
		}

		try {
			for (ListIterator<Object[]> it = _data.listIterator(_data.size()); it.hasPrevious(); ) {
				Object[] o = it.previous();
				Integer index = (Integer) o[0];
				String data = (String) o[1];
				String newId = (String) o[2];

				Layer layer = _decoder.decodeElement(data, Layer.class);
				PropertyUtil.setProperty(layer, "id", newId);
				layer.afterDecode(project, comp);

				if (_splitTime != null) {
					layer.setInPoint(_splitTime);
					origLayers.add(comp.getLayer((String)o[3]));
				}

				layers.add(index, layer);
				newLayers.add(layer);
			}
		} catch (ProjectDecodeException e) {
			layers.removeAll(newLayers);
			throw new ExecutionException("error decoding Layer data", e);
		}

		if (_splitTime != null) {
			for (Layer layer : origLayers) {
				layer.setOutPoint(_splitTime);
			}
		}

		if (pm != null) {
			if (_splitTime != null) {
				pm.fireLayerTimesChange(comp, origLayers);
			}
			pm.fireLayersAdd(comp, newLayers);
		}

		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);
		List<Layer> undoingLayers = Util.newList();
		List<Layer> origLayers = null;

		if (_splitTime != null) {
			origLayers = Util.newList();
		}

		for (Object[] o : _data) {
			int index = (Integer) o[0];
			undoingLayers.add(comp.getLayers().remove(index));

			if (_splitTime != null) {
				Layer layer = comp.getLayer((String)o[3]);
				layer.setOutPoint((Time)o[4]);
				origLayers.add(layer);
			}
		}

		if (pm != null) {
			if (_splitTime != null) {
				pm.fireLayerTimesChange(comp, origLayers);
			}
			pm.fireLayersRemove(comp, undoingLayers);
		}

		return Status.OK_STATUS;
	}

}
