/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Map;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.Util;

public class ModifyLayerCompSettingsOperation extends ProjectOperation {

	private final String _compItemId;

	private final Map<String, Object[]> _values = Util.newMap();


	public ModifyLayerCompSettingsOperation(
			ProjectManager pm, CompositionItem compItem,
			String name, ColorMode colorMode, Size2i size,
			Time frameDuration, Time duration, boolean frameDurationPreserved,
			double motionBlurShutterAngle, double motionBlurShutterPhase, int motionBlurSamples) {

		super(pm, "コンポジション設定の変更", true);

		pm.checkItem(compItem);
		_compItemId = compItem.getId();

		LayerComposition comp = (LayerComposition) compItem.getComposition(); 

		_noEffect = true;
		putValues("name",					compItem.getName(),					name);
		putValues("colorMode",				comp.getColorMode(),				colorMode);
		putValues("size",					comp.getSize(),						size);
		putValues("frameDuration",			comp.getFrameDuration(),			frameDuration);
		putValues("duration",				comp.getDuration(),					duration);
		putValues("frameDurationPreserved",	comp.isFrameDurationPreserved(),	frameDurationPreserved);
		putValues("motionBlurShutterAngle",	comp.getMotionBlurShutterAngle(),	motionBlurShutterAngle);
		putValues("motionBlurShutterPhase",	comp.getMotionBlurShutterPhase(),	motionBlurShutterPhase);
		putValues("motionBlurSamples",		comp.getMotionBlurSamples(),		motionBlurSamples);
	}

	private <T> void putValues(String name, T oldValue, T newValue) {
		_values.put(name, new Object[] { oldValue, newValue });
		_noEffect &= oldValue.equals(newValue);
	}

	private <T> T getValue(String name, boolean undo) {
		@SuppressWarnings("unchecked")
		T value = (T) _values.get(name)[undo ? 0 : 1];
		return value;
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		modifySettings(project, pm, false);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		modifySettings(project, pm, true);
		return Status.OK_STATUS;
	}

	private void modifySettings(Project project, ProjectManager pm, boolean undo) {
		CompositionItem compItem = project.getItem(_compItemId);
		LayerComposition comp = (LayerComposition) compItem.getComposition();

		compItem.setName(				(String)	getValue("name",					undo));
		comp.setColorMode(				(ColorMode)	getValue("colorMode",				undo));
		comp.setSize(					(Size2i)	getValue("size",					undo));
		comp.setFrameDuration(			(Time)		getValue("frameDuration",			undo));
		comp.setDuration(				(Time)		getValue("duration",				undo));
		comp.setFrameDurationPreserved(	(Boolean)	getValue("frameDurationPreserved",	undo));
		comp.setMotionBlurShutterAngle(	(Double)	getValue("motionBlurShutterAngle",	undo));
		comp.setMotionBlurShutterPhase(	(Double)	getValue("motionBlurShutterPhase",	undo));
		comp.setMotionBlurSamples(		(Integer)	getValue("motionBlurSamples",		undo));

		if (pm != null) {
			pm.fireItemNameChange(compItem);
			pm.fireCompositionSettingsChange(comp);
		}
	}

	public String getCompositionItemId() {
		return _compItemId;
	}

}
