/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Collections;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.core.Folder;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.SolidColorItem;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectElementFactory;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class NewSolidColorItemOperation extends ProjectOperation {

	private final String _itemId;

	private final String _data;

	@Inject
	private ProjectElementFactory _elementFactory;

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;


	public NewSolidColorItemOperation(ProjectManager projectManager,
			Folder parent, String name, int width, int height, Color color) {
		super(projectManager, "新規平面");
		projectManager.checkItem(parent);

		InjectorHolder.getInjector().injectMembers(this);

		SolidColorItem item = _elementFactory.newSolidColorItem(color, new Size2i(width, height));
		item.setName(name);
		item.setParent(parent);

		_itemId = item.getId();
		_data = _encoder.encodeElement(item);
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		SolidColorItem item = null;

		try {
			item = _decoder.decodeElement(_data, SolidColorItem.class);
			item.afterDecode(project);
		} catch (ProjectDecodeException e) {
			if (item != null) {
				item.dispose();
			}
			throw new ExecutionException("error decoding SolidColorItem data", e);
		}

		project.getItems().add(item);

		if (pm != null) {
			fireItemsAdd(item, pm);
		}

		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		Item item = project.getItem(_itemId);
		project.getItems().remove(item);

		if (pm != null) {
			fireItemsRemove(item, pm);
		}

		item.dispose();
		return Status.OK_STATUS;
	}

	private void fireItemsAdd(Item item, ProjectManager pm) {
		Folder parent = item.getParent();
		if (parent != null) {
			pm.fireItemsAdd(Collections.singleton(item), Collections.singleton(parent));
		} else {
			pm.fireItemsAdd(Collections.singleton(item), null);
		}
	}

	private void fireItemsRemove(Item item, ProjectManager pm) {
		Folder parent = item.getParent();
		if (parent != null) {
			pm.fireItemsRemove(Collections.singleton(item), Collections.singleton(parent));
		} else {
			pm.fireItemsRemove(Collections.singleton(item), null);
		}
	}

}
