/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Arrays;
import java.util.List;
import java.util.Set;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.TAProperty;
import ch.kuramo.javie.core.TARangeSelector;
import ch.kuramo.javie.core.TextAnimator;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectElementFactory;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class NewTextAnimatorOperation extends ProjectOperation {

	private final String compId;

	private final String layerId;

	private final String data;

	@Inject
	private ProjectElementFactory elementFactory;

	@Inject
	private ProjectEncoder encoder;

	@Inject
	private ProjectDecoder decoder;


	public NewTextAnimatorOperation(ProjectManager pm, TextLayer layer, TAProperty...properties) {
		super(pm, "テキストアニメータの追加");
		InjectorHolder.getInjector().injectMembers(this);

		LayerComposition comp = pm.checkLayer(layer);
		compId = comp.getId();
		layerId = layer.getId();

		TextAnimator animator = elementFactory.newTextAnimator();
		animator.setName(uniqueAnimatorNames(layer, animator.getName()));

		TARangeSelector selector = elementFactory.newTARangeSelector();
		selector.setName(selector.getName() + " 1");
		animator.getSelectors().add(selector);

		animator.getProperties().addAll(Arrays.asList(properties));

		data = encoder.encodeElement(animator);
	}

	private String uniqueAnimatorNames(TextLayer layer, String prefix) {
		Set<String> names = Util.newSet();
		for (TextAnimator animator : layer.getTextAnimators()) {
			names.add(animator.getName());
		}

		int n = 1;
		String name;
		while (true) {
			name = prefix + " " + (n++);
			if (!names.contains(name)) {
				return name;
			}
		}
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		TextAnimator ta;
		try {
			ta = decoder.decodeElement(data, TextAnimator.class);
		} catch (ProjectDecodeException e) {
			throw new ExecutionException("error decoding TextAnimator data", e);
		}

		LayerComposition comp = project.getComposition(compId);
		TextLayer layer = (TextLayer) comp.getLayer(layerId);

		List<TextAnimator> list = layer.getTextAnimators();
		int index = list.size();
		list.add(ta);

		if (pm != null) {
			fireTextAnimatorsAddRemove(true, comp, layer, index, pm);
		}

		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(compId);
		TextLayer layer = (TextLayer) comp.getLayer(layerId);

		List<TextAnimator> list = layer.getTextAnimators();
		int index = list.size() - 1;
		list.remove(index);

		if (pm != null) {
			fireTextAnimatorsAddRemove(false, comp, layer, index, pm);
		}

		return Status.OK_STATUS;
	}

	private Object[][] createData(LayerComposition comp, TextLayer layer, int animatorIndex, boolean add) {
		List<Object[]> data = Util.newList();

		if (add) {
			data.add(new Object[] { layer, animatorIndex, "new" });
		} else {
			// TODO 削除のときはインデックスを -1 などにしておいた方がいいかも。
			data.add(new Object[] { layer, animatorIndex });
		}

		return data.toArray(new Object[data.size()][]);
	}

	private void fireTextAnimatorsAddRemove(boolean add, LayerComposition comp, TextLayer layer, int animatorIndex, ProjectManager pm) {
		Object[][] data = createData(comp, layer, animatorIndex, add);

		if (add) {
			pm.fireTextAnimatorsAdd(comp, data);
		} else {
			pm.fireTextAnimatorsRemove(comp, data);
		}
	}

}
