/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.TextAnimator;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class ReplaceTextAnimatorOperation extends ProjectOperation {

	private final String compId;

	private final String layerId;

	private final int animatorIndex;

	private final String oldData;

	private final String newData;

	@Inject
	private ProjectEncoder encoder;

	@Inject
	private ProjectDecoder decoder;


	public ReplaceTextAnimatorOperation(ProjectManager pm,
			TextLayer layer, int animatorIndex, TextAnimator animator) {

		super(pm, "テキストアニメータの設定");
		InjectorHolder.getInjector().injectMembers(this);

		LayerComposition comp = pm.checkLayer(layer);
		this.compId = comp.getId();
		this.layerId = layer.getId();
		this.animatorIndex = animatorIndex;
		this.oldData = encoder.encodeElement(layer.getTextAnimators().get(animatorIndex));
		this.newData = encoder.encodeElement(animator);
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		replace(newData, project, pm);
		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		replace(oldData, project, pm);
		return Status.OK_STATUS;
	}

	private void replace(String data, Project project, ProjectManager pm) throws ExecutionException {
		TextAnimator animator;
		try {
			animator = decoder.decodeElement(data, TextAnimator.class);
		} catch (ProjectDecodeException e) {
			throw new ExecutionException("error decoding TextAnimator data", e);
		}

		LayerComposition comp = project.getComposition(compId);
		TextLayer layer = (TextLayer) comp.getLayer(layerId);

		layer.getTextAnimators().set(animatorIndex, animator);

		if (pm != null) {
			pm.fireLayerPropertyChange(layer, String.format("textAnimators[%d]", animatorIndex));
		}
	}

}
