/*
 * Copyright (c) 2010,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.AnimatableDouble;

public class AngleElementDelegate extends AnimatableValueElementDelegate<Double> {

	protected final double increment;

	protected final int displayPrecision;

	protected final int editorPrecision;

	protected final String unit;

	protected AngleCellEditor editor;


	public AngleElementDelegate(
			AnimatableValueElement element, String name, AnimatableDouble avalue) {

		this(element, name, avalue, null);
	}

	public AngleElementDelegate(
			AnimatableValueElement element, String name, AnimatableDouble avalue, String unit) {

		this(element, name, avalue, 1, 1, 4, unit);
	}

	public AngleElementDelegate(
			AnimatableValueElement element, String name, AnimatableDouble avalue,
			double increment, int displayPrecision, int editorPrecision, String unit) {

		super(element, name, avalue);
		this.increment = increment;
		this.displayPrecision = displayPrecision;
		this.editorPrecision = editorPrecision;
		this.unit = unit;
	}

	protected void drawValue(GC gc, int x, int y, int height, boolean focused) {
		if (editor != null) {
			return;
		}

		x = drawValue(gc, x, y, height, focused, (int)(valueToDraw/360), "x");
		x += 5;
		x = drawValue(gc, x, y, height, focused, valueToDraw%360, true, displayPrecision, unit);
	}

	protected int drawValue(GC gc, int x, int y, int height, int valueIndex, double value) {
		if (editor == null) {
			throw new IllegalStateException("no editor");
		}

		if (this.valueIndex == valueIndex) {
			Rectangle bounds = editor.getTextBounds();
			if (valueIndex == 0) {
				setValueForeground(gc, false);
				return drawString(gc, x + bounds.width + (WIN32 ? 0 : 3), y, height, "x");
			} else if (unit != null) {
				setValueForeground(gc, false);
				return drawString(gc, x + bounds.width + (WIN32 ? 0 : 3), y, height, unit);
			} else {
				return x + bounds.width;
			}
		} else if (valueIndex == 0) {
			return drawValue(gc, x, y, height, false, (int)(value/360), "x");
		} else /*if (valueIndex == 1)*/ {
			return drawValue(gc, x, y, height, false, value%360, true, displayPrecision, unit);
		}
	}

	protected boolean canDragGestureEdit() {
		return true;
	}

	protected Double dragGesture(double dx, double dy) {
		return originalValue + increment * dx * (valueIndex == 0 ? 360 : 1);
	}

	protected boolean isInPlaceEditorActive() {
		return (editor != null);
	}

	public boolean canEdit(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			return canInPlaceEdit;
		}
		return false;
	}

	public CellEditor getCellEditor(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			if (editor == null) {
				editor = new AngleCellEditor(element.viewer.getTree(), SWT.SINGLE | SWT.BORDER);
			}
			return editor;
		}
		return null;
	}

	public Object getCellEditorValue(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			return (valueIndex == 0) ? String.valueOf((int)(valueWithoutExpr/360))
									 : formatValue(valueWithoutExpr%360, editorPrecision);
		}
		return null;
	}

	public void setCellEditorValue(int columnIndex, Object value) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			try {
				double newValue;
				if (valueIndex == 0) {
					if (value.equals(String.valueOf((int)(valueWithoutExpr/360)))) {
						return;
					}
					newValue = Double.valueOf((String)value)*360 + (valueWithoutExpr%360);
	
				} else /*if (valueIndex == 1)*/ {
					if (value.equals(formatValue(valueWithoutExpr%360, editorPrecision))) {
						return;
					}
					newValue = (int)(valueWithoutExpr/360)*360 + Double.valueOf((String)value);
				}
				modifyValue(newValue);
			} catch (NumberFormatException e) {
				element.viewer.getTree().getDisplay().beep();
			}
		}
	}

	private class AngleCellEditor extends TextCellEditor {

		private boolean activated;


		protected AngleCellEditor(Composite parent, int style) {
			super(parent, style);
		}

		protected Control createControl(final Composite parent) {
			final Composite composite = new Composite(parent, SWT.NO_FOCUS | (COCOA ? SWT.NO_BACKGROUND : 0));
			composite.setFont(parent.getFont());
			composite.setBackground(parent.getBackground());
			composite.setLayout(new FormLayout());

			super.createControl(composite);
			text.setBackground(text.getDisplay().getSystemColor(SWT.COLOR_WHITE));

			Rectangle valueBounds = valueArea.get(valueIndex);
			int left = valueBounds.x - valueArea.get(0).x + (WIN32 ? 3 : 0);
			int right = left + Math.max(60, valueBounds.width + 30);

			FormData data = new FormData();
			data.top = new FormAttachment(0, 0);
			data.bottom = new FormAttachment(100, 0);
			data.left = new FormAttachment(0, left);
			data.right = new FormAttachment(0, right);
			text.setLayoutData(data);

			composite.addMouseListener(new MouseAdapter() {
				public void mouseDown(MouseEvent e) {
					parent.forceFocus();

					if (e.button == 1) {
						for (int i = 0, n = valueArea.size(); i < n; ++i) {
							Rectangle r = valueArea.get(i);
							if (r.contains(e.x, e.y)) {
								valueIndex = (i < valueIndex) ? i : i+1;
								parent.update();
								openInPlaceEditor();
								break;
							}
						}
					}
				}
			});

			composite.addPaintListener(new PaintListener() {
				public void paintControl(PaintEvent e) {
					GC gc = e.gc;
					int x = WIN32 ? 5 : 2;
					int height = composite.getBounds().height;

					x = drawValue(gc, x, 0, height, 0, valueWithoutExpr);
					x += 5;
					x = drawValue(gc, x, 0, height, 1, valueWithoutExpr);
				}
			});

			return composite;
		}

		private Rectangle getTextBounds() {
			return text.getBounds();
		}

		public void activate() {
			super.activate();
			activated = true;
		}

		public void deactivate() {
			super.deactivate();
			if (activated) {
				editor = null;
				dispose();
			}
		}

	}

}
