/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.AnimatableInteger;

public class AnimatableIntegerElementDelegate extends AnimatableValueElementDelegate<Integer> {

	protected final int increment;

	protected final String unit;

	protected IntegerCellEditor editor;


	public AnimatableIntegerElementDelegate(
			AnimatableValueElement element, String name, AnimatableInteger avalue) {

		this(element, name, avalue, 1, null);
	}

	public AnimatableIntegerElementDelegate(
			AnimatableValueElement element, String name, AnimatableInteger avalue, String unit) {

		this(element, name, avalue, 1, unit);
	}

	public AnimatableIntegerElementDelegate(
			AnimatableValueElement element, String name, AnimatableInteger avalue, int increment, String unit) {

		super(element, name, avalue);
		this.increment = increment;
		this.unit = unit;
	}

	protected void drawValue(GC gc, int x, int y, int height, boolean focused) {
		if (editor == null) {
			drawValue(gc, x, y, height, focused, valueToDraw, unit);
		}
	}

	protected boolean canDragGestureEdit() {
		return true;
	}

	protected Integer dragGesture(double dx, double dy) {
		return originalValue + (int)(increment * dx);
	}

	protected boolean isInPlaceEditorActive() {
		return (editor != null);
	}

	public boolean canEdit(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			return canInPlaceEdit;
		}
		return false;
	}

	public CellEditor getCellEditor(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			if (editor == null) {
				editor = new IntegerCellEditor(element.viewer.getTree(), SWT.SINGLE | SWT.BORDER);
			}
			return editor;
		}
		return null;
	}

	public Object getCellEditorValue(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			return valueWithoutExpr.toString();
		}
		return null;
	}

	public void setCellEditorValue(int columnIndex, Object value) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			try {
				modifyValue(Integer.valueOf((String) value));
			} catch (NumberFormatException e) {
				element.viewer.getTree().getDisplay().beep();
			}
		}
	}


	// TODO DoubleCellEditor と中身が一緒なのでまとめる
	private class IntegerCellEditor extends TextCellEditor {

		private boolean activated;


		private IntegerCellEditor(Composite parent, int style) {
			super(parent, style);
		}

		protected Control createControl(final Composite parent) {
			final Composite composite = new Composite(parent, SWT.NO_FOCUS | (COCOA ? SWT.NO_BACKGROUND : 0));
			composite.setFont(parent.getFont());
			composite.setBackground(parent.getBackground());
			composite.setLayout(new FormLayout());

			super.createControl(composite);
			text.setBackground(text.getDisplay().getSystemColor(SWT.COLOR_WHITE));

			int right = 0;
			if (unit != null) {
				GC gc = new GC(composite);
				right = gc.textExtent(unit, SWT.DRAW_TRANSPARENT).x + 10;
				gc.dispose();
			}

			FormData data = new FormData();
			data.top = new FormAttachment(0, 0);
			data.bottom = new FormAttachment(100, 0);
			data.left = new FormAttachment(0, WIN32 ? 3 : 0);
			data.right = new FormAttachment(100, -right);
			text.setLayoutData(data);

			composite.addMouseListener(new MouseAdapter() {
				public void mouseDown(MouseEvent e) {
					parent.forceFocus();
				}
			});

			if (unit != null) {
				composite.addPaintListener(new PaintListener() {
					public void paintControl(PaintEvent e) {
						Rectangle bounds = text.getBounds();
						int height = composite.getBounds().height;

						setValueForeground(e.gc, false);
						drawString(e.gc, bounds.width + 5, 0, height, unit);
					}
				});
			}

			return composite;
		}

		public void activate() {
			super.activate();
			activated = true;
		}

		public void deactivate() {
			super.deactivate();
			if (activated) {
				editor = null;
				dispose();
			}
		}

	}

}
