/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import org.eclipse.core.commands.operations.IOperationHistoryListener;
import org.eclipse.core.commands.operations.IUndoableOperation;
import org.eclipse.core.commands.operations.OperationHistoryEvent;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuCreator;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.ToolItem;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PlatformUI;

import ch.kuramo.javie.app.ImageUtil;
import ch.kuramo.javie.app.project.LayerCompSuperSamplingOperation;
import ch.kuramo.javie.app.project.ProjectManager;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.SuperSampling;

public class LayerCompSuperSamplingAction extends Action implements IMenuCreator {

	private final ProjectManager pm;

	private final LayerComposition comp;

	private Menu menu;


	public LayerCompSuperSamplingAction(final LayerCompositionView view) {
		super("スーパーサンプリング");
		pm = ProjectManager.forWorkbenchWindow(view.getSite().getWorkbenchWindow());
		comp = view.getLayerComposition();

		setImageDescriptor(ImageUtil.getCompositionSuperSamplingIcon(comp.getSuperSampling()));
		setMenuCreator(this);


		final IOperationHistoryListener historyListener = new IOperationHistoryListener() {
			public void historyNotification(OperationHistoryEvent event) {
				switch (event.getEventType()) {
					case OperationHistoryEvent.DONE:
					case OperationHistoryEvent.REDONE:
					case OperationHistoryEvent.UNDONE:
						IUndoableOperation op = event.getOperation();
						if (op instanceof LayerCompSuperSamplingOperation && op.hasContext(pm.getUndoContext())) {
							setImageDescriptor(ImageUtil.getCompositionSuperSamplingIcon(comp.getSuperSampling()));
						}
						break;
				}
			}
		};

		view.getSite().getPage().addPartListener(new IPartListener() {
			public void partActivated(IWorkbenchPart part) { }
			public void partBroughtToTop(IWorkbenchPart part) { }
			public void partDeactivated(IWorkbenchPart part) { }
			public void partOpened(IWorkbenchPart part) { }

			public void partClosed(IWorkbenchPart part) {
				if (part == view) {
					PlatformUI.getWorkbench().getOperationSupport()
							.getOperationHistory().removeOperationHistoryListener(historyListener);
				}
			}
		});

		PlatformUI.getWorkbench().getOperationSupport()
				.getOperationHistory().addOperationHistoryListener(historyListener);
	}

	public void dispose() {
		disposeMenu();
	}

	private void disposeMenu() {
		if (menu != null) {
			menu.dispose();
			menu = null;
		}
	}

	public Menu getMenu(Control parent) {
		disposeMenu();

		if (!isEnabled()) {
			return null;
		}

		menu = new Menu(parent);


		for (SuperSampling ss : SuperSampling.values()) {
			addAction(menu, ss);
		}

		return menu;
	}

	public Menu getMenu(Menu parent) {
		return null;
	}

	private void addAction(Menu parent, final SuperSampling ss) {
		String text;
		switch (ss) {
			case SS_NONE:
				text = "スーパーサンプリング：なし";
				break;
			default:
				text = ss.name().substring(3).toLowerCase();
				break;
		}

		Action action = new Action(text, IAction.AS_RADIO_BUTTON) {
			public void run() {
				pm.postOperation(new LayerCompSuperSamplingOperation(pm, comp, ss));
			}
		};

		action.setChecked(ss == comp.getSuperSampling());

		ActionContributionItem item = new ActionContributionItem(action);
		item.fill(parent, -1);
	}

	public void runWithEvent(Event event) {
		if (event.widget instanceof ToolItem) {
			ToolItem toolItem = (ToolItem) event.widget;
			Control control = toolItem.getParent();
			Menu menu = getMenu(control);

			Rectangle bounds = toolItem.getBounds();
			Point topLeft = new Point(bounds.x, bounds.y + bounds.height);
			menu.setLocation(control.toDisplay(topLeft));
			menu.setVisible(true);
		}
	}

}
