/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.views.layercomp;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;

import ch.kuramo.javie.api.Vec3d;
import ch.kuramo.javie.app.Activator;
import ch.kuramo.javie.app.views.LayerCompositionView;
import ch.kuramo.javie.core.AnimatableVec3d;

public class ScaleElementDelegate extends AnimatableVec3dElementDelegate {

	private static final Image linkImage = Activator.getDefault().getImageRegistry().get(Activator.IMG_SWITCH_LINK);


	private final SwitchGroup linkSwitch = new SwitchGroup();

	private boolean linked = true;


	public ScaleElementDelegate(
			AnimatableValueElement element, String name, AnimatableVec3d avalue, boolean threeD, String unit) {

		super(element, name, avalue, threeD, unit);

		linkSwitch.add(new Switch() {
			public void mouseDown(MouseEvent event) {
				linked = !linked;
				linkSwitch.redraw((Control) event.widget);
			}

			public boolean hasBorder() {
				return true;
			}

			public Image getImage() {
				return linked ? linkImage : null;
			}
		});
	}

	@Override
	public void paintColumn(Event event) {
		if (event.index == LayerCompositionView.VALUE_COL) {
			linkSwitch.paint(event);
		}
		super.paintColumn(event);
	}

	@Override
	protected void drawValue(GC gc, int x, int y, int height, boolean focused) {
		super.drawValue(gc, x+18, y, height, focused);
	}

	@Override
	protected int drawValue(GC gc, int x, int y, int height, int valueIndex, double value) {
		return super.drawValue(gc, (valueIndex == 0) ? x+18 : x, y, height, valueIndex, value);
	}

	@Override
	public void mouseDown(MouseEvent event, int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			linkSwitch.mouseDown(event);
		}
		super.mouseDown(event, columnIndex);
	}

	@Override
	public CellEditor getCellEditor(int columnIndex) {
		if (columnIndex == LayerCompositionView.VALUE_COL) {
			if (editor == null) {
				editor = new ScaleCellEditor(element.viewer.getTree(), SWT.SINGLE | SWT.BORDER);
			}
			return editor;
		}
		return null;
	}

	@Override
	protected void modifyValue(Vec3d newValue, String relation) {
		if (linked) {
			double[] oldArray = new double[] { valueWithoutExpr.x, valueWithoutExpr.y, valueWithoutExpr.z };
			double[] newArray = new double[] { newValue.x, newValue.y, newValue.z };
			double ratio = newArray[valueIndex] / oldArray[valueIndex];
			for (int i = 0; i < 3; ++i) {
				if (i != valueIndex) {
					if (oldArray[i] == oldArray[valueIndex]) {
						newArray[i] = newArray[valueIndex];
					} else if (!Double.isNaN(ratio) && !Double.isInfinite(ratio)) {
						newArray[i] *= ratio;
					}
				}
			}
			newValue = new Vec3d(newArray[0], newArray[1], newArray[2]);
		}
		super.modifyValue(newValue, relation);
	}

	protected class ScaleCellEditor extends Vec3dCellEditor {

		protected ScaleCellEditor(Composite parent, int style) {
			super(parent, style);
		}

		@Override
		protected Control createControl(Composite parent) {
			Control control = super.createControl(parent);

			FormData data = (FormData) text.getLayoutData();
			data.left.offset += 18;
			data.right.offset += 18;

			if (WIN32) {
				control.addPaintListener(new PaintListener() {
					public void paintControl(PaintEvent e) {
						Event event = new Event();
						event.gc = e.gc;
						event.widget = e.widget;
						event.x = e.x;
						event.y = e.y;
						linkSwitch.paint(event);
					}
				});
			}

			return control;
		}

	}

}
