/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.widgets;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;

public class GridBuilder {

	private final Composite _composite;

	private final List<Control> _tabList = new ArrayList<Control>();


	public GridBuilder(Composite parent, int numColumns, boolean makeColumnsEqualWidth) {
		_composite = new Composite(parent, SWT.NULL);
		_composite.setLayout(new GridLayout(numColumns, makeColumnsEqualWidth));
	}

	public Composite getComposite() {
		return _composite;
	}

	public Control[] getTabList() {
		return _tabList.toArray(new Control[_tabList.size()]);
	}

	public void updateTabList() {
		updateTabList(false);
	}

	public void updateTabList(boolean root) {
		_composite.setTabList(getTabList());
		if (root) {
			setTabRoot();
		}
	}

	public void setTabRoot() {
		Control[] tabList = _composite.getParent().getTabList();
		for (int i = 0; i < tabList.length; ++i) {
			if (tabList[i] == _composite) {
				Control[] newList = new Control[tabList.length-1];
				System.arraycopy(tabList, 0, newList, 0, i);
				System.arraycopy(tabList, i+1, newList, i, newList.length-i);
				_composite.getParent().setTabList(newList);
				break;
			}
		}
	}

	public GridLayout getLayout() {
		return (GridLayout) _composite.getLayout();
	}

	public CellBuilder span(int hSpan, int vSpan) {
		return new CellBuilder().span(hSpan, vSpan);
	}

	public CellBuilder hSpan(int hSpan) {
		return new CellBuilder().hSpan(hSpan);
	}

	public CellBuilder vSpan(int vSpan) {
		return new CellBuilder().vSpan(vSpan);
	}

	public CellBuilder align(int hAlign, int vAlign) {
		return new CellBuilder().align(hAlign, vAlign);
	}

	public CellBuilder hAlign(int hAlign) {
		return new CellBuilder().hAlign(hAlign);
	}

	public CellBuilder vAlign(int vAlign) {
		return new CellBuilder().vAlign(vAlign);
	}

	public CellBuilder grab(boolean hGrab, boolean vGrab) {
		return new CellBuilder().grab(hGrab, vGrab);
	}

	public CellBuilder hGrab() {
		return new CellBuilder().hGrab();
	}

	public CellBuilder vGrab() {
		return new CellBuilder().vGrab();
	}

	public CellBuilder size(int width, int height) {
		return new CellBuilder().size(width, height);
	}

	public CellBuilder width(int width) {
		return new CellBuilder().width(width);
	}

	public CellBuilder height(int height) {
		return new CellBuilder().height(height);
	}

	public class CellBuilder {
		private final GridData data = new GridData();
		private Control tabBefore = null;
		private Control tabAfter = null;

		public CellBuilder span(int hSpan, int vSpan) {
			data.horizontalSpan = hSpan;
			data.verticalSpan = vSpan;
			return this;
		}

		public CellBuilder hSpan(int hSpan) {
			data.horizontalSpan = hSpan;
			return this;
		}

		public CellBuilder vSpan(int vSpan) {
			data.verticalSpan = vSpan;
			return this;
		}

		public CellBuilder align(int hAlign, int vAlign) {
			data.horizontalAlignment = hAlign;
			data.verticalAlignment = vAlign;
			return this;
		}

		public CellBuilder hAlign(int hAlign) {
			data.horizontalAlignment = hAlign;
			return this;
		}

		public CellBuilder vAlign(int vAlign) {
			data.verticalAlignment = vAlign;
			return this;
		}

		public CellBuilder grab(boolean hGrab, boolean vGrab) {
			data.grabExcessHorizontalSpace = hGrab;
			data.grabExcessVerticalSpace = vGrab;
			return this;
		}

		public CellBuilder hGrab() {
			data.grabExcessHorizontalSpace = true;
			return this;
		}

		public CellBuilder vGrab() {
			data.grabExcessVerticalSpace = true;
			return this;
		}

		public CellBuilder size(int width, int height) {
			data.widthHint = width;
			data.heightHint = height;
			return this;
		}

		public CellBuilder width(int width) {
			data.widthHint = width;
			return this;
		}

		public CellBuilder height(int height) {
			data.heightHint = height;
			return this;
		}

		public CellBuilder tabBefore(Control control) {
			tabBefore = control;
			return this;
		}

		public CellBuilder tabAfter(Control control) {
			tabAfter = control;
			return this;
		}

		private void addToTabList(Control control) {
			if (tabBefore != null) {
				_tabList.add(_tabList.indexOf(tabBefore), control);
			} else if (tabAfter != null) {
				_tabList.add(_tabList.indexOf(tabAfter) + 1, control);
			} else {
				_tabList.add(control);
			}
		}

		public Composite composite(int style) {
			Composite c = new Composite(_composite, style);
			c.setLayoutData(data);
			addToTabList(c);
			return c;
		}

		public Label label(int style, String text) {
			Label label = new Label(_composite, style);
			label.setText(text);
			label.setLayoutData(data);
			return label;
		}

		public Text text(int style, String text) {
			Text tx = new Text(_composite, style);
			tx.setText(text);
			tx.setLayoutData(data);
			addToTabList(tx);
			return tx;
		}

		public Button button(int style, String text) {
			Button button = new Button(_composite, style);
			button.setText(text);
			button.setLayoutData(data);
			addToTabList(button);
			return button;
		}

		public Combo combo(int style, String[] items) {
			return combo(style, items, items[0]);
		}

		public Combo combo(int style, String[] items, int defaultItem) {
			return combo(style, items, items[defaultItem]);
		}

		public Combo combo(int style, String[] items, String text) {
			return combo(style, -1, items, text);
		}

		public Combo combo(int style, int visibleItemCount, String[] items, String text) {
			Combo combo = new Combo(_composite, style);
			if (visibleItemCount >= 0) {
				combo.setVisibleItemCount((visibleItemCount > 0) ? visibleItemCount : items.length);
			}
			combo.setItems(items);
			combo.setText(text);
			combo.setLayoutData(data);
			addToTabList(combo);
			return combo;
		}

		public Group group(int style, String text) {
			Group group = new Group(_composite, style);
			if (text != null) {
				group.setText(text);
			}
			group.setLayoutData(data);
			addToTabList(group);
			return group;
		}

		public Tree tree(int style) {
			Tree tree = new Tree(_composite, style);
			tree.setLayoutData(data);
			addToTabList(tree);
			return tree;
		}

	}

}
