/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.widgets;

import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.events.MouseTrackAdapter;
import org.eclipse.swt.events.ShellAdapter;
import org.eclipse.swt.events.ShellEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Monitor;
import org.eclipse.swt.widgets.Shell;

import ch.kuramo.javie.core.Util;

public class InPlaceEditorShell {

	private static final boolean COCOA = SWT.getPlatform().equals("cocoa");

	public static Shell create(
			Control parent, Rectangle locationHint,
			int width, int height, boolean resizable) {

		Display display = parent.getDisplay();

		List<Rectangle> areaList = Util.newList();
		for (Monitor monitor : display.getMonitors()) {
			areaList.add(monitor.getClientArea());
		}
		for (Monitor monitor : display.getMonitors()) {
			areaList.add(monitor.getBounds());
		}

		locationHint = new Rectangle(locationHint.x-4, locationHint.y-2, locationHint.width+8, locationHint.height+4);
		Point location = parent.toDisplay(locationHint.x, locationHint.y+locationHint.height);

		Point cursorPt = display.getCursorLocation();
		for (Rectangle area : areaList) {
			if (area.contains(cursorPt)) {
				width = Math.min(width, area.width);
				height = Math.min(height, area.height);

				if (location.y+height > area.y+area.height) {
					if (location.y-locationHint.height-height >= area.y) {
						location.y -= locationHint.height+height;
					} else {
						location.y = area.y+area.height-height;
						if (location.x+locationHint.width+width <= area.x+area.width) {
							location.x += locationHint.width;
						} else if (location.x-width >= area.x) {
							location.x -= width;
						}
					}
				}

				location.x = Math.min(location.x, area.x+area.width-width);
				location.x = Math.max(location.x, area.x);
				break;
			}
		}

		final Shell shell = new Shell(parent.getShell(), SWT.TOOL);
		shell.setSize(width, height);
		shell.setLocation(location);


		shell.addShellListener(new ShellAdapter() {
			public void shellDeactivated(ShellEvent e) {
				shell.setVisible(false);
			}
		});

		shell.addListener(SWT.Hide, new Listener() {
			public void handleEvent(Event e) {
				e.display.asyncExec(new Runnable() {
					public void run() {
						if (!shell.isDisposed()) {
							shell.dispose();
						}
					}
				});
			}
		});

		final Listener mouseDownFilter = new Listener() {
			public void handleEvent(Event e) {
				if (!shell.isDisposed()) {
					Control control = (e.widget instanceof Control) ? (Control) e.widget : null;
					if (control == null || control.getShell() != shell) {
						e.type = SWT.None;
					}
				}
			}
		};
		display.addFilter(SWT.MouseDown, mouseDownFilter);

		shell.addDisposeListener(new DisposeListener() {
			public void widgetDisposed(DisposeEvent e) {
				e.display.removeFilter(SWT.MouseDown, mouseDownFilter);
			}
		});


		if (resizable) {
			shell.addMouseMoveListener(new MouseMoveListener() {
				public void mouseMove(MouseEvent e0) {
					Point size = shell.getSize();
					final boolean right = (e0.x >= size.x-10);
					//final boolean left = (!right && e0.x <= 10);
					final boolean bottom = (e0.y >= size.y-10);
					//final boolean top = (!bottom && e0.y <= 10);

					int cursorId =	(!COCOA && ((right && bottom) /*|| (left && top)*/)) ? SWT.CURSOR_SIZENWSE :
									//(!COCOA && ((right && top) || (left && bottom))) ? SWT.CURSOR_SIZENESW :
									(right /*|| left*/) ? SWT.CURSOR_SIZEWE :
									(bottom /*|| top*/) ? SWT.CURSOR_SIZENS :
									SWT.CURSOR_ARROW;
					shell.setCursor(e0.display.getSystemCursor(cursorId));
				}
			});
			shell.addMouseTrackListener(new MouseTrackAdapter() {
				public void mouseExit(MouseEvent e) {
					shell.setCursor(null);
				}
			});

			shell.addMouseListener(new MouseAdapter() {
				private MouseMoveListener moveListener;

				public void mouseDown(final MouseEvent e0) {
					if (moveListener != null) return;

					Point size = shell.getSize();
					final boolean right = (e0.x >= size.x-10);
					//final boolean left = (!right && e0.x <= 10);
					final boolean bottom = (e0.y >= size.y-10);
					//final boolean top = (!bottom && e0.y <= 10);

					if (/*left || top ||*/ right || bottom) {
						final Point size0 = shell.getSize();
						//final Point loc0 = shell.getLocation();
						moveListener = new MouseMoveListener() {
							public void mouseMove(MouseEvent e1) {
								int width = size0.x + (e1.x - e0.x) * (right ? 1 : /*left ? -1 :*/ 0);
								int height = size0.y + (e1.y - e0.y) * (bottom ? 1 : /*top ? -1 :*/ 0); 
								width = Math.max(width, 10);
								height = Math.max(height, 10);

								//int x = loc0.x + (left ? size0.x - width : 0);
								//int y = loc0.y + (top ? size0.y - height : 0);

								shell.setSize(width, height);
								//shell.setLocation(x, y);
							}
						};
						shell.addMouseMoveListener(moveListener);
					}
				}

				public void mouseUp(MouseEvent e) {
					if (moveListener != null) {
						shell.removeMouseMoveListener(moveListener);
						moveListener = null;
					}
				}
			});
		}


		return shell;
	}

}
