/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.effects;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.ColorDepthChanger", category=Categories.UTILITY)
public class ColorDepthChanger {

	@Property
	private IAnimatableEnum<ColorMode> colorDepth;


	private final IVideoEffectContext veContext;

	private final RenderContext renderContext;

	private final IVideoRenderSupport support;

	@Inject
	public ColorDepthChanger(
			IVideoEffectContext veContext, RenderContext renderContext, IVideoRenderSupport support) {

		this.veContext = veContext;
		this.renderContext = renderContext;
		this.support = support;
	}

	public IVideoBuffer doVideoEffect() {
		ColorMode curColorMode = veContext.getColorMode();
		ColorMode newColorMode = veContext.value(this.colorDepth);

		IVideoBuffer source = null;
		IVideoBuffer copy = null;
		try {
			renderContext.setColorMode(newColorMode);
			source = veContext.doPreviousEffect();

			IVideoBuffer result;
			if (source.getColorMode() != curColorMode) {
				copy = support.createVideoBuffer(source.getBounds(), curColorMode);
				support.copy(source, copy);
				result = copy;
				copy = null;
			} else {
				result = source;
				source = null;
			}
			return result;

		} finally {
			renderContext.setColorMode(curColorMode);
			if (source != null) source.dispose();
			if (copy != null) copy.dispose();
		}
	}

}
