/*
 * Copyright (c) 2010,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import javax.media.opengl.GL2;

import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.core.MediaInputPlaceholder;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;

class MediaInputPlaceholderImpl implements MediaInputPlaceholder {

	private final RenderContext _context;

	private final IVideoRenderSupport _support;

	private VideoBounds videoFrameBounds;

	@Inject
	MediaInputPlaceholderImpl(RenderContext context, IVideoRenderSupport support) {
		_context = context;
		_support = support;
	}

	public boolean isAudioAvailable() {
		return false;
	}

	public boolean isVideoAvailable() {
		return true;
	}

	public Time getDuration() {
		return null;
	}

	public Time getVideoFrameDuration() {
		return null;
	}

	public VideoBounds getVideoFrameBounds() {
		if (videoFrameBounds != null) {
			return videoFrameBounds;
		} else {
			// videoFrameBounds が null になるのは、オーディオのみのフッテージの場合と
			// プレースホルダを実装する前に作成したプロジェクトのフッテージ。
			// サイズを(0, 0)にするとプレビューに何も表示されなくなるので、適当なサイズを与えている。
			return new VideoBounds(200, 200);
		}
	}

	void setVideoFrameBounds(VideoBounds videoFrameBounds) {
		this.videoFrameBounds = videoFrameBounds;
	}

	public IAudioBuffer getAudioChunk(Time mediaTime) {
		throw new UnsupportedOperationException("audio is not available");
	}

	public IVideoBuffer getVideoFrame(Time mediaTime) {
		Resolution resolution = _context.getVideoResolution();
		final VideoBounds bounds = resolution.scale(getVideoFrameBounds());

		IVideoBuffer videoBuffer = null;
		try {
			videoBuffer = _support.createVideoBuffer(bounds);

			Runnable operation = new Runnable() {
				public void run() {
					float x0 = (float) bounds.x;
					float y0 = (float) bounds.y;
					float x1 = (float) (bounds.x+bounds.width);
					float y1 = (float) (bounds.y+bounds.height);

					_support.ortho2D(bounds);

					GL2 gl = _context.getGL().getGL2();
					gl.glClearColor(0.25f, 0.25f, 0.25f, 0.5f);
					gl.glClear(GL2.GL_COLOR_BUFFER_BIT);

					gl.glColor4f(1, 0, 0, 1);
					gl.glLineWidth(2);

					gl.glBegin(GL2.GL_LINE_LOOP);
					gl.glVertex2f(x0, y0);
					gl.glVertex2f(x1, y0);
					gl.glVertex2f(x1, y1);
					gl.glVertex2f(x0, y1);
					gl.glEnd();

					gl.glLineWidth(1);
					gl.glBegin(GL2.GL_LINES);
					gl.glVertex2f(x0, y0); gl.glVertex2f(x1, y1);
					gl.glVertex2f(x1, y0); gl.glVertex2f(x0, y1);
					gl.glEnd();
				}
			};

			int pushAttribs = GL2.GL_COLOR_BUFFER_BIT | GL2.GL_CURRENT_BIT | GL2.GL_LINE_BIT;
			_support.useFramebuffer(operation, pushAttribs, videoBuffer);

			IVideoBuffer result = videoBuffer;
			videoBuffer = null;
			return result;

		} finally {
			if (videoBuffer != null) videoBuffer.dispose();
		}
	}

}
