/*
 * Copyright (c) 2009-2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.util.Collections;
import java.util.List;
import java.util.ListIterator;

import net.arnx.jsonic.JSONHint;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.core.Composition;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.ExpressionScope;
import ch.kuramo.javie.core.FrameBlend;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaInput;
import ch.kuramo.javie.core.MediaItem;
import ch.kuramo.javie.core.MediaItemLayer;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.VideoLayerRenderer;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.VideoLayerRenderer.CRLayerRenderer;
import ch.kuramo.javie.core.VideoLayerRenderer.MatteLayerRenderers;
import ch.kuramo.javie.core.VideoLayerRenderer.NormalLayerRenderer;
import ch.kuramo.javie.core.annotations.ProjectElement;
import ch.kuramo.javie.core.services.RenderContext;
import ch.kuramo.javie.core.services.VideoEffectPipeline;
import ch.kuramo.javie.core.services.VideoRenderSupport;

import com.google.inject.Inject;

@ProjectElement("mediaItemLayer")
public class MediaItemLayerImpl extends AbstractMediaLayer implements MediaItemLayer {

	private String _itemId;

	private MediaItem _mediaItem;

	private boolean _videoEnabled;

	private boolean _audioEnabled;

	private boolean _ctcr;

	private FrameBlend _frameBlend = FrameBlend.NONE;

	@Inject
	private RenderContext _context;

	@Inject
	private VideoRenderSupport _vrSupport;

	@Inject
	private VideoEffectPipeline _vePipeline;


	@Override
	public void initialize() {
		throw new UnsupportedOperationException("Use initialize(MediaItem) method instead.");
	}

	public void initialize(MediaItem mediaItem) {
		MediaInput input = mediaItem.getMediaInput();
		boolean videoAvailable = input.isVideoAvailable();
		boolean audioAvailable = input.isAudioAvailable();

		if (!videoAvailable && !audioAvailable) {
			throw new IllegalArgumentException("No video nor audio is available.");
		}

		super.initialize();

		_itemId = mediaItem.getId();
		_mediaItem = mediaItem;
		_videoEnabled = videoAvailable;
		_audioEnabled = audioAvailable;

		setName(mediaItem.getName());
	}

	public String getItemId() {
		return _itemId;
	}

	public void setItemId(String itemId) {
		_itemId = itemId;
	}

	@JSONHint(ignore=true)
	public MediaItem getItem() {
		return _mediaItem;
	}

	@JSONHint(ignore=true)
	public void setItem(MediaItem item) {
		_itemId = item.getId();
		_mediaItem = item;
	}

	public boolean isVideoEnabled() {
		return _videoEnabled;
	}

	public void setVideoEnabled(boolean enabled) {
		_videoEnabled = enabled;
	}

	public boolean isAudioEnabled() {
		return _audioEnabled;
	}

	public void setAudioEnabled(boolean enabled) {
		_audioEnabled = enabled;
	}

	public boolean isCTCR() {
		return _ctcr;
	}

	public void setCTCR(boolean ctcr) {
		_ctcr = ctcr;
	}

	public FrameBlend getFrameBlend() {
		return _frameBlend;
	}

	public void setFrameBlend(FrameBlend frameBlend) {
		_frameBlend = frameBlend;
	}

	@JSONHint(ignore=true)
	public MediaInput getMediaInput() {
		return _mediaItem.getMediaInput();
	}

	@JSONHint(ignore=true)
	public boolean isPrecompositionLayer() {
		return (getPrecomposition() != null);
	}

	private LayerComposition getPrecomposition() {
		if (_mediaItem instanceof CompositionItem) {
			Composition comp = ((CompositionItem) _mediaItem).getComposition();
			if (comp instanceof LayerComposition) {
				return (LayerComposition) comp;
			}
		}
		return null;
	}

	@Override
	public void afterDecode(Project p, LayerComposition c) throws ProjectDecodeException {
		super.afterDecode(p, c);

		_mediaItem = p.getItem(_itemId);
		if (_mediaItem == null) {
			throw new ProjectDecodeException(
					"no such MediaItem found: id=" + _itemId);
		}
	}

	@Override
	public void prepareExpression(ExpressionScope scope) {
		super.prepareExpression(scope);

		LayerComposition preComp = getPrecomposition();
		if (preComp != null) {
			ExpressionScope preCompScope = scope.createPrecompositionScope(preComp);
			if (preCompScope != null) {
				preComp.prepareExpression(preCompScope);
			}
		}
	}

	public List<VideoLayerRenderer> createCollapsedRenderers() {

		if (LayerNature.isCTCR(this)) {
			LayerComposition preComp = getPrecomposition();
			if (preComp != null) {
				return createCollapsedRenderers(preComp);
			}
		}

		throw new IllegalStateException("not collapsed");
	}

	private List<VideoLayerRenderer> createCollapsedRenderers(final LayerComposition preComp) {
		List<VideoLayerRenderer> renderers = _context.saveAndExecute(new WrappedOperation<List<VideoLayerRenderer>>() {
			public List<VideoLayerRenderer> execute() {
				return preComp.createCollapsedRenderers();
			}
		});

		for (ListIterator<VideoLayerRenderer> it = renderers.listIterator(); it.hasNext(); ) {
			VideoLayerRenderer r = it.next();

			if (r instanceof NormalLayerRenderer) {
				it.set(wrapNormalLayerRenderer(preComp, (NormalLayerRenderer) r));
			} else if (r instanceof CRLayerRenderer) {
				it.set(wrapCRLayerRenderer(preComp, (CRLayerRenderer) r));
			} else if (r instanceof MatteLayerRenderers) {
				wrapMatteLayerRenderers(preComp, (MatteLayerRenderers) r);
			} else {
				throw new RuntimeException("unsupported VideoLayerRenderer: " + r.getClass().getName());
			}
		}

		return renderers;
	}

	private NormalLayerRenderer wrapNormalLayerRenderer(final LayerComposition preComp, final NormalLayerRenderer r) {
		return new NormalLayerRenderer() {

			public MediaLayer getLayer() {
				return r.getLayer();
			}

			public void addMatteId(String matteId) {
				r.addMatteId(matteId);
			}

			public List<String> getMatteIds() {
				return r.getMatteIds();
			}

			public double getOpacity() {
				double opacity = _context.saveAndExecute(new WrappedOperation<Double>() {
					public Double execute() {
						_context.setTime(calcVideoMediaTime(getMediaInput()));
						_context.setComposition(preComp);
						return r.getOpacity();
					}
				});

				return opacity * MediaItemLayerImpl.this.getOpacity().value(_context) / 100;
			}

			public void multModelViewMatrix(final double[] mvMatrix) {
				_vrSupport.setMatrix(null, mvMatrix);
				LayerMatrixUtil.multModelViewMatrix(MediaItemLayerImpl.this, _context, _vrSupport);
				_vrSupport.getMatrix(null, mvMatrix);

				_context.saveAndExecute(new WrappedOperation<Object>() {
					public Object execute() {
						_context.setTime(calcVideoMediaTime(getMediaInput()));
						_context.setComposition(preComp);
						r.multModelViewMatrix(mvMatrix);
						return null;
					}
				});
			}

			public VideoBounds calcBounds(boolean withEffects) {
				return _vePipeline.getVideoBounds(
						MediaItemLayerImpl.this, getEffects(withEffects),
						createInputBoundsOperation(withEffects));
			}

			public IVideoBuffer render(boolean withEffects, boolean frameBlendEnabled) {
				return _vePipeline.doVideoEffects(
						MediaItemLayerImpl.this, getEffects(withEffects),
						createInputBoundsOperation(withEffects),
						createInputBufferOperation(withEffects, frameBlendEnabled));
			}

			private List<Effect> getEffects(boolean withEffects) {
				return (withEffects && isEffectsEnabled())
						? MediaItemLayerImpl.this.getEffects() : Collections.<Effect>emptyList();
			}

			private WrappedOperation<VideoBounds> createInputBoundsOperation(final boolean withEffects) {
				return new WrappedOperation<VideoBounds>() {
					public VideoBounds execute() {
						_context.setTime(calcVideoMediaTime(getMediaInput()));
						_context.setComposition(preComp);
						return r.calcBounds(withEffects);
					}
				};
			}

			private WrappedOperation<IVideoBuffer> createInputBufferOperation(final boolean withEffects, final boolean frameBlendEnabled) {
				return new WrappedOperation<IVideoBuffer>() {
					public IVideoBuffer execute() {
						_context.setTime(calcVideoMediaTime(getMediaInput()));
						_context.setComposition(preComp);
						return r.render(withEffects, frameBlendEnabled);
					}
				};
			}
		};
	}

	private CRLayerRenderer wrapCRLayerRenderer(final LayerComposition preComp, final CRLayerRenderer r) {
		return new CRLayerRenderer() {

			public MediaLayer getLayer() {
				return r.getLayer();
			}

			public void addMatteId(String matteId) {
				r.addMatteId(matteId);
			}

			public List<String> getMatteIds() {
				return r.getMatteIds();
			}

			public double getOpacity() {
				double opacity = _context.saveAndExecute(new WrappedOperation<Double>() {
					public Double execute() {
						_context.setTime(calcVideoMediaTime(getMediaInput()));
						_context.setComposition(preComp);
						return r.getOpacity();
					}
				});

				return opacity * MediaItemLayerImpl.this.getOpacity().value(_context) / 100;
			}

			public void multModelViewMatrix(final double[] mvMatrix) {
				_vrSupport.setMatrix(null, mvMatrix);
				LayerMatrixUtil.multModelViewMatrix(MediaItemLayerImpl.this, _context, _vrSupport);
				_vrSupport.getMatrix(null, mvMatrix);

				_context.saveAndExecute(new WrappedOperation<Object>() {
					public Object execute() {
						_context.setTime(calcVideoMediaTime(getMediaInput()));
						_context.setComposition(preComp);
						r.multModelViewMatrix(mvMatrix);
						return null;
					}
				});
			}

			public VideoBounds calcBounds(boolean withEffects, VideoBounds viewport) {
				return _vePipeline.getVideoBounds(
						MediaItemLayerImpl.this, getEffects(withEffects),
						createInputBoundsOperation(withEffects, viewport));
			}

			public IVideoBuffer render(boolean withEffects, VideoBounds viewport, double[] prjMatrix, double[] mvMatrix) {
				return _vePipeline.doVideoEffects(
						MediaItemLayerImpl.this, getEffects(withEffects),
						createInputBoundsOperation(withEffects, viewport),
						createInputBufferOperation(withEffects, viewport, prjMatrix, mvMatrix));
			}

			public IVideoBuffer render() {
				return _context.saveAndExecute(new WrappedOperation<IVideoBuffer>() {
					public IVideoBuffer execute() {
						_context.setTime(calcVideoMediaTime(getMediaInput()));
						_context.setComposition(preComp);
						return r.render();
					}
				});
			}

			private List<Effect> getEffects(boolean withEffects) {
				return (withEffects && isEffectsEnabled())
						? MediaItemLayerImpl.this.getEffects() : Collections.<Effect>emptyList();
			}

			private WrappedOperation<VideoBounds> createInputBoundsOperation(final boolean withEffects, final VideoBounds viewport) {
				return new WrappedOperation<VideoBounds>() {
					public VideoBounds execute() {
						_context.setTime(calcVideoMediaTime(getMediaInput()));
						_context.setComposition(preComp);
						return r.calcBounds(withEffects, viewport);
					}
				};
			}

			private WrappedOperation<IVideoBuffer> createInputBufferOperation(
					final boolean withEffects, final VideoBounds viewport, final double[] prjMatrix, final double[] mvMatrix) {

				return new WrappedOperation<IVideoBuffer>() {
					public IVideoBuffer execute() {
						_context.setTime(calcVideoMediaTime(getMediaInput()));
						_context.setComposition(preComp);
						return r.render(withEffects, viewport, prjMatrix, mvMatrix);
					}
				};
			}
		};
	}

	private void wrapMatteLayerRenderers(LayerComposition preComp, MatteLayerRenderers mlr) {
		for (ListIterator<VideoLayerRenderer> it = mlr.getRenderers().listIterator(); it.hasNext(); ) {
			VideoLayerRenderer r = it.next();

			if (r instanceof NormalLayerRenderer) {
				it.set(wrapNormalLayerRenderer(preComp, (NormalLayerRenderer) r));
			} else if (r instanceof CRLayerRenderer) {
				it.set(wrapCRLayerRenderer(preComp, (CRLayerRenderer) r));
			} else if (r instanceof MatteLayerRenderers) {
				wrapMatteLayerRenderers(preComp, (MatteLayerRenderers) r);
			} else {
				throw new RuntimeException("unsupported VideoLayerRenderer: " + r.getClass().getName());
			}
		}
	}

}
