/*
 * Copyright (c) 2010,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal.services;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.BlendMode;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.services.IBlendSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.core.Util;

import com.google.inject.Inject;

public class BlendSupportImpl implements IBlendSupport {

	private static final String BLEND_FUNCTIONS = "ch.kuramo.javie.core.shaders.BlendModeShaders.blend_functions";


	public static class Premult {

		private static String[] createSource(String name, boolean dissolve) {
			return new String[] {
									"uniform sampler2D texDst;",
									"uniform sampler2D texSrc;",
									"uniform float opacity;",
					dissolve ?		"uniform float dissolveSeed;" : "",
									"",
					String.format(	"vec4 blend_%s(vec4 pDst, vec4 pSrc, float opacity%s);", name, dissolve ? ", float dissolveSeed" : ""),
									"",
									"void main(void)",
									"{",
									"	vec4 dst = texture2D(texDst, gl_TexCoord[0].st);",
									"	vec4 src = texture2D(texSrc, gl_TexCoord[1].st);",
					String.format(	"	gl_FragColor = blend_%s(dst, src, opacity%s);", name, dissolve ? ", dissolveSeed" : ""),
									"}"
			};
		}

		private static String[] createSource(String name) {
			return createSource(name, false);
		}

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] NONE = createSource("none");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] NORMAL = createSource("normal");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DISSOLVE = createSource("dissolve", true);

//		@ShaderSource(attach=BLEND_FUNCTIONS)
//		public static final String[] DANCING_DISSOLVE = DISSOLVE;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DARKEN = createSource("darken");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] MULTIPLY = createSource("multiply");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR_BURN = createSource("color_burn");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_BURN = createSource("linear_burn");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DARKER_COLOR = createSource("darker_color");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] ADD = createSource("add");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LIGHTEN = createSource("lighten");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SCREEN = createSource("screen");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR_DODGE = createSource("color_dodge");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_DODGE = createSource("linear_dodge");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LIGHTER_COLOR = createSource("lighter_color");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] OVERLAY = createSource("overlay");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SOFT_LIGHT = createSource("soft_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HARD_LIGHT = createSource("hard_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_LIGHT = createSource("linear_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] VIVID_LIGHT = createSource("vivid_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] PIN_LIGHT = createSource("pin_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HARD_MIX = createSource("hard_mix");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DIFFERENCE = createSource("difference");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] EXCLUSION = createSource("exclusion");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HUE = createSource("hue");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SATURATION = createSource("saturation");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR = createSource("color");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LUMINOSITY = createSource("luminosity");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] STENCIL_ALPHA = createSource("stencil_alpha");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] STENCIL_LUMA = createSource("stencil_luma");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SILHOUETTE_ALPHA = createSource("silhouette_alpha");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SILHOUETTE_LUMA = createSource("silhouette_luma");

	}


	public static class UnmultSrc {

		private static String[] createSource(String name, boolean dissolve) {
			return new String[] {
									"uniform sampler2D texDst;",
									"uniform sampler2D texSrc;",
									"uniform float opacity;",
					dissolve ?		"uniform float dissolveSeed;" : "",
									"",
					String.format(	"vec4 blend_%s(vec4 pDst, vec4 pSrc, float opacity%s);", name, dissolve ? ", float dissolveSeed" : ""),
									"",
									"void main(void)",
									"{",
									"	vec4 dst = texture2D(texDst, gl_TexCoord[0].st);",
									"	vec4 src = texture2D(texSrc, gl_TexCoord[1].st);",
					String.format(	"	gl_FragColor = blend_%s(dst, vec4(src.rgb*src.a, src.a), opacity%s);", name, dissolve ? ", dissolveSeed" : ""),
									"}"
			};
		}

		private static String[] createSource(String name) {
			return createSource(name, false);
		}

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] NONE = createSource("none");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] NORMAL = createSource("normal");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DISSOLVE = createSource("dissolve", true);

//		@ShaderSource(attach=BLEND_FUNCTIONS)
//		public static final String[] DANCING_DISSOLVE = DISSOLVE;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DARKEN = createSource("darken");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] MULTIPLY = createSource("multiply");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR_BURN = createSource("color_burn");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_BURN = createSource("linear_burn");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DARKER_COLOR = createSource("darker_color");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] ADD = createSource("add");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LIGHTEN = createSource("lighten");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SCREEN = createSource("screen");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR_DODGE = createSource("color_dodge");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_DODGE = createSource("linear_dodge");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LIGHTER_COLOR = createSource("lighter_color");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] OVERLAY = createSource("overlay");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SOFT_LIGHT = createSource("soft_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HARD_LIGHT = createSource("hard_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_LIGHT = createSource("linear_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] VIVID_LIGHT = createSource("vivid_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] PIN_LIGHT = createSource("pin_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HARD_MIX = createSource("hard_mix");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DIFFERENCE = createSource("difference");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] EXCLUSION = createSource("exclusion");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HUE = createSource("hue");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SATURATION = createSource("saturation");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR = createSource("color");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LUMINOSITY = createSource("luminosity");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] STENCIL_ALPHA = createSource("stencil_alpha");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] STENCIL_LUMA = createSource("stencil_luma");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SILHOUETTE_ALPHA = createSource("silhouette_alpha");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SILHOUETTE_LUMA = createSource("silhouette_luma");

	}


	public static class PremultAndMatteWithNormalCategory {

		private static String[] createSource(String name, boolean dissolve) {
			return new String[] {
									"uniform sampler2D texDst;",
									"uniform sampler2D texSrc;",
									"uniform float opacity;",
					dissolve ?		"uniform float dissolveSeed;" : "",
									"",
					String.format(	"vec4 blend_%s(vec4 pDst, vec4 pSrc, float opacity%s);", name, dissolve ? ", float dissolveSeed" : ""),
									"",
									"void main(void)",
									"{",
									"	vec4 dst = texture2D(texDst, gl_TexCoord[0].st);",
									"	vec4 src = texture2D(texSrc, gl_TexCoord[1].st);",
					String.format(	"	vec4 blend = blend_%s(dst, src, opacity%s);", name, dissolve ? ", dissolveSeed" : ""),
									"	if (blend.a != 0.0) {",
									"		gl_FragColor = blend/blend.a * dst.a;",
									"	} else {",
									"		gl_FragColor = vec4(0.0);",
									"	}",
									"}"
			};
		}

		private static String[] createSource(String name) {
			return createSource(name, false);
		}

		@ShaderSource/*(attach=BLEND_FUNCTIONS)*/
		public static final String[] NONE = {
			"uniform sampler2D texDst;",
			"uniform sampler2D texSrc;",
			"uniform float opacity;",
			"",
		//	"vec4 blend_none(vec4 pDst, vec4 pSrc, float opacity);",
			"",
			"void main(void)",
			"{",
			"	vec4 dst = texture2D(texDst, gl_TexCoord[0].st);",
			"	vec4 src = texture2D(texSrc, gl_TexCoord[1].st);",
		//	"	gl_FragColor = blend_none(dst, src, opacity) * dst.a;",
			"	gl_FragColor = src * opacity * dst.a;",
			"}"
		};

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] NORMAL = createSource("normal");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DISSOLVE = createSource("dissolve", true);

//		@ShaderSource(attach=BLEND_FUNCTIONS)
//		public static final String[] DANCING_DISSOLVE = DISSOLVE;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DARKEN = createSource("darken");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] MULTIPLY = createSource("multiply");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR_BURN = createSource("color_burn");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_BURN = createSource("linear_burn");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DARKER_COLOR = createSource("darker_color");

		@ShaderSource
		public static final String[] ADD = {
			"uniform sampler2D texDst;",
			"uniform sampler2D texSrc;",
			"uniform float opacity;",
			"",
			"void main(void)",
			"{",
			"	vec4 dst = texture2D(texDst, gl_TexCoord[0].st);",
			"	vec4 src = texture2D(texSrc, gl_TexCoord[1].st);",
			"	gl_FragColor = vec4(src.rgb*opacity*dst.a + dst.rgb, dst.a);",
			"}"
		};

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LIGHTEN = createSource("lighten");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SCREEN = createSource("screen");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR_DODGE = createSource("color_dodge");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_DODGE = createSource("linear_dodge");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LIGHTER_COLOR = createSource("lighter_color");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] OVERLAY = createSource("overlay");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SOFT_LIGHT = createSource("soft_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HARD_LIGHT = createSource("hard_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_LIGHT = createSource("linear_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] VIVID_LIGHT = createSource("vivid_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] PIN_LIGHT = createSource("pin_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HARD_MIX = createSource("hard_mix");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DIFFERENCE = createSource("difference");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] EXCLUSION = createSource("exclusion");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HUE = createSource("hue");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SATURATION = createSource("saturation");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR = createSource("color");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LUMINOSITY = createSource("luminosity");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] STENCIL_ALPHA = Premult.STENCIL_ALPHA;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] STENCIL_LUMA = Premult.STENCIL_LUMA;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SILHOUETTE_ALPHA = Premult.SILHOUETTE_ALPHA;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SILHOUETTE_LUMA = Premult.SILHOUETTE_LUMA;

	}


	public static class PremultAndMatteWithoutNormalCategory {

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] NONE = Premult.NONE;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] NORMAL = Premult.NORMAL;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DISSOLVE = Premult.DISSOLVE;

//		@ShaderSource(attach=BLEND_FUNCTIONS)
//		public static final String[] DANCING_DISSOLVE = DISSOLVE;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DARKEN = PremultAndMatteWithNormalCategory.DARKEN;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] MULTIPLY = PremultAndMatteWithNormalCategory.MULTIPLY;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR_BURN = PremultAndMatteWithNormalCategory.COLOR_BURN;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_BURN = PremultAndMatteWithNormalCategory.LINEAR_BURN;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DARKER_COLOR = PremultAndMatteWithNormalCategory.DARKER_COLOR;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] ADD = PremultAndMatteWithNormalCategory.ADD;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LIGHTEN = PremultAndMatteWithNormalCategory.LIGHTEN;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SCREEN = PremultAndMatteWithNormalCategory.SCREEN;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR_DODGE = PremultAndMatteWithNormalCategory.COLOR_DODGE;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_DODGE = PremultAndMatteWithNormalCategory.LINEAR_DODGE;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LIGHTER_COLOR = PremultAndMatteWithNormalCategory.LIGHTER_COLOR;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] OVERLAY = PremultAndMatteWithNormalCategory.OVERLAY;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SOFT_LIGHT = PremultAndMatteWithNormalCategory.SOFT_LIGHT;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HARD_LIGHT = PremultAndMatteWithNormalCategory.HARD_LIGHT;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_LIGHT = PremultAndMatteWithNormalCategory.LINEAR_LIGHT;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] VIVID_LIGHT = PremultAndMatteWithNormalCategory.VIVID_LIGHT;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] PIN_LIGHT = PremultAndMatteWithNormalCategory.PIN_LIGHT;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HARD_MIX = PremultAndMatteWithNormalCategory.HARD_MIX;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DIFFERENCE = PremultAndMatteWithNormalCategory.DIFFERENCE;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] EXCLUSION = PremultAndMatteWithNormalCategory.EXCLUSION;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HUE = PremultAndMatteWithNormalCategory.HUE;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SATURATION = PremultAndMatteWithNormalCategory.SATURATION;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR = PremultAndMatteWithNormalCategory.COLOR;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LUMINOSITY = PremultAndMatteWithNormalCategory.LUMINOSITY;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] STENCIL_ALPHA = Premult.STENCIL_ALPHA;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] STENCIL_LUMA = Premult.STENCIL_LUMA;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SILHOUETTE_ALPHA = Premult.SILHOUETTE_ALPHA;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SILHOUETTE_LUMA = Premult.SILHOUETTE_LUMA;

	}


	private final IVideoRenderSupport support;

	private final IShaderRegistry shaders;

	private Class<?> programsClass;

	private Map<BlendMode, IShaderProgram> programs;

	@Inject
	public BlendSupportImpl(IVideoRenderSupport support, IShaderRegistry shaders) {
		this.support = support;
		this.shaders = shaders;
		setProgramsClass(Premult.class);
	}

	public Class<?> getUnmultSrcClass() {
		return UnmultSrc.class;
	}

	public Class<?> getPremultAndMatteClass(boolean normalCategory) {
		return normalCategory ? PremultAndMatteWithNormalCategory.class
							  : PremultAndMatteWithoutNormalCategory.class;
	}

	public void setProgramsClass(Class<?> programsClass) {
		Map<BlendMode, IShaderProgram> programs = new HashMap<BlendMode, IShaderProgram>();

		Set<BlendMode> values = Util.newSet(Arrays.asList(BlendMode.values()));
		values.remove(BlendMode.DANCING_DISSOLVE);

		for (BlendMode blendMode : values) {
			IShaderProgram program = shaders.getProgram(programsClass, blendMode.name());
			if (program == null) {
				throw new IllegalArgumentException(String.format(
						"no shader program found for BlendMode.%s in class %s",
						blendMode.name(), programsClass.getName()));
			}

			// TODO program に texSrc などの Uniform 変数があるかチェックすべき？
			//      チェックするなら Uniform 変数の型のチェックとかも必要？

			programs.put(blendMode, program);
		}

		this.programsClass = programsClass;
		this.programs = programs;
	}

	public void replace(BlendMode blendMode, IShaderProgram program) {
		if (blendMode == BlendMode.DANCING_DISSOLVE) {
			throw new IllegalArgumentException("DANCING_DISSOLVE uses same program as DISSOLVE");
		}

		if (program == null) {
			program = shaders.getProgram(programsClass, blendMode.name());
		}

		programs.put(blendMode, program);
	}

	private IVideoBuffer blend(
			IVideoBuffer src, IVideoBuffer dstIn, IVideoBuffer dstOut,
			BlendMode blendMode, double opacity, double dissolveSeed) {

		Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
		uniforms.add(new GLUniformData("opacity", (float)opacity));

		switch (blendMode) {
			case DANCING_DISSOLVE:
				blendMode = BlendMode.DISSOLVE;
				// fall through
			case DISSOLVE:
				uniforms.add(new GLUniformData("dissolveSeed", (float)dissolveSeed));
				// fall through
			default:
				uniforms.add(new GLUniformData("texDst", 0));
				uniforms.add(new GLUniformData("texSrc", 1));
				return support.useShaderProgram(programs.get(blendMode), uniforms, dstOut, dstIn, src);
		}
	}

	public IVideoBuffer blend(
			IVideoBuffer src, IVideoBuffer dstIn, IVideoBuffer dstOut,
			BlendMode blendMode, double opacity, IVideoEffectContext context) {

		double dissolveSeed;

		switch (blendMode) {
			case DISSOLVE:
				dissolveSeed = 100.0*context.getEffectName().hashCode()/Integer.MAX_VALUE;
				break;

			case DANCING_DISSOLVE:
				dissolveSeed = 100.0*context.getEffectName().hashCode()/Integer.MAX_VALUE
								+ context.getTime().toSecond();
				break;

			default:
				dissolveSeed = 0;
				break;
		}

		return blend(src, dstIn, dstOut, blendMode, opacity, dissolveSeed);
	}

	public IVideoBuffer blend(
			IVideoBuffer src, IVideoBuffer dstIn, IVideoBuffer dstOut,
			BlendMode blendMode, double opacity) {

		if (blendMode == BlendMode.DISSOLVE || blendMode == BlendMode.DANCING_DISSOLVE) {
			throw new IllegalArgumentException(
					"Neither DISSOLVE nor DANCING_DISSOLVE can be used."
					+ " Use anothor blend method or dissolve method.");
		}

		return blend(src, dstIn, dstOut, blendMode, opacity, 0);
	}

	public IVideoBuffer dissolve(
			IVideoBuffer src, IVideoBuffer dstIn, IVideoBuffer dstOut,
			double opacity, double seed) {

		return blend(src, dstIn, dstOut, BlendMode.DISSOLVE, opacity, seed);
	}

}
