/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#import "QTMovieOutput.h"
#import <AudioToolbox/AudioFormat.h>
#import <QTKit/QTTime.h>


NSString* HandleToNSString(Handle text)
{
	// OSXではハンドルのロックは(SetHandleSizeを使わないなら)必要ないらしいのでロックしない。
	
	Size textLen = GetHandleSize(text);
	
	NSMutableData* textData = [NSMutableData dataWithCapacity:textLen+1];
	[textData appendBytes:*text length:textLen];
	
	char nullTerm = 0;
	[textData appendBytes:&nullTerm length:1];
	
	return [NSString stringWithCString:[textData bytes] encoding:[NSString defaultCStringEncoding]];
}


@interface QTMovieOutput (private)

-(void) enterMovies;
-(void) exitMovies;

-(OSErr) getVideoProp:(OSType)propertyType:(void*)propertyValue;
-(OSErr) getVideoData:(MovieExportGetDataParams*)params;
-(OSErr) getAudioProp:(OSType)propertyType:(void*)propertyValue;
-(OSErr) getAudioData:(MovieExportGetDataParams*)params;
-(OSErr) updateProgress:(short)message:(Fixed)percentDone;

@end


static OSErr videoPropProc(void* refcon, long trackID, OSType propertyType, void* propertyValue)
{
	return [(QTMovieOutput*)refcon getVideoProp:propertyType:propertyValue];
}

static OSErr videoDataProc(void* refcon, MovieExportGetDataParams* params)
{
	return [(QTMovieOutput*)refcon getVideoData:params];
}

static OSErr audioPropProc(void* refcon, long trackID, OSType propertyType, void* propertyValue)
{
	return [(QTMovieOutput*)refcon getAudioProp:propertyType:propertyValue];
}

static OSErr audioDataProc(void* refcon, MovieExportGetDataParams* params)
{
	return [(QTMovieOutput*)refcon getAudioData:params];
}

static OSErr progressProc(Movie movie, short message, short whatOperation, Fixed percentDone, long refcon)
{
	return [(QTMovieOutput*)refcon updateProgress:message:percentDone];
}


@implementation QTMovieOutput

-(id) init
{
	if ([super init] == NULL) {
		return NULL;
	}
	
	[self performSelectorOnMainThread:@selector(enterMovies) withObject:NULL waitUntilDone:YES];
	
	return self;
}

-(void) dealloc
{
	if (mSoundDesc    != NULL) DisposeHandle((Handle)mSoundDesc);
	if (mAudioDataUPP != NULL) DisposeMovieExportGetDataUPP(mAudioDataUPP);
	if (mAudioPropUPP != NULL) DisposeMovieExportGetPropertyUPP(mAudioPropUPP);
	if (mImageDesc    != NULL) DisposeHandle((Handle)mImageDesc);
	if (mVideoDataUPP != NULL) DisposeMovieExportGetDataUPP(mVideoDataUPP);
	if (mVideoPropUPP != NULL) DisposeMovieExportGetPropertyUPP(mVideoPropUPP);
	if (mExporter     != NULL) CloseComponent(mExporter);
	
	mSoundDesc    = NULL;
	mAudioDataUPP = NULL;
	mAudioPropUPP = NULL;
	mImageDesc    = NULL;
	mVideoDataUPP = NULL;
	mVideoPropUPP = NULL;
	mExporter     = NULL;
	
	[self performSelectorOnMainThread:@selector(exitMovies) withObject:NULL waitUntilDone:YES];
	
	[super dealloc];
}

-(void) enterMovies
{
	EnterMovies();
}

-(void) exitMovies
{
	ExitMovies();
}

+(jint) doVideoCompressorSettings:(QTAtomContainer)defaultSettings
								 :(double)defaultFrameRate
								 :(BOOL)showDialog
								 :(QTAtomContainer*)newSettings
								 :(short*)newDepth
								 :(double*)newFrameRate
								 :(NSString**)newSettingsAsText
{
	jint error = noErr;
	
	ComponentInstance ci = 0;
	SCSpatialSettings ss;
	SCTemporalSettings ts;
	QTAtomContainer ac = NULL;
	Handle text = NULL;
	
	error = OpenADefaultComponent(StandardCompressionType, StandardCompressionSubType, &ci);
	if (error != noErr) {
		NSLog(@"OpenADefaultComponent: error %d", error);
		goto bail;
	}
	
//	long scPreferences = scAllowEncodingWithCompressionSession;
//	error = SCSetInfo(ci, scPreferenceFlagsType, &scPreferences);
//	if (error != noErr) {
//		NSLog(@"SCSetInfo with scPreferenceFlagsType: error %d", error);
//		goto bail;
//	}
	
	if (defaultSettings == NULL) {
		ss.codecType = kAnimationCodecType;
		ss.codec = 0;
		ss.depth = 24;
		ss.spatialQuality = codecLosslessQuality;
		
		error = SCSetInfo(ci, scSpatialSettingsType, &ss);
		if (error != noErr) {
			NSLog(@"SCSetInfo with scSpatialSettingsType: error %d", error);
			goto bail;
		}
		
		ts.keyFrameRate = 0;
		ts.temporalQuality = 0;
		
	} else {
		error = SCSetSettingsFromAtomContainer(ci, defaultSettings);
		if (error != noErr) {
			NSLog(@"SCSetSettingsFromAtomContainer: error %d", error);
			goto bail;
		}
		
		error = SCGetInfo(ci, scTemporalSettingsType, &ts);
		if (error != noErr) {
			NSLog(@"SCGetInfo with scTemporalSettingsType: error %d", error);
			goto bail;
		}
	}
	
	ts.frameRate = X2Fix(defaultFrameRate);
	
	error = SCSetInfo(ci, scTemporalSettingsType, &ts);
	if (error != noErr) {
		NSLog(@"SCSetInfo with scTemporalSettingsType: error %d", error);
		goto bail;
	}
	
	if (showDialog) {
		error = SCRequestSequenceSettings(ci);
		if (error != noErr) {
			NSLog(@"SCRequestSequenceSettings: error %d", error);
			goto bail;
		}
	}
	
	error = SCGetSettingsAsAtomContainer(ci, &ac);
	if (error != noErr) {
		NSLog(@"SCGetSettingsAsAtomContainer: error %d", error);
		goto bail;
	}
	
	error = SCGetInfo(ci, scSpatialSettingsType, &ss);
	if (error != noErr) {
		NSLog(@"SCGetInfo with scSpatialSettingsType: error %d", error);
		goto bail;
	}
	
	error = SCGetInfo(ci, scTemporalSettingsType, &ts);
	if (error != noErr) {
		NSLog(@"SCGetInfo with scTemporalSettingsType: error %d", error);
		goto bail;
	}
	
	error = SCGetSettingsAsText(ci, &text);
	if (error != noErr) {
		NSLog(@"SCGetSettingsAsText: error %d", error);
		goto bail;
	}
	
bail:
	if (error == noErr) {
		*newSettings = ac;
		*newDepth = ss.depth;
		*newFrameRate = Fix2X(ts.frameRate);
		*newSettingsAsText = HandleToNSString(text);
	} else {
		if (ac != NULL) QTDisposeAtomContainer(ac);
	}
	if (text != NULL) DisposeHandle(text);
	if (ci != 0) CloseComponent(ci);
	return error;
}

+(jint) doAudioCompressorSettings:(QTAtomContainer)defaultSettings
								 :(double)defaultSampleRate
								 :(BOOL)showDialog
								 :(QTAtomContainer*)newSettings
								 :(double*)newSampleRate
								 :(NSString**)newSettingsAsText
{
	jint error = noErr;
	
	ComponentInstance ci = 0;
	QTAtomContainer ac = NULL;
	Handle text = NULL;
	
	error = OpenADefaultComponent(StandardCompressionType, StandardCompressionSubTypeAudio, &ci);
	if (error != noErr) {
		NSLog(@"OpenADefaultComponent: error %d", error);
		goto bail;
	}
	
	AudioStreamBasicDescription asbd;
	
	if (defaultSettings == NULL) {
		AudioChannelLayoutTag layoutTag = kAudioChannelLayoutTag_Stereo;
		UInt32 layoutSize;
		error = AudioFormatGetPropertyInfo(kAudioFormatProperty_ChannelLayoutForTag,
										   sizeof(AudioChannelLayoutTag), &layoutTag, &layoutSize);
		if (error != noErr) {
			NSLog(@"AudioFormatGetPropertyInfo: error %d", error);
			goto bail;
		}
		
		NSMutableData* layoutData = [NSMutableData dataWithLength:layoutSize];
		error = AudioFormatGetProperty(kAudioFormatProperty_ChannelLayoutForTag,
									   sizeof(AudioChannelLayoutTag), &layoutTag, &layoutSize, [layoutData mutableBytes]);
		if (error != noErr) {
			NSLog(@"AudioFormatGetProperty: error %d", error);
			goto bail;
		}
		
		error = QTSetComponentProperty(ci, kQTPropertyClass_SCAudio, kQTSCAudioPropertyID_ChannelLayout,
									   layoutSize, [layoutData bytes]);
		if (error != noErr) {
			NSLog(@"QTSetComponentProperty with kQTSCAudioPropertyID_ChannelLayout: error %d", error);
			goto bail;
		}
		
		asbd.mFormatID = kAudioFormatLinearPCM;
		asbd.mFormatFlags = kAudioFormatFlagIsPacked | kAudioFormatFlagsNativeEndian | kAudioFormatFlagIsSignedInteger;
		asbd.mBitsPerChannel = 16;
		asbd.mChannelsPerFrame = 2;
		asbd.mBytesPerFrame = 4;
		asbd.mFramesPerPacket = 1;
		asbd.mBytesPerPacket = asbd.mBytesPerFrame * asbd.mFramesPerPacket;
		
	} else {
		error = SCSetSettingsFromAtomContainer(ci, defaultSettings);
		if (error != noErr) {
			NSLog(@"SCSetSettingsFromAtomContainer: error %d", error);
			goto bail;
		}
		
		error = QTGetComponentProperty(ci, kQTPropertyClass_SCAudio, kQTSCAudioPropertyID_BasicDescription,
									   sizeof(asbd), &asbd, NULL);
		if (error != noErr) {
			NSLog(@"QTGetComponentProperty with kQTSCAudioPropertyID_BasicDescription: error %d", error);
			goto bail;
		}
	}
	
	asbd.mSampleRate = defaultSampleRate;
	
	error = QTSetComponentProperty(ci, kQTPropertyClass_SCAudio, kQTSCAudioPropertyID_BasicDescription,
								   sizeof(asbd), &asbd);
	if (error != noErr) {
		NSLog(@"QTSetComponentProperty with kQTSCAudioPropertyID_BasicDescription: error %d", error);
		goto bail;
	}
	
	if (showDialog) {
		error = SCRequestImageSettings(ci);
		if (error != noErr) {
			NSLog(@"SCRequestImageSettings: error %d", error);
			goto bail;
		}
	}
	
	error = SCGetSettingsAsAtomContainer(ci, &ac);
	if (error != noErr) {
		NSLog(@"SCGetSettingsAsAtomContainer: error %d", error);
		goto bail;
	}
	
	error = QTGetComponentProperty(ci, kQTPropertyClass_SCAudio, kQTSCAudioPropertyID_BasicDescription,
								   sizeof(asbd), &asbd, NULL);
	if (error != noErr) {
		NSLog(@"QTGetComponentProperty with kQTSCAudioPropertyID_BasicDescription: error %d", error);
		goto bail;
	}
	
	error = SCGetSettingsAsText(ci, &text);
	if (error != noErr) {
		NSLog(@"SCGetSettingsAsText: error %d", error);
		goto bail;
	}
	
bail:
	if (error == noErr) {
		*newSettings = ac;
		*newSampleRate = asbd.mSampleRate;
		*newSettingsAsText = HandleToNSString(text);
	} else {
		if (ac != NULL) QTDisposeAtomContainer(ac);
	}
	if (text != NULL) DisposeHandle(text);
	if (ci != 0) CloseComponent(ci);
	return error;
}

-(jint) openMovieExportComponent
{
	ComponentDescription compDesc;
	compDesc.componentType = MovieExportType;
	compDesc.componentSubType = MovieFileType;
	compDesc.componentManufacturer = kAppleManufacturer;
	compDesc.componentFlags = canMovieExportFromProcedures;
	compDesc.componentFlagsMask = canMovieExportFromProcedures;
	
	mExporter = OpenComponent(FindNextComponent(NULL, &compDesc));
	if (mExporter == NULL) {
		NSLog(@"could not find export compontent !");
		return -1;
	}
	
	Boolean useHighResolutionAudio = true;
	QTSetComponentProperty(mExporter, kQTPropertyClass_MovieExporter,
						   kQTMovieExporterPropertyID_EnableHighResolutionAudioFeatures,
						   sizeof(Boolean), &useHighResolutionAudio);
	
	return noErr;
}

-(jint) setCompressorSettings:(QTAtomContainer)videoSettings
							 :(QTAtomContainer)audioSettings
{
	if (videoSettings == NULL && audioSettings == NULL) {
		return noErr;
	}
	
	jint error = noErr;
	QTAtomContainer ac = NULL;
	QTAtomContainer merged = NULL;
		
	if (videoSettings != NULL && audioSettings == NULL) {
		ac = videoSettings;
		
	} else if (videoSettings == NULL && audioSettings != NULL) {
		ac = audioSettings;
		
	} else if (videoSettings != NULL && audioSettings != NULL) {
		error = QTCopyAtom(videoSettings, kParentAtomIsContainer, &merged);
		if (error != noErr) {
			NSLog(@"QTCopyAtom: error %d", error);
			goto bail;
		}
		
		error = QTInsertChildren(merged, kParentAtomIsContainer, audioSettings);
		if (error != noErr) {
			NSLog(@"QTInsertChildren: error %d", error);
			goto bail;
		}
		
		ac = merged;
	}
	
	error = MovieExportSetSettingsFromAtomContainer(mExporter, ac);
	if (error != noErr) {
		NSLog(@"MovieExportSetSettingsFromAtomContainer: error %d", error);
		goto bail;
	}
	
bail:
	if (merged != NULL) QTDisposeAtomContainer(merged);
	return error;
}

-(jint) addVideoSource:(long)vWidth
					  :(long)vHeight
					  :(TimeScale)vTimeScale
					  :(TimeValue64)vFrameDuration
					  :(TimeValue64)vDuration
{
	mVideoTimeScale = vTimeScale;
	mVideoFrameDuration = vFrameDuration;
	mVideoDuration = vDuration;
	
	mVideoPropUPP = NewMovieExportGetPropertyUPP(videoPropProc);
	mVideoDataUPP = NewMovieExportGetDataUPP(videoDataProc);
	
	long trackID;
	MovieExportAddDataSource(mExporter, VideoMediaType,
							 vTimeScale,
							 &trackID,
							 mVideoPropUPP,
							 mVideoDataUPP,
							 self);
	
	mImageDesc = (ImageDescriptionHandle)NewHandleClear(sizeof(ImageDescription));
	(*mImageDesc)->idSize = sizeof(ImageDescription);
#ifdef __BIG_ENDIAN__
	(*mImageDesc)->cType = k32ARGBPixelFormat;
#else
	(*mImageDesc)->cType = k32BGRAPixelFormat;
#endif
	(*mImageDesc)->vendor = kAppleManufacturer;
	(*mImageDesc)->spatialQuality = codecLosslessQuality;
	(*mImageDesc)->width = vWidth;
	(*mImageDesc)->height = vHeight;
	(*mImageDesc)->hRes = 72L<<16;
	(*mImageDesc)->vRes = 72L<<16;
	(*mImageDesc)->dataSize = vWidth * vHeight * 4;
	(*mImageDesc)->frameCount = 1;
	(*mImageDesc)->depth = 32;
	(*mImageDesc)->clutID = -1;
	
	return noErr;
}

-(jint) addAudioSource:(long)aChannels
					  :(long)aSampleRate
					  :(long)aSampleSize
					  :(BOOL)aFloat
					  :(TimeValue64)aDuration
{
	mAudioChannels = aChannels;
	mAudioSampleRate = aSampleRate;
	mAudioSampleSize = aSampleSize;
	mAudioDuration = aDuration;
	
	mAudioPropUPP = NewMovieExportGetPropertyUPP(audioPropProc);
	mAudioDataUPP = NewMovieExportGetDataUPP(audioDataProc);
	
	long trackID;
	MovieExportAddDataSource(mExporter, SoundMediaType,
							 aSampleRate,
							 &trackID,
							 mAudioPropUPP,
							 mAudioDataUPP,
							 self);
	
	AudioStreamBasicDescription asbd;
	asbd.mFormatID = kAudioFormatLinearPCM;
	asbd.mFormatFlags = kAudioFormatFlagIsPacked | kAudioFormatFlagsNativeEndian
			| (aFloat ? kAudioFormatFlagIsFloat : kAudioFormatFlagIsSignedInteger);
	asbd.mSampleRate = aSampleRate;
	asbd.mBitsPerChannel = aSampleSize * 8;
	asbd.mChannelsPerFrame = aChannels;
	asbd.mBytesPerFrame = aSampleSize * asbd.mChannelsPerFrame;
	asbd.mFramesPerPacket = 1;
	asbd.mBytesPerPacket = asbd.mBytesPerFrame * asbd.mFramesPerPacket;
	
	AudioChannelLayoutTag layoutTag;
	switch (aChannels) {
		case 2:
			layoutTag = kAudioChannelLayoutTag_Stereo;
			break;
		default:
			NSLog(@"unsupported number of channels: channels=%d", aChannels);
			return paramErr;
	}
	
	UInt32 layoutSize;
	OSStatus error = AudioFormatGetPropertyInfo(kAudioFormatProperty_ChannelLayoutForTag,
												sizeof(AudioChannelLayoutTag), &layoutTag, &layoutSize);
	if (error != noErr) {
		NSLog(@"AudioFormatGetPropertyInfo: error %d", error);
		return error;
	}
	
	NSMutableData* layoutData = [NSMutableData dataWithLength:layoutSize];
	error = AudioFormatGetProperty(kAudioFormatProperty_ChannelLayoutForTag,
								   sizeof(AudioChannelLayoutTag), &layoutTag, &layoutSize, [layoutData mutableBytes]);
	if (error != noErr) {
		NSLog(@"AudioFormatGetProperty: error %d", error);
		return error;
	}
	
	error = QTSoundDescriptionCreate(&asbd, (AudioChannelLayout*)[layoutData bytes], layoutSize, NULL, 0,
									 kQTSoundDescriptionKind_Movie_LowestPossibleVersion, &mSoundDesc);
	if (error != noErr) {
		NSLog(@"QTSoundDescriptionCreate: error %d", error);
		return error;
	}
	
	return noErr;
}

-(jint) doOutput:(JNIEnv*)jniEnv
				:(jobject)javaObj
				:(CFStringRef)filename
{
	mJNIEnv = jniEnv;
	mJavaObj = javaObj;
	jclass class = (*jniEnv)->GetObjectClass(jniEnv, javaObj);
	
	jint error = noErr;
	
	if (mVideoDataUPP != NULL) {
		size_t videoBufSize = (*mImageDesc)->dataSize;
		mVideoBuffer = malloc(videoBufSize);
		if (mVideoBuffer == NULL) {
			NSLog(@"malloc: failed");
			error = memFullErr;
			goto bail;
		}
		mVideoBufferDirect = (*jniEnv)->NewDirectByteBuffer(jniEnv, mVideoBuffer, videoBufSize);
		if (mVideoBufferDirect == NULL) {
			NSLog(@"NewDirectByteBuffer: failed");
			error = memFullErr;
			goto bail;
		}
		mVideoRequestMethod = (*jniEnv)->GetMethodID(jniEnv, class, "videoRequest", "(JILjava/nio/ByteBuffer;)I");
	}
	
	if (mAudioDataUPP != NULL) {
		size_t audioBufSize = mAudioChannels * mAudioSampleSize * mAudioSampleRate;
		mAudioBuffer = malloc(audioBufSize);
		if (mAudioBuffer == NULL) {
			NSLog(@"malloc: failed");
			error = memFullErr;
			goto bail;
		}
		mAudioBufferDirect = (*jniEnv)->NewDirectByteBuffer(jniEnv, mAudioBuffer, audioBufSize);
		if (mAudioBufferDirect == NULL) {
			NSLog(@"NewDirectByteBuffer: failed");
			error = memFullErr;
			goto bail;
		}
		mAudioRequestMethod = (*jniEnv)->GetMethodID(jniEnv, class, "audioRequest", "(JILjava/nio/ByteBuffer;)I");
	}
	
	mProgressMethod = (*jniEnv)->GetMethodID(jniEnv, class, "updateProgress", "(SI)Z");
	
	MovieProgressUPP progressUPP = NewMovieProgressUPP(progressProc);
	MovieExportSetProgressProc(mExporter, progressUPP, (long)self);
	
	Handle dataRef;
	OSType dataType;
	QTNewDataReferenceFromFullPathCFString(filename, kQTNativeDefaultPathStyle, 0, &dataRef, &dataType);
	
	error = MovieExportFromProceduresToDataRef(mExporter, dataRef, dataType);
	
	DisposeHandle(dataRef);
	DisposeMovieProgressUPP(progressUPP);
	
	if (error == userCanceledErr) {
		goto bail;
	} else if (error != noErr) {
		NSLog(@"MovieExportFromProceduresToDataRef: error %d", error);
		goto bail;
	}
	
	
bail:
	if (mAudioBufferDirect != NULL) (*jniEnv)->DeleteLocalRef(jniEnv, mAudioBufferDirect);
	if (mAudioBuffer != NULL) free(mAudioBuffer);
	if (mVideoBufferDirect != NULL) (*jniEnv)->DeleteLocalRef(jniEnv, mVideoBufferDirect);
	if (mVideoBuffer != NULL) free(mVideoBuffer);
	
	mAudioBufferDirect = NULL;
	mAudioBuffer = NULL;
	mVideoBufferDirect = NULL;
	mVideoBuffer = NULL;
	
	return error;
}

-(OSErr) getVideoProp:(OSType)propertyType:(void*)propertyValue
{
	switch (propertyType) {
		case movieExportDuration:
			QTGetTimeRecord(QTMakeTime(mVideoDuration, mVideoTimeScale), (TimeRecord*)propertyValue);
			return noErr;
			
		case movieExportUseConfiguredSettings:
			*(Boolean*)propertyValue = true;
			return noErr;
			
		default:
			return paramErr;	// non-zero value means: use default value provided by export component
	}
}

-(OSErr) getVideoData:(MovieExportGetDataParams*)params
{
	if (params->requestedTime >= mVideoDuration) {
		return eofErr;
	}
	
	jint actualDataSize = (*mJNIEnv)->CallIntMethod(mJNIEnv, mJavaObj, mVideoRequestMethod,
													(jlong)params->requestedTime, (jint)mVideoTimeScale, mVideoBufferDirect);
	if (actualDataSize == -1) {
		return userCanceledErr;
	}
	
	params->actualTime = params->requestedTime;
	params->dataPtr = mVideoBuffer;
	params->dataSize = actualDataSize;
	params->desc = (SampleDescriptionHandle)mImageDesc;
	params->descType = VideoMediaType;
	params->descSeed = 1;
	params->actualSampleCount = 1;
	params->durationPerSample = mVideoFrameDuration;
	params->sampleFlags = 0L;
	
	return noErr;
}

-(OSErr) getAudioProp:(OSType)propertyType:(void*)propertyValue
{
	switch (propertyType) {
		case movieExportDuration:
			QTGetTimeRecord(QTMakeTime(mAudioDuration, mAudioSampleRate), (TimeRecord*)propertyValue);
			return noErr;
			
		case movieExportUseHighResolutionAudioProperties:
		case movieExportUseConfiguredSettings:
			*(Boolean*)propertyValue = true;
			return noErr;
			
		default:
			return paramErr;	// non-zero value means: use default value provided by export component
	}
}

-(OSErr) getAudioData:(MovieExportGetDataParams*)params
{
	if (params->requestedTime >= mAudioDuration) {
		return eofErr;
	}
	
	jint actualDataSize = (*mJNIEnv)->CallIntMethod(mJNIEnv, mJavaObj, mAudioRequestMethod,
													(jlong)params->requestedTime, (jint)mAudioSampleRate, mAudioBufferDirect);
	if (actualDataSize == -1) {
		return userCanceledErr;
	}
	
	params->actualTime = params->requestedTime;
	params->dataPtr = mAudioBuffer;
	params->dataSize = actualDataSize;
	params->desc = (SampleDescriptionHandle)mSoundDesc;
	params->descType = SoundMediaType;
	params->descSeed = 1;
	params->actualSampleCount = actualDataSize / (mAudioChannels * mAudioSampleSize);
	params->durationPerSample = 1;
	params->sampleFlags = 0L;
	
	return noErr;
}

-(OSErr) updateProgress:(short)message:(Fixed)percentDone
{
	jboolean progressOK = (*mJNIEnv)->CallBooleanMethod(mJNIEnv, mJavaObj, mProgressMethod, message, percentDone);
	return progressOK ? noErr : userCanceledErr;
}

@end
