/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.blurSharpen;

import java.nio.FloatBuffer;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.media.opengl.GL2;
import javax.media.opengl.GLUniformData;
import javax.vecmath.Vector2d;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableLayerReference;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.ShaderType;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.IVideoBuffer.TextureWrapMode;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IAntiAliasSupport;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.ITexture1DSupport;
import ch.kuramo.javie.api.services.IVBOCache;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.api.services.IVBOCache.VBOCacheRecord;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.LensBlur", category=Categories.BLUR_AND_SHARPEN)
public class LensBlur {

	public enum DepthMapChannel { LUMINANCE, RED, GREEN, BLUE, ALPHA }

	public enum IrisShape { TRIANGLE, SQUARE, PENTAGON, HEXAGON, HEPTAGON, OCTAGON, IRIS_LAYER }

	public enum RadiusLimit { NONE, LIMIT20, LIMIT40, LIMIT60, LIMIT80 }

//	public enum NoiseDistribution { UNIFORM, GAUSSIAN }

	@Property
	private IAnimatableLayerReference depthMapLayer;

	@Property("LUMINANCE")
	private IAnimatableEnum<DepthMapChannel> depthMapChannel;

	@Property
	private IAnimatableBoolean invertDepthMap;

	@Property("true")
	private IAnimatableBoolean stretchMapToFit;

	@Property(value="0", min="0", max="255")
	private IAnimatableDouble blurFocalDistance;

	@Property("HEXAGON")
	private IAnimatableEnum<IrisShape> irisShape;

	@Property
	private IAnimatableLayerReference irisLayer;

	@Property(value="LIMIT20")
	private IAnimatableEnum<RadiusLimit> radiusLimit;

	@Property(value="0", min="0", max="100")
	private IAnimatableDouble irisRadius;

	@Property(value="0", min="0", max="100")
	private IAnimatableDouble irisBladeCurvature;

	@Property(value="0"/*, min="0", max="360"*/)
	private IAnimatableDouble irisRotation;

	@Property(value="10", min="0")
	private IAnimatableDouble nonlinear;

//	@Property(value="0", min="0", max="100")
//	private IAnimatableDouble specularBrightness;
//
//	@Property(value="255", min="0", max="255")
//	private IAnimatableDouble specularThreshold;
//
//	@Property(value="0", min="0", max="100")
//	private IAnimatableDouble noiseAmount;
//
//	@Property("UNIFORM")
//	private IAnimatableEnum<NoiseDistribution> noiseDistribution;
//
//	@Property("false")
//	private IAnimatableBoolean monochromaticNoise;

	@Property("false")
	private IAnimatableBoolean repeatEdgePixels;


	private static final ConcurrentMap<String, float[]> coveragesCache = new ConcurrentHashMap<String, float[]>();


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IAntiAliasSupport aaSupport;

	private final IArrayPools arrayPools;

	private final IVBOCache vboCache;

	private final ITexture1DSupport tex1dSupport;

	private final IShaderRegistry shaders;

	private final IShaderProgram logarithmProgram;

	private final IShaderProgram coverageProgram;

	@Inject
	public LensBlur(IVideoEffectContext context, IVideoRenderSupport support,
			IAntiAliasSupport aaSupport, IArrayPools arrayPools,
			IVBOCache vboCache, ITexture1DSupport tex1dSupport, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		this.aaSupport = aaSupport;
		this.arrayPools = arrayPools;
		this.vboCache = vboCache;
		this.tex1dSupport = tex1dSupport;
		this.shaders = shaders;

		logarithmProgram = shaders.getProgram(LensBlur.class, "LOGARITHM");
		coverageProgram = shaders.getProgram(LensBlur.class, "COVERAGE");
	}

	public VideoBounds getVideoBounds() {
		VideoBounds srcBounds = context.getPreviousBounds();
		if (srcBounds.isEmpty()) {
			return srcBounds;
		}

		if (context.value(this.repeatEdgePixels)) {
			return srcBounds;
		}

		double irisRadius = context.value(this.irisRadius);
		RadiusLimit radiusLimit = context.value(this.radiusLimit);
		if (radiusLimit != RadiusLimit.NONE) {
			irisRadius = Math.min(irisRadius, radiusLimit.ordinal()*20);
		}
		irisRadius = context.getVideoResolution().scale(irisRadius);
		int ceiledRadius = (int) Math.ceil(irisRadius);

		return new VideoBounds(srcBounds.x-ceiledRadius, srcBounds.y-ceiledRadius,
						srcBounds.width+ceiledRadius*2, srcBounds.height+ceiledRadius*2);
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();
		VideoBounds srcBounds = source.getBounds();
		if (srcBounds.isEmpty()) {
			return source;
		}

		IVideoBuffer irisBuffer = null;
		IVideoBuffer depthBuffer = null;
		IVideoBuffer buffer1 = null;
		IVideoBuffer buffer2 = null;
		int coveragesTex = 0;
		try {
			boolean repeatEdgePixels = context.value(this.repeatEdgePixels);

			double irisRadius = context.value(this.irisRadius);
			RadiusLimit radiusLimit = context.value(this.radiusLimit);
			if (radiusLimit != RadiusLimit.NONE) {
				irisRadius = Math.min(irisRadius, radiusLimit.ordinal()*20);
			}
			irisRadius = context.getVideoResolution().scale(irisRadius);
			int ceiledRadius = (int) Math.ceil(irisRadius);

			final int[] vboIdAndNumPoints;
			final VideoBounds resultBounds;
			if (repeatEdgePixels) {
				vboIdAndNumPoints = createPointsBuffer(srcBounds, ceiledRadius);
				resultBounds = srcBounds;
			} else {
				vboIdAndNumPoints = createPointsBuffer(srcBounds, 0);
				resultBounds = new VideoBounds(srcBounds.x-ceiledRadius, srcBounds.y-ceiledRadius,
										srcBounds.width+ceiledRadius*2, srcBounds.height+ceiledRadius*2);
			}


			IrisShape irisShape = context.value(this.irisShape);
			double irisRotation = context.value(this.irisRotation);
			float[] irisCoverages = new float[101];

			if (irisShape != IrisShape.IRIS_LAYER) {
				double irisBladeCurvature = context.value(this.irisBladeCurvature) / 100;
				irisBuffer = createPolygonIris(irisRadius, irisShape.ordinal()+3,
											irisBladeCurvature, irisRotation, irisCoverages);
			} else {
				IVideoBuffer irisLayer = null;
				try {
					irisLayer = context.getLayerVideoFrame(this.irisLayer);
					if (irisLayer != null) {
						irisLayer.setTextureFilter(TextureFilter.MIPMAP);
						irisBuffer = createLayerIris(irisLayer, irisRadius, irisRotation, irisCoverages);
					} else {
						irisBuffer = createPolygonIris(irisRadius, 4, 0, irisRotation-45, irisCoverages);
					}
				} finally {
					if (irisLayer != null) irisLayer.dispose();
				}
			}


			depthBuffer = context.getLayerVideoFrame(this.depthMapLayer);
			
			boolean stretchDepth;
			if (depthBuffer == null) {
				depthBuffer = context.createVideoBuffer(new VideoBounds(1, 1));
				depthBuffer.clear(Color.WHITE);
				stretchDepth = true;
			} else {
				stretchDepth = context.value(this.stretchMapToFit);
			}

			final IShaderProgram program = getProgram(
					context.value(depthMapChannel), context.value(invertDepthMap), stretchDepth);

			final int coveragesTex_ = coveragesTex = tex1dSupport.texture1DFromArray(
					irisCoverages, GL2.GL_ALPHA, GL2.GL_ALPHA16F, GL2.GL_NEAREST, GL2.GL_CLAMP_TO_EDGE);

			int maxPointSize = ceiledRadius * 2 + 3;
			final int maxNumPointsPerOnePass = (int) Math.ceil(vboIdAndNumPoints[1] * 400.0 / (maxPointSize*maxPointSize));

			Runnable operation = new Runnable() {
				public void run() {
					support.ortho2D(resultBounds);

					GL2 gl = context.getGL().getGL2();
					gl.glClearColor(0, 0, 0, 0);
					gl.glClear(GL2.GL_COLOR_BUFFER_BIT);

					gl.glEnable(GL2.GL_BLEND);
					gl.glBlendFunc(GL2.GL_ONE, GL2.GL_ONE);
					gl.glBlendEquation(GL2.GL_FUNC_ADD);

					gl.glEnable(GL2.GL_VERTEX_PROGRAM_POINT_SIZE);
					gl.glEnable(GL2.GL_POINT_SPRITE);

					// FIXME GL_POINT_SPRITE_COORD_ORIGIN はどの属性ビットで保存されるのか不明なので、個別に保存／復帰している。
					int[] pointSpriteCoordOrigin = new int[1];
					gl.glGetIntegerv(GL2.GL_POINT_SPRITE_COORD_ORIGIN, pointSpriteCoordOrigin, 0);

					gl.glActiveTexture(GL2.GL_TEXTURE3);
					gl.glBindTexture(GL2.GL_TEXTURE_1D, coveragesTex_);

					int attr1Loc = program.getAttributeLocation("attr1");
					try {
						gl.glPointParameteri(GL2.GL_POINT_SPRITE_COORD_ORIGIN, GL2.GL_LOWER_LEFT);
						gl.glEnableVertexAttribArray(attr1Loc);
						gl.glBindBuffer(GL2.GL_ARRAY_BUFFER, vboIdAndNumPoints[0]);

						gl.glVertexAttribPointer(attr1Loc, 2, GL2.GL_FLOAT, false, 2*4, 0);
													// 2: ひとつの頂点あたり配列要素2つ
													// 2*4: ストライドの要素数 x floatのバイト数

						for (int i = 0, offset = 0; offset < vboIdAndNumPoints[1]; ++i) {
							int count = Math.min(vboIdAndNumPoints[1] - offset, maxNumPointsPerOnePass);
							gl.glDrawArrays(GL2.GL_POINTS, offset, count);
							gl.glFinish();
							offset += count;
						}

					} finally {
						gl.glBindBuffer(GL2.GL_ARRAY_BUFFER, 0);
						gl.glDisableVertexAttribArray(attr1Loc);
						gl.glPointParameteri(GL2.GL_POINT_SPRITE_COORD_ORIGIN, pointSpriteCoordOrigin[0]);
					}
				}
			};

			double focalDistance = context.value(this.blurFocalDistance) / 255;
			double nonlinear = context.value(this.nonlinear)*10+1.0001;

			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture", 0));
			uniforms.add(new GLUniformData("iris", 1));
			uniforms.add(new GLUniformData("depth", 2));
			uniforms.add(new GLUniformData("coverages", 3));
			uniforms.add(new GLUniformData("texOffset", 2, toFloatBuffer(-srcBounds.x, -srcBounds.y)));
			uniforms.add(new GLUniformData("o_div_txsz", 2, toFloatBuffer(1.0/srcBounds.width, 1.0/srcBounds.height)));
			uniforms.add(new GLUniformData("focalDistance", (float)focalDistance));
			uniforms.add(new GLUniformData("irisRadius", (float)irisRadius));
			uniforms.add(new GLUniformData("nonlinear", (float)nonlinear));
			uniforms.add(new GLUniformData("o_div_nmo", (float)(1/(nonlinear-1))));

			buffer1 = support.createVideoBuffer(resultBounds, ColorMode.RGBA32_FLOAT);

			source.setTextureWrapMode(repeatEdgePixels ? TextureWrapMode.CLAMP_TO_EDGE : TextureWrapMode.CLAMP_TO_BORDER);

			VideoBounds depthBounds = depthBuffer.getBounds();
			boolean mipmap = false;
			if (stretchDepth) {
				mipmap = (depthBounds.width < srcBounds.width || depthBounds.height < srcBounds.height);
			} else {
				uniforms.add(new GLUniformData("depTexOffset", 2, toFloatBuffer(
						(depthBounds.width-srcBounds.width)*0.5, (depthBounds.height-srcBounds.height)*0.5)));
				uniforms.add(new GLUniformData("o_div_depsz", 2, toFloatBuffer(
						1.0/depthBounds.width, 1.0/depthBounds.height)));
			}
			depthBuffer.setTextureFilter(mipmap ? TextureFilter.MIPMAP : TextureFilter.LINEAR);
			depthBuffer.setTextureWrapMode(TextureWrapMode.CLAMP_TO_EDGE);

			int pushAttribs = GL2.GL_COLOR_BUFFER_BIT | GL2.GL_ENABLE_BIT | GL2.GL_TEXTURE_BIT;
			support.useShaderProgram(program, uniforms, operation, pushAttribs, buffer1, source, irisBuffer, depthBuffer);


			uniforms.clear();
			uniforms.add(new GLUniformData("texture", 0));
			uniforms.add(new GLUniformData("n_minus_o", (float)(nonlinear-1.0)));
			uniforms.add(new GLUniformData("o_div_ln", (float)(1/Math.log(nonlinear))));

			buffer2 = support.createVideoBuffer(resultBounds);
			support.useShaderProgram(logarithmProgram, uniforms, buffer2, buffer1);

			IVideoBuffer result = buffer2;
			buffer2 = null;
			return result;

		} finally {
			if (coveragesTex != 0) context.getGL().glDeleteTextures(1, new int[] { coveragesTex }, 0);
			if (buffer2 != null) buffer2.dispose();
			if (buffer1 != null) buffer1.dispose();
			if (depthBuffer != null) depthBuffer.dispose();
			if (irisBuffer != null) irisBuffer.dispose();
			if (source != null) source.dispose();
		}
	}

	private int[] createPointsBuffer(VideoBounds srcBounds, int ceiledRadius) {
		int numVerticesX = srcBounds.width + ceiledRadius * 2;
		int numVerticesY = srcBounds.height + ceiledRadius * 2;
		int numVertices = numVerticesX * numVerticesY;

		String token = String.format("%d,%d,%d,%d/%d",
				Double.doubleToLongBits(srcBounds.x), Double.doubleToLongBits(srcBounds.y),
				srcBounds.width, srcBounds.height, ceiledRadius);

		VBOCacheRecord rec = vboCache.get(this);
		if (rec != null && token.equals(rec.data)) {
			return new int[] { rec.vboId, numVertices };
		}

		GL2 gl = context.getGL().getGL2();
		int[] vboId = new int[1];
		IArray<float[]> data = null;
		try {
			data = arrayPools.getFloatArray(numVertices * 2);
			float[] array = data.getArray();

			for (int j = 0, k = 0; j < numVerticesY; ++j) {
				for (int i = 0; i < numVerticesX; ++i) {
					array[k++] = (float)(srcBounds.x-ceiledRadius + i + 0.5);
					array[k++] = (float)(srcBounds.y-ceiledRadius + j + 0.5);
				}
			}

			gl.glGenBuffers(1, vboId, 0);
			gl.glBindBuffer(GL2.GL_ARRAY_BUFFER, vboId[0]);
			gl.glBufferData(GL2.GL_ARRAY_BUFFER, data.getLength()*4,
					FloatBuffer.wrap(data.getArray()), GL2.GL_STATIC_DRAW);

			vboCache.put(this, new VBOCacheRecord(vboId[0], token));

			int[] result = { vboId[0], numVertices };
			vboId = null;
			return result;

		} finally {
			gl.glBindBuffer(GL2.GL_ARRAY_BUFFER, 0);
			if (vboId != null && vboId[0] != 0) gl.glDeleteBuffers(1, vboId, 0);
			if (data != null) data.release();
		}
	}

	private IVideoBuffer createPolygonIris(
			double radius, final int n, final double bladeCurvature,
			final double rotation, float[] coverages) {

		int pot = 1;
		while (pot < radius*2+1 && pot < 128) {
			pot <<= 1;
		}
		int texSize = Math.max(8, pot);
		final double texRadius = texSize * 0.5;
		final VideoBounds bounds = new VideoBounds(texSize, texSize);

		IVideoBuffer buffer = null;
		try {
			Runnable operation = new Runnable() {
				public void run() {
					aaSupport.antiAlias(bounds.width, bounds.height, new Runnable() {
						public void run() {
							support.ortho2D(bounds);

							GL2 gl = context.getGL().getGL2();
							gl.glColor4f(1, 1, 1, 1);
							gl.glBegin(GL2.GL_POLYGON);

							int n2 = (int)Math.ceil(360.0/n/5);
							double dr = 360.0/n/n2;

							for (int i = 0; i < n; ++i) {
								double r0 = Math.toRadians(rotation + 360.0/n*i);
								double x0 = texRadius + texRadius*Math.sin(r0);
								double y0 = texRadius - texRadius*Math.cos(r0);
								gl.glVertex2f((float)x0, (float)y0);

								if (bladeCurvature > 0) {
									double r1 = r0 + Math.toRadians(360.0/n);
									double x1 = texRadius + texRadius*Math.sin(r1);
									double y1 = texRadius - texRadius*Math.cos(r1);
									Vector2d v1 = new Vector2d(x1-x0, y1-y0);
									v1.normalize();

									for (int j = 1; j < n2; ++j) {
										double r2 = r0 + Math.toRadians(dr*j);
										double x2 = texRadius + texRadius*Math.sin(r2);
										double y2 = texRadius - texRadius*Math.cos(r2);
										Vector2d v2 = new Vector2d(x2-x0, y2-y0);

										Vector2d v3 = new Vector2d(v1);
										v3.scale(v2.dot(v1));

										gl.glVertex2f((float)((x0+v3.x)*(1-bladeCurvature) + x2*bladeCurvature),
													  (float)((y0+v3.y)*(1-bladeCurvature) + y2*bladeCurvature));
									}
								}
							}

							gl.glEnd();
						}
					});
				}
			};

			buffer = context.createVideoBuffer(bounds);

			int pushAttribs = GL2.GL_CURRENT_BIT;
			support.useFramebuffer(operation, pushAttribs, buffer);

			buffer.setTextureFilter(TextureFilter.MIPMAP);

			String key = String.format("%d,%d,%d,%d", texSize, n,
					Double.doubleToLongBits(bladeCurvature), Double.doubleToLongBits(rotation));
			float[] coveragesFromCache = coveragesCache.get(key);
			if (coveragesFromCache != null) {
				System.arraycopy(coveragesFromCache, 0, coverages, 0, coverages.length);
			} else {
				measureCoverages(buffer, coverages);
				coveragesCache.putIfAbsent(key, coverages.clone());
			}

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

	private IVideoBuffer createLayerIris(
			IVideoBuffer irisLayer, double radius, final double rotation, float[] coverages) {

		int pot = 1;
		while (pot < radius*2+1 && pot < 128) {
			pot <<= 1;
		}
		int texSize = Math.max(8, pot);
		final double texRadius = texSize * 0.5;
		final VideoBounds bounds = new VideoBounds(texSize, texSize);

		IVideoBuffer buffer = null;
		IVideoBuffer copy = null;
		try {
			Runnable operation = new Runnable() {
				public void run() {
					aaSupport.antiAlias(bounds.width, bounds.height, new Runnable() {
						public void run() {
							support.ortho2D(bounds);

							GL2 gl = context.getGL().getGL2();
							gl.glClearColor(0, 0, 0, 0);
							gl.glClear(GL2.GL_COLOR_BUFFER_BIT);
							gl.glColor4f(1, 1, 1, 1);

							gl.glTranslatef((float)texRadius, (float)texRadius, 0);
							gl.glRotatef((float)rotation, 0, 0, 1);
							gl.glTranslatef((float)-texRadius, (float)-texRadius, 0);

							double lt = texRadius - texRadius / Math.sqrt(2);
							double rb = texRadius + texRadius / Math.sqrt(2);
							support.quad2D(lt, lt, rb, rb, new double[][] {{0,0},{1,0},{1,1},{0,1}});
						}
					});
				}
			};

			buffer = context.createVideoBuffer(bounds);

			int pushAttribs = GL2.GL_CURRENT_BIT | GL2.GL_COLOR_BUFFER_BIT;
			support.useFramebuffer(operation, pushAttribs, buffer, irisLayer);

			buffer.setTextureFilter(TextureFilter.MIPMAP);
			measureCoverages(buffer, coverages);

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (copy != null) copy.dispose();
			if (buffer != null) buffer.dispose();
		}
	}

	private FloatBuffer toFloatBuffer(double...values) {
		float[] farray = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			farray[i] = (float)values[i];
		}
		return FloatBuffer.wrap(farray);
	}

	private IShaderProgram getProgram(
			DepthMapChannel depthChannel, boolean invertDepth, boolean stretchDepth) {

		String programName = LensBlur.class.getName()
						+ "." + depthChannel.name()
						+ (invertDepth ? ".INVERT_DEPTH" : "")
						+ (stretchDepth ? ".STRETCH_DEPTH" : "");
		IShaderProgram program = shaders.getProgram(programName);
		if (program == null) {
			String[] vertexSource = createVertexShaderSource(depthChannel, invertDepth, stretchDepth);
			String vertexName = programName + "_VS";
			shaders.registerShader(vertexName, ShaderType.VERTEX_SHADER, vertexSource);
			program = shaders.registerProgram(programName, ShaderType.FRAGMENT_SHADER,
											new String[] { vertexName }, FS_SOURCE);
		}
		return program;
	}

	private static String[] createVertexShaderSource(
			DepthMapChannel depthChannel, boolean invertDepth, boolean stretchDepth) {

		boolean inv = invertDepth;
		boolean str = stretchDepth;

		return new String[] {
				"#define CHANNEL " + depthChannel.ordinal(),	// 0:LUMINANCE, 1:RED, 2:GREEN, 3:BLUE, 3:ALPHA
		  inv ? "#define INVERT_DEPTH" : "",
		  str ? "#define STRETCH_DEPTH" : "",
				"",
				"attribute vec2 attr1;",
				"",
				"uniform sampler2D texture;",
				"uniform sampler2D depth;",
				"uniform sampler1D coverages;",
				"uniform vec2 texOffset;",
				"uniform vec2 o_div_txsz;",			// 1.0/texSize
				"uniform float focalDistance;",
				"uniform float irisRadius;",
				"uniform float nonlinear;",
				"uniform float o_div_nmo;",			// 1.0/(nonlinear-1.0)
				"",
				"#ifndef STRETCH_DEPTH",
				"	uniform vec2 depTexOffset;",	// (depTexSize-texSize)*0.5
				"	uniform vec2 o_div_depsz;",		// 1.0/depTexSize
				"#endif",
				"",
				"varying vec4 frontColorA;",
				"varying vec4 frontColorB;",
				"varying vec2 pointCoordAdjAB;",
				"",
				"void main(void)",
				"{",
				"", // モデルビュー行列は単位行列なので gl_ProjectionMatrix と gl_ModelViewProjectionMatrix は同じ。
				"	gl_Position = gl_ProjectionMatrix * vec4(attr1, 0.0, 1.0);",
				"",
				"	vec2 texPoint = attr1.xy + texOffset;",
				"	vec2 texCoord = texPoint * o_div_txsz;",
				"	vec4 color = texture2D(texture, texCoord);",
				"",
				"#ifdef STRETCH_DEPTH",
				"	vec4 depColor = texture2D(depth, texCoord);",
				"#else",
				"	vec4 depColor = texture2D(depth, (texPoint + depTexOffset) * o_div_depsz);",
				"#endif",
				"#if CHANNEL == 0",
				"	float depValue = dot(depColor.rgb, vec3(0.299, 0.587, 0.114));",
				"#elif CHANNEL == 1",
				"	float depValue = depColor.r;",
				"#elif CHANNEL == 2",
				"	float depValue = depColor.g;",
				"#elif CHANNEL == 3",
				"	float depValue = depColor.b;",
				"#elif CHANNEL == 4",
				"	float depValue = depColor.a;",
				"#endif",
				"#ifdef INVERT_DEPTH",
				"	depValue = 1.0 - depValue;",
				"#endif",
				"	float blur = depValue - focalDistance;",
				"	float absBlur = abs(blur);",
				"",
				"	float radius = irisRadius * absBlur;",
				"	float radiusA = floor(radius);",
				"	float radiusB = radiusA + 1.0;",
				"	vec2 radiusAB = vec2(radiusA, radiusB);",
				"	vec2 radiusAB05 = radiusAB + 0.5;",
				"	vec2 squareAB = radiusAB05 * radiusAB05 * 4.0;",
				"	vec2 cvrCoordAB = radiusAB05 / 101.0;",
				"	vec2 areaAB = squareAB * vec2(texture1D(coverages, cvrCoordAB.x).a,",
				"								  texture1D(coverages, cvrCoordAB.y).a);",
				"	float t = radius - radiusA;",
				"	color.rgb = (pow(vec3(nonlinear), color.rgb)-1.0) * o_div_nmo;",
				"	frontColorA = color / areaAB.x * (1.0-t);",
				"	frontColorB = color / areaAB.y * t;",
				"	gl_PointSize = radiusB * 2.0 + 3.0;",
				"	pointCoordAdjAB = gl_PointSize / (radiusAB * 2.0 + 1.0) * (blur < 0.0 ? -1.0 : 1.0);",
				"}"
		};
	}

	private static final String[] FS_SOURCE = {
		"#version 120",
		"",
		"uniform sampler2D iris;",
		"",
		"varying vec4 frontColorA;",
		"varying vec4 frontColorB;",
		"varying vec2 pointCoordAdjAB;",
		"",
		"void main(void)",
		"{",
		"	vec2 pointCoord2 = gl_PointCoord - 0.5;",
		"	vec2 irisCoordA = pointCoord2 * pointCoordAdjAB.x + 0.5;",
		"	vec2 irisCoordB = pointCoord2 * pointCoordAdjAB.y + 0.5;",
		"	vec4 irisColorA = texture2D(iris, irisCoordA);",
		"	vec4 irisColorB = texture2D(iris, irisCoordB);",
		"	irisColorA.a = dot(irisColorA.rgb, vec3(0.299, 0.587, 0.114));",
		"	irisColorB.a = dot(irisColorB.rgb, vec3(0.299, 0.587, 0.114));",
		"	gl_FragColor = frontColorA * irisColorA + frontColorB * irisColorB;",
		"}"
	};

	@ShaderSource
	public static final String[] LOGARITHM = {
		"uniform sampler2D texture;",
		"uniform float n_minus_o;",		// nonlinear-1
		"uniform float o_div_ln;",		// 1/log(nonlinear)
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2D(texture, gl_TexCoord[0].st);",
		"	color.rgb = log(color.rgb * n_minus_o + 1.0) * o_div_ln;",
		"	gl_FragColor = color;",
		"}"
	};

	private void measureCoverages(IVideoBuffer irisBuffer, float[] coverages) {
		VideoBounds bounds0 = irisBuffer.getBounds();
		int maxLv = 0;
		int pot = 1;
		while (pot < bounds0.width) {
			pot <<= 1;
			++maxLv;
		}

		for (int i = 0, lv = maxLv; i < coverages.length; ++i) {
			int irisSize = i*2+1;

			// FIXME 20より大きい場合、10間隔でいいか？
			if (i <= 20 || i%10 == 0) {
				int pointSize = i*2+3;
				coverages[i] = (float)measureCoverage(irisBuffer, irisSize, pointSize);
			} else {
				coverages[i] = coverages[i-1];
			}

			while (irisSize > bounds0.width/(1<<lv)) {
				if (--lv < 0) {
					Arrays.fill(coverages, i+1, coverages.length, coverages[i]);
					return;
				}
			}
		}
	}

	private double measureCoverage(IVideoBuffer irisBuffer, double irisSize, final int pointSize) {
		IVideoBuffer tmp = null;
		try {
			final VideoBounds bounds = new VideoBounds(pointSize, pointSize);
			tmp = support.createVideoBuffer(bounds, ColorMode.RGBA16_FLOAT);

			Runnable operation = new Runnable() {
				public void run() {
					support.ortho2D(bounds);

					GL2 gl = context.getGL().getGL2();
					gl.glClearColor(0, 0, 0, 0);
					gl.glClear(GL2.GL_COLOR_BUFFER_BIT);

					gl.glEnable(GL2.GL_POINT_SPRITE);
					gl.glPointSize((float)pointSize);

					// FIXME GL_POINT_SPRITE_COORD_ORIGIN はどの属性ビットで保存されるのか不明なので、個別に保存／復帰している。
					int[] pointSpriteCoordOrigin = new int[1];
					gl.glGetIntegerv(GL2.GL_POINT_SPRITE_COORD_ORIGIN, pointSpriteCoordOrigin, 0);

					try {
						gl.glPointParameteri(GL2.GL_POINT_SPRITE_COORD_ORIGIN, GL2.GL_LOWER_LEFT);

						gl.glBegin(GL2.GL_POINTS);
						gl.glVertex2f((float)(bounds.x+bounds.width*0.5),
									  (float)(bounds.y+bounds.height*0.5));
						gl.glEnd();

					} finally {
						gl.glPointParameteri(GL2.GL_POINT_SPRITE_COORD_ORIGIN, pointSpriteCoordOrigin[0]);
					}
				}
			};

			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("iris", 0));
			uniforms.add(new GLUniformData("pointCoordAdj", (float)(pointSize/irisSize)));

			int pushAttribs = GL2.GL_COLOR_BUFFER_BIT | GL2.GL_ENABLE_BIT | GL2.GL_POINT_BIT;
			support.useShaderProgram(coverageProgram, uniforms, operation, pushAttribs, tmp, irisBuffer);

			float[] array = (float[])tmp.getArray();
			double r = 0, g = 0, b = 0;
			for (int i = 0, j = 0, n = bounds.width*bounds.height; i < n; ++i) {
				b += array[j++];
				g += array[j++];
				r += array[j++];
				j++;
			}

			return (0.299*r + 0.587*g + 0.114*b) / (irisSize*irisSize);

		} finally {
			if (tmp != null) tmp.dispose();
		}
	}

	@ShaderSource
	public static final String[] COVERAGE = {
		"#version 120",
		"",
		"uniform sampler2D iris;",
		"uniform float pointCoordAdj;",
		"",
		"void main(void)",
		"{",
		"	vec2 irisCoord = (gl_PointCoord - 0.5) * pointCoordAdj + 0.5;",
		"	gl_FragColor = texture2D(iris, irisCoord);",
		"}"
	};

}
