/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.distort;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableLayerReference;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Quality;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.ShaderType;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.IVideoBuffer.TextureWrapMode;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.DisplacementMap", category=Categories.DISTORT)
public class DisplacementMap {

	public enum UseForDisplacement {
		RED,
		GREEN,
		BLUE,
		ALPHA,
		LUMINANCE,

		HUE,
		LIGHTNESS,
		SATURATION,

//		FULL,
//		HALF,

		OFF
	}

	public enum MapPlacement { STRETCH, CENTER, TILE, MIRRORED_TILE }

	public enum WrapPixelsAround { NONE, CLAMP_TO_EDGE, REPEAT, MIRRORED_REPEAT }

	@Property
	private IAnimatableLayerReference displacementMapLayer;

	@Property("RED")
	private IAnimatableEnum<UseForDisplacement> useForHorizontalDisplacement;

	@Property(value="5", min="-32000", max="32000")
	private IAnimatableDouble maxHorizontalDisplacement;

	@Property("GREEN")
	private IAnimatableEnum<UseForDisplacement> useForVerticalDisplacement;

	@Property(value="5", min="-32000", max="32000")
	private IAnimatableDouble maxVerticalDisplacement;

	@Property
	private IAnimatableEnum<MapPlacement> mapPlacement;

	@Property
	private IAnimatableEnum<WrapPixelsAround> wrapPixelsAround;

	@Property
	private IAnimatableBoolean expandOutput;

	@Property("true")
	private IAnimatableBoolean interpolation;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderRegistry shaders;

	@Inject
	public DisplacementMap(IVideoEffectContext context,
			IVideoRenderSupport support, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		this.shaders = shaders;
	}

	private VideoBounds expandBounds(VideoBounds sourceBounds,
			double maxHorzDisplacement, double maxVertDisplacement) {

		maxHorzDisplacement = Math.abs(maxHorzDisplacement);
		maxVertDisplacement = Math.abs(maxVertDisplacement);

		double left = sourceBounds.x + (int)Math.floor(-maxHorzDisplacement);
		double top = sourceBounds.y + (int)Math.floor(-maxVertDisplacement);
		double right = sourceBounds.x + sourceBounds.width + maxHorzDisplacement;
		double bottom = sourceBounds.y + sourceBounds.height + maxVertDisplacement;

		return new VideoBounds(left, top, (int)Math.ceil(right-left), (int)Math.ceil(bottom-top));
	}

	public VideoBounds getVideoBounds() {
		VideoBounds bounds = context.getPreviousBounds();
		if (bounds.isEmpty()) {
			return bounds;
		}

		MapPlacement mapPlacement = context.value(this.mapPlacement);
		boolean expandOutput = (mapPlacement == MapPlacement.TILE || mapPlacement == MapPlacement.MIRRORED_TILE)
								? context.value(this.expandOutput) : false;
		if (expandOutput) {
			Resolution resolution = context.getVideoResolution();

			UseForDisplacement useForHorz = context.value(this.useForHorizontalDisplacement);
			UseForDisplacement useForVert = context.value(this.useForVerticalDisplacement);

			double maxHorzDisplacement = (useForHorz == UseForDisplacement.OFF)
					? 0 : resolution.scale(context.value(this.maxHorizontalDisplacement));
			double maxVertDisplacement = (useForVert == UseForDisplacement.OFF)
					? 0 : resolution.scale(context.value(this.maxVerticalDisplacement));

			return expandBounds(bounds, maxHorzDisplacement, maxVertDisplacement);

		} else {
			return bounds;
		}
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();
		VideoBounds bounds = source.getBounds();
		if (bounds.isEmpty()) {
			return source;
		}

		Resolution resolution = context.getVideoResolution();

		UseForDisplacement useForHorz = context.value(this.useForHorizontalDisplacement);
		UseForDisplacement useForVert = context.value(this.useForVerticalDisplacement);

		double maxHorzDisplacement = (useForHorz == UseForDisplacement.OFF)
				? 0 : resolution.scale(context.value(this.maxHorizontalDisplacement));
		double maxVertDisplacement = (useForVert == UseForDisplacement.OFF)
				? 0 : resolution.scale(context.value(this.maxVerticalDisplacement));

		MapPlacement mapPlacement = context.value(this.mapPlacement);
		boolean expandOutput = (mapPlacement == MapPlacement.TILE || mapPlacement == MapPlacement.MIRRORED_TILE)
								? context.value(this.expandOutput) : false;

		VideoBounds outputBounds;
		if (expandOutput) {
			outputBounds = expandBounds(bounds, maxHorzDisplacement, maxVertDisplacement);
		} else {
			outputBounds = bounds;
		}

		IVideoBuffer mapBuffer = null;
		IVideoBuffer buffer = null;
		try {
			mapBuffer = context.getLayerVideoFrame(this.displacementMapLayer);
			if (mapBuffer == null) {
				if (expandOutput) {
					// expandOutput が true の場合、getVideoBounds からは拡張したサイズが返されるので、
					// それに合わせて拡張したサイズのバッファにコピーする。
					buffer = support.createVideoBuffer(outputBounds);
					buffer.clear();
					support.copy(source, buffer);

					IVideoBuffer result = buffer;
					buffer = null;
					return result;

				} else {
					IVideoBuffer result = source;
					source = null;
					return result;
				}
			}

			buffer = support.createVideoBuffer(outputBounds);
			buffer.clear();

			WrapPixelsAround wrapPixelsAround = context.value(this.wrapPixelsAround);
			boolean interpolation = context.value(this.interpolation);
			Quality quality = context.getQuality();

			source.setTextureWrapMode(wrapPixelsAround == WrapPixelsAround.CLAMP_TO_EDGE ? TextureWrapMode.CLAMP_TO_EDGE
									: wrapPixelsAround == WrapPixelsAround.REPEAT ? TextureWrapMode.REPEAT
									: wrapPixelsAround == WrapPixelsAround.MIRRORED_REPEAT ? TextureWrapMode.MIRRORED_REPEAT
									: TextureWrapMode.CLAMP_TO_BORDER);

			source.setTextureFilter(quality == Quality.DRAFT || resolution.scale < 1 ? TextureFilter.NEAREST
									: interpolation ? TextureFilter.MIPMAP : TextureFilter.LINEAR);

			mapBuffer.setTextureFilter(quality == Quality.DRAFT || resolution.scale < 1 ? TextureFilter.NEAREST
									: interpolation ? TextureFilter.MIPMAP : TextureFilter.LINEAR);

			IShaderProgram program = getProgram(useForHorz, useForVert, mapPlacement);
			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture", 0));
			uniforms.add(new GLUniformData("mapBuffer", 1));
			uniforms.add(new GLUniformData("size0", 2, toFloatBuffer(bounds.width, bounds.height)));
			uniforms.add(new GLUniformData("maxDisplacement", 2, toFloatBuffer(maxHorzDisplacement, maxVertDisplacement)));

			if (mapPlacement == MapPlacement.TILE || mapPlacement == MapPlacement.MIRRORED_TILE) {
				uniforms.add(new GLUniformData("offset0", 2,
						toFloatBuffer(outputBounds.x-bounds.x, outputBounds.y-bounds.y)));
				mapBuffer.setTextureWrapMode(mapPlacement == MapPlacement.TILE
						? TextureWrapMode.REPEAT : TextureWrapMode.MIRRORED_REPEAT);
			}

			if (mapPlacement != MapPlacement.STRETCH) {
				VideoBounds mapBounds = mapBuffer.getBounds();
				uniforms.add(new GLUniformData("size1", 2, toFloatBuffer(mapBounds.width, mapBounds.height)));
			}

			support.useShaderProgram(program, uniforms, buffer, source, mapBuffer);

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
			if (mapBuffer != null) mapBuffer.dispose();
			if (source != null) source.dispose();
		}
	}

	private FloatBuffer toFloatBuffer(double...values) {
		float[] farray = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			farray[i] = (float)values[i];
		}
		return FloatBuffer.wrap(farray);
	}

	// TODO KeyingShadersにあるrgb2hslをどこか適切な場所に移動する。
	private static final String RGB2HSL = "ch.kuramo.javie.effects.keying.KeyingShaders.rgb2hsl";

	private IShaderProgram getProgram(
			UseForDisplacement useForHorz, UseForDisplacement useForVert, MapPlacement mapPlacement) {

		String programName = DisplacementMap.class.getName()
					+ "." + useForHorz.name() + "." + useForVert.name() + "." + mapPlacement.name();
		IShaderProgram program = shaders.getProgram(programName);
		if (program == null) {
			String[] source = createShaderSource(useForHorz, useForVert, mapPlacement);
			program = shaders.registerProgram(programName, ShaderType.FRAGMENT_SHADER, new String[] { RGB2HSL } , source);
		}
		return program;
	}

	private String[] createShaderSource(
			UseForDisplacement useForHorz, UseForDisplacement useForVert, MapPlacement mapPlacement) {

		boolean stretch = (mapPlacement == MapPlacement.STRETCH);
		boolean center = (mapPlacement == MapPlacement.CENTER);
		boolean tile = (mapPlacement == MapPlacement.TILE || mapPlacement == MapPlacement.MIRRORED_TILE);

		return new String[] {
				"#define USE_FOR_HORZ " + useForHorz.ordinal(),
				"#define USE_FOR_VERT " + useForVert.ordinal(),
	  stretch ? "#define STRETCH" : "",
	   center ? "#define CENTER" : "",
		 tile ? "#define TILE" : "",
				"",
				"uniform sampler2D texture;",
				"uniform vec2 size0;",
				"uniform sampler2D mapBuffer;",
				"uniform vec2 maxDisplacement;",
				"",
				"#ifdef TILE",
				"	uniform vec2 offset0;",
				"#endif",
				"#ifndef STRETCH",
				"	uniform vec2 size1;",
				"#endif",
				"",
				"const vec3 lumaVec = vec3(0.299, 0.587, 0.114);",
				"",
				"vec3 rgb2hsl(vec3 rgb);",
				"",
				"vec2 mapValue(vec4 mapColor)",
				"{",
				"	vec2 value = vec2(0.0);",
				"",
				"#if USE_FOR_HORZ == " + UseForDisplacement.RED.ordinal(),
				"	value.x = (mapColor.a > 0.0) ? (mapColor.r/mapColor.a*2.0-1.0)*mapColor.a : 0.0;",
				"#elif USE_FOR_HORZ == " + UseForDisplacement.GREEN.ordinal(),
				"	value.x = (mapColor.a > 0.0) ? (mapColor.g/mapColor.a*2.0-1.0)*mapColor.a : 0.0;",
				"#elif USE_FOR_HORZ == " + UseForDisplacement.BLUE.ordinal(),
				"	value.x = (mapColor.a > 0.0) ? (mapColor.b/mapColor.a*2.0-1.0)*mapColor.a : 0.0;",
				"#elif USE_FOR_HORZ == " + UseForDisplacement.ALPHA.ordinal(),
				"	value.x = mapColor.a*2.0-1.0;",
				"#elif USE_FOR_HORZ == " + UseForDisplacement.LUMINANCE.ordinal(),
				"	value.x = (mapColor.a > 0.0) ? (dot(mapColor.rgb/mapColor.a, lumaVec)*2.0-1.0)*mapColor.a : 0.0;",
				"#elif USE_FOR_HORZ == " + UseForDisplacement.HUE.ordinal(),
				"	value.x = (mapColor.a > 0.0) ? (rgb2hsl(mapColor.rgb/mapColor.a).x*2.0-1.0) : 0.0;",
				"#elif USE_FOR_HORZ == " + UseForDisplacement.LIGHTNESS.ordinal(),
				"	value.x = (mapColor.a > 0.0) ? (rgb2hsl(mapColor.rgb/mapColor.a).z*2.0-1.0)*mapColor.a : 0.0;",
				"#elif USE_FOR_HORZ == " + UseForDisplacement.SATURATION.ordinal(),
				"	value.x = (mapColor.a > 0.0) ? (rgb2hsl(mapColor.rgb/mapColor.a).y*2.0-1.0)*mapColor.a : 0.0;",
				"#endif",
				"",
				"#if USE_FOR_VERT == " + UseForDisplacement.RED.ordinal(),
				"	value.y = (mapColor.a > 0.0) ? (mapColor.r/mapColor.a*2.0-1.0)*mapColor.a : 0.0;",
				"#elif USE_FOR_VERT == " + UseForDisplacement.GREEN.ordinal(),
				"	value.y = (mapColor.a > 0.0) ? (mapColor.g/mapColor.a*2.0-1.0)*mapColor.a : 0.0;",
				"#elif USE_FOR_VERT == " + UseForDisplacement.BLUE.ordinal(),
				"	value.y = (mapColor.a > 0.0) ? (mapColor.b/mapColor.a*2.0-1.0)*mapColor.a : 0.0;",
				"#elif USE_FOR_VERT == " + UseForDisplacement.ALPHA.ordinal(),
				"	value.y = mapColor.a*2.0-1.0;",
				"#elif USE_FOR_VERT == " + UseForDisplacement.LUMINANCE.ordinal(),
				"	value.y = (mapColor.a > 0.0) ? (dot(mapColor.rgb/mapColor.a, lumaVec)*2.0-1.0)*mapColor.a : 0.0;",
				"#elif USE_FOR_VERT == " + UseForDisplacement.HUE.ordinal(),
				"	value.y = (mapColor.a > 0.0) ? (rgb2hsl(mapColor.rgb/mapColor.a).x*2.0-1.0) : 0.0;",
				"#elif USE_FOR_VERT == " + UseForDisplacement.LIGHTNESS.ordinal(),
				"	value.y = (mapColor.a > 0.0) ? (rgb2hsl(mapColor.rgb/mapColor.a).z*2.0-1.0)*mapColor.a : 0.0;",
				"#elif USE_FOR_VERT == " + UseForDisplacement.SATURATION.ordinal(),
				"	value.y = (mapColor.a > 0.0) ? (rgb2hsl(mapColor.rgb/mapColor.a).y*2.0-1.0)*mapColor.a : 0.0;",
				"#endif",
				"",
				"	return value;",
				"}",
				"",
				"void main(void)",
				"{",
				"#ifdef STRETCH",
				"	vec2 tc = gl_TexCoord[0].st;",
				"	vec4 mapColor = texture2D(mapBuffer, tc);",
				"	vec2 displacement = mapValue(mapColor) * maxDisplacement;",
				"#endif",
				"",
				"#ifdef CENTER",
				"	vec2 tc = ((gl_FragCoord.xy - 0.5*size0) / size1) + vec2(0.5);",
				"	vec4 mapColor = texture2D(mapBuffer, tc);",
				"	vec2 displacement;",
				"	if (all(greaterThanEqual(tc, vec2(0.0))) && all(lessThanEqual(tc, vec2(1.0)))) {",
				"		displacement = mapValue(mapColor) * maxDisplacement;",
				"	} else {",
				"		displacement = vec2(0.0);",
				"	}",
				"#endif",
				"",
				"#ifdef TILE",
				"	vec2 tc = ((gl_FragCoord.xy + offset0 - 0.5*size0) / size1) + vec2(0.5);",
				"	vec4 mapColor = texture2D(mapBuffer, tc);",
				"	vec2 displacement = mapValue(mapColor) * maxDisplacement;",
				"#endif",
				"",
				"#ifndef STRETCH",
				"	tc = gl_TexCoord[0].st;",
				"#endif",
				"",
				"	gl_FragColor = texture2D(texture, tc + displacement/size0);",
				"}"
		};
	}

}
