/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.distort;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Quality;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.IVideoBuffer.TextureWrapMode;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.OpticsCompensation", category=Categories.DISTORT)
public class OpticsCompensation {

	public enum Resize { OFF, MAX_2X, MAX_4X/*, UNLIMITED*/ }


	@Property(value="0", min="0", max="100")
	private IAnimatableDouble amount;

	@Property("false")
	private IAnimatableBoolean reverseLensDistortion;

	@Property
	private IAnimatableVec2d viewCenter;

	@Property("OFF")
	private IAnimatableEnum<Resize> resize;

	@Property("false")
	private IAnimatableBoolean interpolation;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram normalProgram;

	private final IShaderProgram reverseProgram;

	@Inject
	public OpticsCompensation(IVideoEffectContext context,
			IVideoRenderSupport support, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		normalProgram = shaders.getProgram(OpticsCompensation.class, "NORMAL_PROGRAM");
		reverseProgram = shaders.getProgram(OpticsCompensation.class, "REVERSE_PROGRAM");
	}

	private double toFocalDistance(double amount) {
		// この500は適当に決めた値
		return 500 / Math.tan(0.5 * amount * Math.PI);
	}

	private void transformPoint(double[] pt, double focalDistance, Vec2d center) {
		double d = Math.hypot(pt[0], pt[1]);
		if (d > 0) {
			double theta = d / focalDistance;
			theta = Math.min(Math.max(theta, -0.5*Math.PI), 0.5*Math.PI);
			double dd = focalDistance * Math.tan(theta) / d;
			pt[0] = center.x + dd*pt[0];
			pt[1] = center.y + dd*pt[1];
		} else {
			pt[0] = center.x;
			pt[1] = center.y;
		}
	}

	private VideoBounds calcResultBounds(
			VideoBounds srcBounds, Resize resize,
			double focalDistance, Vec2d center) {

		if (resize == Resize.OFF) {
			return srcBounds;
		}

		double left   = srcBounds.x;
		double top    = srcBounds.y;
		double right  = srcBounds.x + srcBounds.width;
		double bottom = srcBounds.y + srcBounds.height;

		double dLeft   = left   - center.x;
		double dTop    = top    - center.y;
		double dRight  = right  - center.x;
		double dBottom = bottom - center.y;

		double left2   = Double.POSITIVE_INFINITY;
		double top2    = Double.POSITIVE_INFINITY;
		double right2  = Double.NEGATIVE_INFINITY;
		double bottom2 = Double.NEGATIVE_INFINITY;

		double[] pt = new double[2];

		pt[0] = dLeft;
		pt[1] = dTop;
		transformPoint(pt, focalDistance, center);
		left2 = Math.min(pt[0], left2);
		top2  = Math.min(pt[1], top2);

		pt[0] = dRight;
		pt[1] = dTop;
		transformPoint(pt, focalDistance, center);
		right2 = Math.max(pt[0], right2);
		top2   = Math.min(pt[1], top2);

		pt[0] = dRight;
		pt[1] = dBottom;
		transformPoint(pt, focalDistance, center);
		right2  = Math.max(pt[0], right2);
		bottom2 = Math.max(pt[1], bottom2);

		pt[0] = dLeft;
		pt[1] = dBottom;
		transformPoint(pt, focalDistance, center);
		left2   = Math.min(pt[0], left2);
		bottom2 = Math.max(pt[1], bottom2);

		pt[0] = dLeft;
		pt[1] = 0;
		transformPoint(pt, focalDistance, center);
		left2 = Math.min(pt[0], left2);

		pt[0] = 0;
		pt[1] = dTop;
		transformPoint(pt, focalDistance, center);
		top2 = Math.min(pt[1], top2);

		pt[0] = dRight;
		pt[1] = 0;
		transformPoint(pt, focalDistance, center);
		right2 = Math.max(pt[0], right2);

		pt[0] = 0;
		pt[1] = dBottom;
		transformPoint(pt, focalDistance, center);
		bottom2 = Math.max(pt[1], bottom2);

		double max;
		switch (resize) {
			case MAX_2X: max = 2; break;
			case MAX_4X: max = 4; break;
			default: max = Double.NaN; break;
		}
		if (!Double.isNaN(max)) {
			double leftLimit, topLimit, rightLimit, bottomLimit;

			if (dLeft > 0) {
				leftLimit  = left2;
				rightLimit = left2 + max*srcBounds.width;
			} else if (dRight < 0) {
				leftLimit  = right2 - max*srcBounds.width;
				rightLimit = right2;
			} else {
				leftLimit  = center.x + max*dLeft;
				rightLimit = center.x + max*dRight;
			}
			if (dTop > 0) {
				topLimit    = top2;
				bottomLimit = top2 + max*srcBounds.height;
			} else if (dBottom < 0) {
				topLimit    = bottom2 - max*srcBounds.height;
				bottomLimit = bottom2;
			} else {
				topLimit    = center.y + max*dTop;
				bottomLimit = center.y + max*dBottom;
			}

			left2   = Math.max(left2,   leftLimit);
			top2    = Math.max(top2,    topLimit);
			right2  = Math.min(right2,  rightLimit);
			bottom2 = Math.min(bottom2, bottomLimit);
		}

		left2 = left - Math.ceil(left - left2);
		top2  = top  - Math.ceil(top - top2);

		return new VideoBounds(left2, top2,
				(int)Math.ceil(right2-left2), (int)Math.ceil(bottom2-top2));
	}

	public VideoBounds getVideoBounds() {
		VideoBounds bounds = context.getPreviousBounds();
		if (bounds.isEmpty()) {
			return bounds;
		}

		double amount = context.value(this.amount) / 100;
		if (amount == 0) {
			return bounds;
		}

		Resolution resolution = context.getVideoResolution();
		boolean reverse = context.value(this.reverseLensDistortion);
		Vec2d center = resolution.scale(context.value(this.viewCenter));
		Resize resize = reverse ? context.value(this.resize) : Resize.OFF;

		double focalDistance = toFocalDistance(amount);
		return calcResultBounds(bounds, resize, focalDistance, center);
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();
		VideoBounds bounds = source.getBounds();
		if (bounds.isEmpty()) {
			return source;
		}

		double amount = context.value(this.amount) / 100;
		if (amount == 0) {
			return source;
		}

		Resolution resolution = context.getVideoResolution();
		boolean reverse = context.value(this.reverseLensDistortion);
		Vec2d center = resolution.scale(context.value(this.viewCenter));
		Resize resize = reverse ? context.value(this.resize) : Resize.OFF;
		boolean interpolation = context.value(this.interpolation);

		TextureFilter filter = (context.getQuality() == Quality.DRAFT || resolution.scale < 1)
				? TextureFilter.NEAREST : interpolation ? TextureFilter.MIPMAP : TextureFilter.LINEAR;

		double focalDistance = toFocalDistance(amount);
		VideoBounds resultBounds = calcResultBounds(bounds, resize, focalDistance, center);

		IVideoBuffer buffer = null;
		try {
			buffer = context.createVideoBuffer(resultBounds);

			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture", 0));
			uniforms.add(new GLUniformData("cen_minus_fragOff", 2, toFloatBuffer(center.x-resultBounds.x, center.y-resultBounds.y)));
			uniforms.add(new GLUniformData("cen_minus_texOff", 2, toFloatBuffer(center.x-bounds.x, center.y-bounds.y)));
			uniforms.add(new GLUniformData("fd", (float)focalDistance));
			uniforms.add(new GLUniformData("o_div_fd", (float)(1/focalDistance)));
			uniforms.add(new GLUniformData("o_div_ts", 2, toFloatBuffer(1.0/bounds.width, 1.0/bounds.height)));

			source.setTextureWrapMode(TextureWrapMode.CLAMP_TO_BORDER);
			source.setTextureFilter(filter);

			support.useShaderProgram(reverse ? reverseProgram : normalProgram, uniforms, buffer, source);

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
			if (source != null) source.dispose();
		}
	}

	private FloatBuffer toFloatBuffer(double...values) {
		float[] farray = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			farray[i] = (float)values[i];
		}
		return FloatBuffer.wrap(farray);
	}


	@ShaderSource
	public static final String[] NORMAL_PROGRAM = {
		"uniform sampler2D texture;",
		"uniform vec2 cen_minus_fragOff;",	// center - resultBounds.xy
		"uniform vec2 cen_minus_texOff;",	// center - bounds.xy
		"uniform float fd;",				// foculDistance
		"uniform float o_div_fd;",			// 1 / foculDistance
		"uniform vec2 o_div_ts;",			// 1 / texSize
		"",
		"const float PI_0_5 = 3.14159265358979323846264 * 0.5;",
		"",
		"void main(void)",
		"{",
		"	vec2 v = gl_FragCoord.xy - cen_minus_fragOff;",
		"	float d = length(v);",
		"	vec2 xy = v/d * fd*tan(clamp(d*o_div_fd, -PI_0_5, PI_0_5)) + cen_minus_texOff;",
		"	gl_FragColor = texture2D(texture, xy*o_div_ts);",
		"}"
	};

	@ShaderSource
	public static final String[] REVERSE_PROGRAM = {
		"uniform sampler2D texture;",
		"uniform vec2 cen_minus_fragOff;",	// center - resultBounds.xy
		"uniform vec2 cen_minus_texOff;",	// center - bounds.xy
		"uniform float fd;",				// foculDistance
		"uniform float o_div_fd;",			// 1 / foculDistance
		"uniform vec2 o_div_ts;",			// 1 / texSize
		"",
		"void main(void)",
		"{",
		"	vec2 v = gl_FragCoord.xy - cen_minus_fragOff;",
		"	float d = length(v);",
		"	vec2 xy = v/d * fd*atan(d*o_div_fd) + cen_minus_texOff;",
		"	gl_FragColor = texture2D(texture, xy*o_div_ts);",
		"}"
	};

}
