/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.distort;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Quality;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.ShaderType;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.IVideoBuffer.TextureWrapMode;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.Ripple", category=Categories.DISTORT)
public class Ripple {

	public enum TypeOfConversion { ASYMMETRIC, SYMMETRIC }


	@Property(value="0", min="0"/*, max="100"*/)
	private IAnimatableDouble radius;

	@Property
	private IAnimatableVec2d centerOfRipple;

	@Property("ASYMMETRIC")
	private IAnimatableEnum<TypeOfConversion> typeOfConversion;

	@Property(value="1", min="-15", max="15")
	private IAnimatableDouble waveSpeed;

	@Property(value="20", min="2"/*, max="100"*/)
	private IAnimatableDouble waveWidth;

	@Property(value="20", min="0"/*, max="400"*/)
	private IAnimatableDouble waveHeight;

	@Property
	private IAnimatableDouble ripplePhase;

//	@Property("true")
//	private IAnimatableBoolean repeatEdgePixels;

	@Property("true")
	private IAnimatableBoolean interpolation;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderRegistry shaders;

	@Inject
	public Ripple(IVideoEffectContext context, IVideoRenderSupport support, IShaderRegistry shaders) {
		this.context = context;
		this.support = support;
		this.shaders = shaders;
	}

	private VideoBounds calcVideoBounds(
			VideoBounds srcBounds, double radius, Vec2d center,
			TypeOfConversion type, double waveHeight) {

		// TODO
		return srcBounds;
	}

	public VideoBounds getVideoBounds() {
		VideoBounds srcBounds = context.getPreviousBounds();
		if (srcBounds.isEmpty()) {
			return srcBounds;
		}

		boolean repeatEdgePixels = true; //context.value(this.repeatEdgePixels);
		if (repeatEdgePixels) {
			return srcBounds;
		}

		Resolution resolution = context.getVideoResolution();
		double radius = context.value(this.radius) / 100 * Math.max(srcBounds.width, srcBounds.height);
		Vec2d center = resolution.scale(context.value(this.centerOfRipple));
		TypeOfConversion type = context.value(this.typeOfConversion);
		double waveHeight = resolution.scale(context.value(this.waveHeight)*0.3);	// 0.3でほぼAEと同じになるが、完全には一致しない。

		return calcVideoBounds(srcBounds, radius, center, type, waveHeight);
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();
		VideoBounds srcBounds = source.getBounds();
		if (srcBounds.isEmpty()) {
			return source;
		}

		IVideoBuffer buffer = null;
		try {
			Resolution resolution = context.getVideoResolution();
			double radius = context.value(this.radius) / 100 * Math.max(srcBounds.width, srcBounds.height);
			Vec2d center = resolution.scale(context.value(this.centerOfRipple));
			TypeOfConversion type = context.value(this.typeOfConversion);
			double waveSpeed = context.value(this.waveSpeed);
			double waveWidth = resolution.scale(context.value(this.waveWidth));
			double waveHeight = resolution.scale(context.value(this.waveHeight)*0.3);	// 0.3でほぼAEと同じになるが、完全には一致しない。
			double phase = context.value(this.ripplePhase);
			boolean repeatEdgePixels = true; //context.value(this.repeatEdgePixels);
			boolean interpolation = context.value(this.interpolation);
			Time time = context.getTime();

			VideoBounds resultBounds;
			if (repeatEdgePixels) {
				resultBounds = srcBounds;
			} else {
				resultBounds = calcVideoBounds(srcBounds, radius, center, type, waveHeight);
			}

			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture", 0));
			uniforms.add(new GLUniformData("fragCoordOffset", 2, toFloatBuffer(resultBounds.x-center.x, resultBounds.y-center.y)));
			uniforms.add(new GLUniformData("texCoordOffset", 2, toFloatBuffer(srcBounds.x-center.x, srcBounds.y-center.y)));
			uniforms.add(new GLUniformData("o_div_txsz", 2, toFloatBuffer(1.0/srcBounds.width, 1.0/srcBounds.height)));
			uniforms.add(new GLUniformData("radius", (float)radius));
			uniforms.add(new GLUniformData("o_div_r", (float)(1.0/radius)));
			uniforms.add(new GLUniformData("phase", (float)Math.toRadians(phase-360*(waveSpeed*time.toSecond()))));
			uniforms.add(new GLUniformData("o_div_ww", (float)(1.0/waveWidth)));
			uniforms.add(new GLUniformData("waveHeight", (float)waveHeight));

			TextureFilter filter = (context.getQuality() == Quality.DRAFT || resolution.scale < 1)
						? TextureFilter.NEAREST : interpolation ? TextureFilter.MIPMAP : TextureFilter.LINEAR;
			source.setTextureFilter(filter);

			TextureWrapMode wrapMode = repeatEdgePixels ? TextureWrapMode.MIRRORED_REPEAT
														: TextureWrapMode.CLAMP_TO_BORDER;
			source.setTextureWrapMode(wrapMode);

			buffer = context.createVideoBuffer(resultBounds);

			support.useShaderProgram(getProgram(type), uniforms, buffer, source);

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
			if (source != null) source.dispose();
		}
	}

	private FloatBuffer toFloatBuffer(double...values) {
		float[] farray = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			farray[i] = (float)values[i];
		}
		return FloatBuffer.wrap(farray);
	}

	private IShaderProgram getProgram(TypeOfConversion type) {
		String programName = Ripple.class.getName() + "." + type.name();
		IShaderProgram program = shaders.getProgram(programName);
		if (program == null) {
			program = shaders.registerProgram(programName,
					ShaderType.FRAGMENT_SHADER, null, createProgramSource(type));
		}
		return program;
	}

	private String[] createProgramSource(TypeOfConversion type) {
		return new String[] {
				"#define " + type.name(),
				"",
				"uniform sampler2D texture;",
				"uniform vec2 fragCoordOffset;",	// resultBounds.xy - center
				"uniform vec2 texCoordOffset;",		// srcBounds.xy - center
				"uniform vec2 o_div_txsz;",			// 1.0/texSize
				"uniform float radius;",
				"uniform float o_div_r;",			// 1.0/radius
				"uniform float phase;",
				"uniform float o_div_ww;",			// 1.0/waveWidth
				"uniform float waveHeight;",
				"",
				"const float PI = 3.14159265358979323846264;",
				"",
				"void main(void)",
				"{",
				"	vec2 v = gl_FragCoord.xy + fragCoordOffset;",
				"	float d = length(v);",
				"",
				"	v += max(0.0, (radius-d)*o_div_r) * waveHeight * cos(PI*d*o_div_ww + phase)",
				"#ifdef SYMMETRIC",
				"		* (d > 0.0 ? v/d : vec2(0.0))",
				"#endif",
				"		;",
				"",
				"	vec2 texCoord = (v - texCoordOffset) * o_div_txsz;",
				"	gl_FragColor = texture2D(texture, texCoord);",
				"}"
		};
	}

}
