/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.generate;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.BlendMode;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IBlendSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.FourColorGradient", category=Categories.GENERATE)
public class FourColorGradient {

	@Property
	private IAnimatableVec2d point1;

	@Property("1,1,0")
	private IAnimatableColor color1;

	@Property
	private IAnimatableVec2d point2;

	@Property("0,1,0")
	private IAnimatableColor color2;

	@Property
	private IAnimatableVec2d point3;

	@Property("1,0,1")
	private IAnimatableColor color3;

	@Property
	private IAnimatableVec2d point4;

	@Property("0,0,1")
	private IAnimatableColor color4;

	@Property(value="100", min="0"/*, max="10000"*/)
	private IAnimatableDouble blend;

//	@Property(value="0", min="0", max="100")
//	private IAnimatableDouble jitter;

	@Property(value="100", min="0", max="100")
	private IAnimatableDouble opacity;

	@Property
	private IAnimatableEnum<BlendMode> blendMode;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IBlendSupport blendSupport;

	private final IShaderProgram program;

	@Inject
	public FourColorGradient(
			IVideoEffectContext context, IVideoRenderSupport support,
			IBlendSupport blendSupport, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		this.blendSupport = blendSupport;
		program = shaders.getProgram(FourColorGradient.class, "FOUR_COLOR_GRADIENT");
		blendSupport.setProgramsClass(blendSupport.getPremultAndMatteClass(true));
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer original = context.doPreviousEffect();
		VideoBounds bounds = original.getBounds();
		if (bounds.isEmpty()) {
			return original;
		}

		IVideoBuffer buffer = null;
		try {
			Resolution resolution = context.getVideoResolution();
			Vec2d point1 = resolution.scale(context.value(this.point1));
			Color color1 = context.value(this.color1);
			Vec2d point2 = resolution.scale(context.value(this.point2));
			Color color2 = context.value(this.color2);
			Vec2d point3 = resolution.scale(context.value(this.point3));
			Color color3 = context.value(this.color3);
			Vec2d point4 = resolution.scale(context.value(this.point4));
			Color color4 = context.value(this.color4);
			double blend = context.value(this.blend);
			double opacity = context.value(this.opacity) / 100;
			BlendMode blendMode = context.value(this.blendMode);

			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("coordOffset", 2, toFloatBuffer(bounds.x, bounds.y)));
			uniforms.add(new GLUniformData("point1", 2, toFloatBuffer(point1.x, point1.y)));
			uniforms.add(new GLUniformData("color1", 4, toFloatBuffer(color1.r, color1.g, color1.b, 1)));
			uniforms.add(new GLUniformData("point2", 2, toFloatBuffer(point2.x, point2.y)));
			uniforms.add(new GLUniformData("color2", 4, toFloatBuffer(color2.r, color2.g, color2.b, 1)));
			uniforms.add(new GLUniformData("point3", 2, toFloatBuffer(point3.x, point3.y)));
			uniforms.add(new GLUniformData("color3", 4, toFloatBuffer(color3.r, color3.g, color3.b, 1)));
			uniforms.add(new GLUniformData("point4", 2, toFloatBuffer(point4.x, point4.y)));
			uniforms.add(new GLUniformData("color4", 4, toFloatBuffer(color4.r, color4.g, color4.b, 1)));
			uniforms.add(new GLUniformData("blend", (float)(blend*blend)));
			uniforms.add(new GLUniformData("rscale", (float)(1/resolution.scale)));

			buffer = context.createVideoBuffer(bounds);
			support.useShaderProgram(program, uniforms, buffer);

			return blendSupport.blend(buffer, original, null, blendMode, opacity, context);

		} finally {
			if (buffer != null) buffer.dispose();
			if (original != null) original.dispose();
		}
	}

	private FloatBuffer toFloatBuffer(double...values) {
		float[] farray = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			farray[i] = (float)values[i];
		}
		return FloatBuffer.wrap(farray);
	}

	@ShaderSource
	public static final String[] FOUR_COLOR_GRADIENT = {
		"uniform vec2 coordOffset;",
		"uniform vec2 point1;",
		"uniform vec4 color1;",
		"uniform vec2 point2;",
		"uniform vec4 color2;",
		"uniform vec2 point3;",
		"uniform vec4 color3;",
		"uniform vec2 point4;",
		"uniform vec4 color4;",
		"uniform float blend;",
		"uniform float rscale;",
		"",
		"void main(void)",
		"{",
		"	vec2 fc = gl_FragCoord.xy + coordOffset;",
		"",
		"	float d1 = distance(fc, point1) * rscale;",
		"	float d2 = distance(fc, point2) * rscale;",
		"	float d3 = distance(fc, point3) * rscale;",
		"	float d4 = distance(fc, point4) * rscale;",
		"",
		"	d1 = 1.0 / (d1 * d1 + blend);",
		"	d2 = 1.0 / (d2 * d2 + blend);",
		"	d3 = 1.0 / (d3 * d3 + blend);",
		"	d4 = 1.0 / (d4 * d4 + blend);",
		"",
		"	gl_FragColor = (color1*d1 + color2*d2 + color3*d3 + color4*d4) / (d1 + d2 + d3 + d4);",
		"}"
	};

}
