/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.keying;

import java.util.HashSet;
import java.util.Set;

import ch.kuramo.javie.api.BlendMode;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.services.IAntiAliasSupport;
import ch.kuramo.javie.api.services.IBlendSupport;
import ch.kuramo.javie.api.services.IBlurSupport;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.api.services.IBlurSupport.BlurDimensions;

public abstract class GarbageMatteBase {

	protected final IVideoEffectContext context;

	protected final IVideoRenderSupport support;

	protected final IAntiAliasSupport aaSupport;

	protected final IBlurSupport blurSupport;

	protected final IBlendSupport blendSupport;

	protected GarbageMatteBase(
			IVideoEffectContext context, IVideoRenderSupport support,
			IAntiAliasSupport aaSupport, IBlurSupport blurSupport, IBlendSupport blendSupport) {

		this.context = context;
		this.support = support;
		this.aaSupport = aaSupport;
		this.blurSupport = blurSupport;
		this.blendSupport = blendSupport;
	}

	protected IVideoBuffer doMask(boolean invert, double feather, Vec2d ... points) {
		final IVideoBuffer input = context.doPreviousEffect();
		VideoBounds bounds = input.getBounds();
		if (bounds.isEmpty()) {
			return input;
		}

		final int w = bounds.width;
		final int h = bounds.height;

		final double[][][] contours = new double[invert ? 2 : 1][][];

		contours[0] = new double[points.length][];
		for (int i = 0, n = points.length; i < n; ++i) {
			contours[0][i] = new double[] { points[i].x, points[i].y };
		}
		if (invert) {
			double x = bounds.x;
			double y = bounds.y;
			contours[1] = new double[][] { {x,y}, {x+w,y}, {x+w,y+h}, {x,y+h} };
		}


		Set<IVideoBuffer> tmpBuffers = new HashSet<IVideoBuffer>();
		tmpBuffers.add(input);

		try {
			final IVideoBuffer buffer = context.createVideoBuffer(bounds);
			tmpBuffers.add(buffer);

			Runnable operation = new Runnable() {
				public void run() {
					aaSupport.antiAlias(w, h, new Runnable() {
						public void run() {
							support.ortho2D(buffer);
							support.polygon2D(contours, buffer, input);
						}
					});
				}
			};

			if (feather == 0) {
				support.useFramebuffer(operation, 0, buffer, input);

				tmpBuffers.remove(buffer);
				return buffer;

			} else {
				support.useFramebuffer(operation, 0, buffer);

				IVideoBuffer buffer2 = blurSupport.gaussianBlur(buffer, feather, BlurDimensions.BOTH, true, true);
				tmpBuffers.add(buffer2);

				return blendSupport.blend(buffer2, input, null, BlendMode.STENCIL_ALPHA, 1.0);
			}

		} finally {
			for (IVideoBuffer vb : tmpBuffers) {
				vb.dispose();
			}
		}
	}

}
