/*
 * Copyright (c) 2010,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.simulation;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Random;
import java.util.Set;

import javax.media.opengl.GL2;
import javax.media.opengl.GLUniformData;
import javax.vecmath.Vector3d;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IAnimatableVec2d;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.ICamera;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.ShaderType;
import ch.kuramo.javie.api.Size2i;
import ch.kuramo.javie.api.Vec2d;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.IVideoBuffer.TextureFilter;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IAntiAliasSupport;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVBOCache;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.api.services.IVBOCache.VBOCacheRecord;

import com.google.inject.Inject;
import com.iabcinc.mapping.voronoi.Point;
import com.iabcinc.mapping.voronoi.QuadEdge;
import com.iabcinc.mapping.voronoi.Triangulation;
import com.iabcinc.mapping.voronoi.VoronoiPolygon;

@Effect(id="ch.kuramo.javie.Shatter", category=Categories.SIMULATION)
public class Shatter {

	public enum Pattern {
		RECTANGLE,
		BRICK,
		HEXAGON,
		OCTAGON_AND_DIAMOND,
		POLYGON_UNIFORMLY,
		POLYGON_RADIALLY
	}

	public enum TimeBase {
		LOCAL_TIME,
		COMPOSITION_TIME
	}

	public enum CameraSystem {
		LOCAL_CAMERA,
		COMPOSITION_CAMERA
	}


	@Property("RECTANGLE")
	private IAnimatableEnum<Pattern> pattern;

	@Property
	private IAnimatableDouble patternDirection;

	@Property
	private IAnimatableVec2d patternOrigin;

	@Property(value="30,30", min="1")
	private IAnimatableVec2d fragmentSize;


	@Property(value="0.1", min="0", max="1")
	private IAnimatableDouble randomness;

	@Property(value="3", min="0")
	private IAnimatableDouble rotationSpeed;

	@Property(value="3", min="0")
	private IAnimatableDouble gravity;

	@Property("180")
	private IAnimatableDouble gravityDirection;

	@Property(value="0", min="-90", max="90")
	private IAnimatableDouble gravityInclination;


	@Property("COMPOSITION_TIME")
	private IAnimatableEnum<TimeBase> timeBase;

	@Property
	private IAnimatableDouble time;


//	@Property("true")
//	private IAnimatableBoolean force1Enabled;

	@Property
	private IAnimatableDouble force1Time;

	@Property
	private IAnimatableVec2d force1Position;

	@Property(value="200", min="0")
	private IAnimatableDouble force1Radius;

	@Property(value="0.5", min="0")
	private IAnimatableDouble force1Delay;

	@Property("300")
	private IAnimatableDouble force1Strength;

	@Property
	private IAnimatableDouble force1Decay;

	@Property(value="45", min="0", max="90")
	private IAnimatableDouble force1Spread;


//	@Property("false")
//	private IAnimatableBoolean force2Enabled;
//
//	@Property
//	private IAnimatableDouble force2Time;
//
//	@Property
//	private IAnimatableVec2d force2Position;
//
//	@Property(value="200", min="0")
//	private IAnimatableDouble force2Radius;
//
//	@Property(value="0.5", min="0")
//	private IAnimatableDouble force2Delay;
//
//	@Property("300")
//	private IAnimatableDouble force2Strength;
//
//	@Property
//	private IAnimatableDouble force2Decay;
//
//	@Property(value="45", min="0", max="90")
//	private IAnimatableDouble force2Spread;
//
//
//	@Property("false")
//	private IAnimatableBoolean force3Enabled;
//
//	@Property
//	private IAnimatableDouble force3Time;
//
//	@Property
//	private IAnimatableVec2d force3Position;
//
//	@Property(value="200", min="0")
//	private IAnimatableDouble force3Radius;
//
//	@Property(value="0.5", min="0")
//	private IAnimatableDouble force3Delay;
//
//	@Property("300")
//	private IAnimatableDouble force3Strength;
//
//	@Property
//	private IAnimatableDouble force3Decay;
//
//	@Property(value="45", min="0", max="90")
//	private IAnimatableDouble force3Spread;


	@Property
	private IAnimatableEnum<CameraSystem> cameraSystem;

	@Property
	private IAnimatableDouble cameraRotationX;

	@Property
	private IAnimatableDouble cameraRotationY;

	@Property
	private IAnimatableDouble cameraRotationZ;

	@Property
	private IAnimatableVec2d cameraPositionXY;

	@Property
	private IAnimatableDouble cameraPositionZ;

	@Property(value="100", min="0")
	private IAnimatableDouble cameraZoom;

	@Property(value="10", min="0")
	private IAnimatableDouble cameraNear;

	@Property(value="100000", min="0")
	private IAnimatableDouble cameraFar;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IAntiAliasSupport aaSupport;

	private final IArrayPools arrayPools;

	private final IVBOCache vboCache;

	private final IShaderProgram program;

	@Inject
	public Shatter(
			IVideoEffectContext context, IVideoRenderSupport support,
			IAntiAliasSupport aaSupport, IArrayPools arrayPools,
			IVBOCache vboCache, IShaderRegistry shaders) {

		this.context = context;
		this.support = support;
		this.aaSupport = aaSupport;
		this.arrayPools = arrayPools;
		this.vboCache = vboCache;
		program = shaders.getProgram(Shatter.class, "SHATTER");
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer input = context.doPreviousEffect();

		VideoBounds bounds = input.getBounds();
		if (bounds.isEmpty()) {
			return input;
		}

		Resolution resolution = context.getVideoResolution();
		double patDir = context.value(this.patternDirection);
		Vec2d patOrigin = resolution.scale(context.value(this.patternOrigin));
		Vec2d fragSize = resolution.scale(context.value(this.fragmentSize));

		patDir %= 360;
		if (patDir < 0) {
			patDir += 360;
		}

		double patDirRad = Math.toRadians(patDir);

		double[] b2 = rotate(bounds, patOrigin, -patDirRad);
		int i1 = (int)Math.ceil((b2[0]-(patOrigin.x+fragSize.x/2))/fragSize.x);
		int j1 = (int)Math.ceil((b2[1]-(patOrigin.y+fragSize.y/2))/fragSize.y);
		int i2 = (int)Math.ceil((b2[2]-(patOrigin.x+fragSize.x/2))/fragSize.x);
		int j2 = (int)Math.ceil((b2[3]-(patOrigin.y+fragSize.y/2))/fragSize.y);

		int[] vboIdAndNumVerts;
		int primitiveType;
		try {
			switch (context.value(pattern)) {
				case RECTANGLE:
					vboIdAndNumVerts = rectangle(fragSize, patOrigin, i1, j1, i2, j2);
					primitiveType = GL2.GL_QUADS;
					break;
				case BRICK:
					vboIdAndNumVerts = brickOrHexagon(true, fragSize, patOrigin, i1, j1, i2, j2);
					primitiveType = GL2.GL_TRIANGLES;
					break;
				case HEXAGON:
					vboIdAndNumVerts = brickOrHexagon(false, fragSize, patOrigin, i1, j1, i2, j2);
					primitiveType = GL2.GL_TRIANGLES;
					break;
				case OCTAGON_AND_DIAMOND:
					vboIdAndNumVerts = octagonAndDiamond(fragSize, patOrigin, i1, j1, i2, j2);
					primitiveType = GL2.GL_TRIANGLES;
					break;
				case POLYGON_UNIFORMLY:
					vboIdAndNumVerts = polygon(false, fragSize, patOrigin, i1, j1, i2, j2);
					primitiveType = GL2.GL_TRIANGLES;
					break;
				case POLYGON_RADIALLY:
					vboIdAndNumVerts = polygon(true, fragSize, patOrigin, i1, j1, i2, j2);
					primitiveType = GL2.GL_TRIANGLES;
					break;
				default:
					throw new RuntimeException("unknown Pattern: " + pattern);
			}

			// TODO ミップマップ／異方性フィルタ／スーパーサンプリング
			input.setTextureFilter(TextureFilter.LINEAR);

			return doShatter(input, vboIdAndNumVerts[0], vboIdAndNumVerts[1], primitiveType, patOrigin, patDirRad);

		} finally {
			input.dispose();
		}
	}

	private int[] getVBO(String token) {
		VBOCacheRecord rec = vboCache.get(this);
		if (rec != null) {
			Object[] data = (Object[]) rec.data;
			if (token.equals(data[1])) {
				return new int[] { rec.vboId, (Integer)data[0] };
			}
		}
		return null;
	}

	private int[] createVBO(IArray<float[]> data, int numVertices, String token) {
		GL2 gl = context.getGL().getGL2();
		int[] vboId = new int[1];
		try {
			gl.glGenBuffers(1, vboId, 0);
			gl.glBindBuffer(GL2.GL_ARRAY_BUFFER, vboId[0]);
			gl.glBufferData(GL2.GL_ARRAY_BUFFER, data.getLength()*4,
					FloatBuffer.wrap(data.getArray()), GL2.GL_STATIC_DRAW);

			vboCache.put(this, new VBOCacheRecord(vboId[0], new Object[] { numVertices, token }));

			int result = vboId[0];
			vboId = null;
			return new int[] { result, numVertices };

		} finally {
			gl.glBindBuffer(GL2.GL_ARRAY_BUFFER, 0);
			if (vboId != null && vboId[0] != 0) gl.glDeleteBuffers(1, vboId, 0);
		}
	}

	private int[] rectangle(Vec2d size, Vec2d origin, int i1, int j1, int i2, int j2) {

		String token = String.format("%s/%d,%d/%d,%d/%d,%d,%d,%d",
				Pattern.RECTANGLE.name(),
				Double.doubleToLongBits(size.x), Double.doubleToLongBits(size.y),
				Double.doubleToLongBits(origin.x), Double.doubleToLongBits(origin.y),
				i1, j1, i2, j2);

		int[] vboIdAndNumVerts = getVBO(token);
		if (vboIdAndNumVerts != null) {
			return vboIdAndNumVerts;
		}


		double halfX = size.x/2;
		double halfY = size.y/2;

		// 四角形の頂点の座標 (四角形の中心からの相対座標)
		float[][] pt = new float[][] {
				{ (float)-halfX, (float)-halfY },
				{ (float) halfX, (float)-halfY },
				{ (float) halfX, (float) halfY },
				{ (float)-halfX, (float) halfY }
		};

		IArray<float[]> data = null;
		try {
			int numFragments = (i2-i1+1)*(j2-j1+1);
			int numVertices = numFragments*4;		// 頂点数 ＝ 四角形の数 x 四角形1つあたりの頂点数(4)

			// 配列の要素数 ＝ 頂点数 x 頂点あたりの座標数(4)
			// 頂点あたりの座標数は、頂点の座標XとYの2つ、四角形の中心の座標XとYの計4つ
			data = arrayPools.getFloatArray(numVertices*4);
			float[] array = data.getArray();

			for (int j = j1, k = 0; j <= j2; ++j) {
				for (int i = i1; i <= i2; ++i) {
					float x = (float)(origin.x + size.x*i);
					float y = (float)(origin.y + size.y*j);

					for (int h = 0; h < 4; ++h) {
						array[k++] = pt[h][0];
						array[k++] = pt[h][1];
						array[k++] = x;
						array[k++] = y;
					}
				}
			}

			return createVBO(data, numVertices, token);

		} finally {
			if (data != null) data.release();
		}
	}

	private int[] brickOrHexagon(boolean brick, Vec2d size, Vec2d origin, int i1, int j1, int i2, int j2) {

		++i2;
		if (!brick) {
			--j1; 
			++j2;
		}

		String token = String.format("%s/%d,%d/%d,%d/%d,%d,%d,%d",
				brick ? Pattern.BRICK.name() : Pattern.HEXAGON.name(),
				Double.doubleToLongBits(size.x), Double.doubleToLongBits(size.y),
				Double.doubleToLongBits(origin.x), Double.doubleToLongBits(origin.y),
				i1, j1, i2, j2);

		int[] vboIdAndNumVerts = getVBO(token);
		if (vboIdAndNumVerts != null) {
			return vboIdAndNumVerts;
		}


		double halfX = size.x/2;
		float[][] pt;

		if (brick) {
			double halfY = size.y/2;

			// レンガの頂点の座標 (レンガの中心からの相対座標)
			pt = new float[][] {
					{ (float)-halfX, (float)-halfY },
					{             0, (float)-halfY },
					{ (float) halfX, (float)-halfY },
					{ (float) halfX, (float) halfY },
					{             0, (float) halfY },
					{ (float)-halfX, (float) halfY },
					{ (float)-halfX, (float)-halfY }
			};
		} else {
			double sizeY13 = size.y*1/3;
			double sizeY23 = size.y*2/3;

			// 六角形の頂点の座標 (六角形の中心からの相対座標)
			pt = new float[][] {
					{             0, (float)-sizeY23 },
					{ (float) halfX, (float)-sizeY13 },
					{ (float) halfX, (float) sizeY13 },
					{             0, (float) sizeY23 },
					{ (float)-halfX, (float) sizeY13 },
					{ (float)-halfX, (float)-sizeY13 },
					{             0, (float)-sizeY23 }
			};
		}

		IArray<float[]> data = null;
		try {
			int numFragments = (i2-i1+1)*(j2-j1+1);
			int numVertices = numFragments*6*3;		// 頂点数 ＝ 破片の数 x 破片1つあたりの頂点数(6) x 3(GL_TRIANGLESを使用するため)

			// 配列の要素数 ＝ 頂点数 x 頂点あたりの座標数(4)
			// 頂点あたりの座標数は、頂点の座標XとYの2つ、破片の中心の座標XとYの計4つ
			data = arrayPools.getFloatArray(numVertices*4);
			float[] array = data.getArray();

			for (int j = j1, k = 0; j <= j2; ++j) {
				double offsetX = (j%2) != 0 ? -halfX : 0;

				for (int i = i1; i <= i2; ++i) {
					float x = (float)(origin.x + size.x*i + offsetX);
					float y = (float)(origin.y + size.y*j);

					for (int h = 0; h < 6; ++h) {
						array[k++] = 0;
						array[k++] = 0;
						array[k++] = x;
						array[k++] = y;

						array[k++] = pt[h][0];
						array[k++] = pt[h][1];
						array[k++] = x;
						array[k++] = y;

						array[k++] = pt[h+1][0];
						array[k++] = pt[h+1][1];
						array[k++] = x;
						array[k++] = y;
					}
				}
			}

			return createVBO(data, numVertices, token);

		} finally {
			if (data != null) data.release();
		}
	}

	private int[] octagonAndDiamond(Vec2d size, Vec2d origin, int i1, int j1, int i2, int j2) {
		// 右端と下端それぞれに１つずつ増やす必要があるのは、厳密には菱形の方だけ。
		// 処理を分けるのも面倒なので八角形も増やしてしまっている。
		++i2;
		++j2;

		String token = String.format("%s/%d,%d/%d,%d/%d,%d,%d,%d",
				Pattern.OCTAGON_AND_DIAMOND.name(),
				Double.doubleToLongBits(size.x), Double.doubleToLongBits(size.y),
				Double.doubleToLongBits(origin.x), Double.doubleToLongBits(origin.y),
				i1, j1, i2, j2);

		int[] vboIdAndNumVerts = getVBO(token);
		if (vboIdAndNumVerts != null) {
			return vboIdAndNumVerts;
		}


		double theta = Math.atan2(size.y, size.x);
		double cos = Math.cos(theta);
		double sin = Math.sin(theta);
		double p = size.x / (1 + 2*cos);
		double q = size.y / (1 + 2*sin);
		double pcos = p*cos;
		double qsin = q*sin;

		float[][][] pt = new float[][][] {
				{	// 八角形の頂点の座標 (八角形の中心からの相対座標)
					{ (float)-(p/2)     , (float)-(q/2+qsin) },
					{ (float) (p/2)     , (float)-(q/2+qsin) },
					{ (float) (p/2+pcos), (float)-(q/2)      },
					{ (float) (p/2+pcos), (float) (q/2)      },
					{ (float) (p/2)     , (float) (q/2+qsin) },
					{ (float)-(p/2)     , (float) (q/2+qsin) },
					{ (float)-(p/2+pcos), (float) (q/2)      },
					{ (float)-(p/2+pcos), (float)-(q/2)      },
					{ (float)-(p/2)     , (float)-(q/2+qsin) }
				},
				{	// 菱形の頂点の座標 (菱形の中心からの相対座標)
					{ 0, (float)-qsin },
					{ (float) pcos, 0 },
					{ 0, (float) qsin },
					{ (float)-pcos, 0 },
					{ 0, (float)-qsin }
				}
		};

		IArray<float[]> data = null;
		try {
			int numFragments = (i2-i1+1)*(j2-j1+1);
			int numVertices = numFragments*(8+4)*3;		// 頂点数 ＝ 破片の数 x 破片1つあたりの頂点数(8および4) x 3(GL_TRIANGLESを使用するため)

			// 配列の要素数 ＝ 頂点数 x 頂点あたりの座標数(4)
			// 頂点あたりの座標数は、頂点の座標XとYの2つ、破片の中心の座標XとYの計4つ
			data = arrayPools.getFloatArray(numVertices*4);
			float[] array = data.getArray();

			double halfX = size.x/2;
			double halfY = size.y/2;

			for (int j = j1, k = 0; j <= j2; ++j) {
				for (int i = i1; i <= i2; ++i) {
					float x = (float)(origin.x + size.x*i);
					float y = (float)(origin.y + size.y*j);

					for (int h = 0; h < 8; ++h) {
						array[k++] = 0;
						array[k++] = 0;
						array[k++] = x;
						array[k++] = y;

						array[k++] = pt[0][h][0];
						array[k++] = pt[0][h][1];
						array[k++] = x;
						array[k++] = y;

						array[k++] = pt[0][h+1][0];
						array[k++] = pt[0][h+1][1];
						array[k++] = x;
						array[k++] = y;
					}

					x -= halfX;
					y -= halfY;

					for (int h = 0; h < 4; ++h) {
						array[k++] = 0;
						array[k++] = 0;
						array[k++] = x;
						array[k++] = y;

						array[k++] = pt[1][h][0];
						array[k++] = pt[1][h][1];
						array[k++] = x;
						array[k++] = y;

						array[k++] = pt[1][h+1][0];
						array[k++] = pt[1][h+1][1];
						array[k++] = x;
						array[k++] = y;
					}
				}
			}

			return createVBO(data, numVertices, token);

		} finally {
			if (data != null) data.release();
		}
	}

	private int[] polygon(boolean radially, Vec2d size, Vec2d origin, int i1, int j1, int i2, int j2) {

		String token = String.format("%s/%d,%d/%d,%d/%d,%d,%d,%d",
				radially ? Pattern.POLYGON_RADIALLY.name() : Pattern.POLYGON_UNIFORMLY.name(),
				Double.doubleToLongBits(size.x), Double.doubleToLongBits(size.y),
				Double.doubleToLongBits(origin.x), Double.doubleToLongBits(origin.y),
				i1, j1, i2, j2);

		int[] vboIdAndNumVerts = getVBO(token);
		if (vboIdAndNumVerts != null) {
			return vboIdAndNumVerts;
		}


		Triangulation tri = new Triangulation(
				new Point(origin.x+size.x*(i1-3), origin.y+size.y*(j1-3)),
				new Point(origin.x+size.x*(i2+3), origin.y+size.y*(j2+3)));

		Random rand = new Random(context.getEffectName().hashCode());

		if (radially) {
			int n = (i2-i1+1) * (j2-j1+1);
			double radius = Math.max(size.x*(i2-i1), size.y*(j2-j1));
			double a = 0, b = 15, c = 0;
			for (int i = 0; i < n; ++i) {
				a += 2 * Math.PI / b;
				double k = Math.floor(a / (2*Math.PI)) + 0.5;
				double t = k * b / n;
				double r = t * t * radius * (1+rand.nextGaussian()/10);
				double x = origin.x + r*Math.cos(a);
				double y = origin.y + r*Math.sin(a);
				tri.insertSite(x, y);
				if ((c += 1/b) >= 1) { ++b; c = 0; }
			}
		} else {
			for (int j = j1; j <= j2; ++j) {
				for (int i = i1; i <= i2; ++i) {
					double x = origin.x + size.x*(i+rand.nextDouble()*2-1);
					double y = origin.y + size.y*(j+rand.nextDouble()*2-1);
					tri.insertSite(x, y);
				}
			}
		}

		for (int i = i1-1, n = i2+1; i <= n; ++i) {
			double x = origin.x + size.x*i;
			tri.insertSite(x, origin.y + size.y*(j1-0.5-rand.nextDouble()));
			tri.insertSite(x, origin.y + size.y*(j1-0.5-rand.nextDouble()));
			tri.insertSite(x, origin.y + size.y*(j2+0.5+rand.nextDouble()));
			tri.insertSite(x, origin.y + size.y*(j2+0.5+rand.nextDouble()));
		}
		for (int j = j1-1, n = j2+1; j <= n; ++j) {
			double y = origin.y + size.y*j;
			tri.insertSite(origin.x + size.x*(i1-0.5-rand.nextDouble()), y);
			tri.insertSite(origin.x + size.x*(i1-0.5-rand.nextDouble()), y);
			tri.insertSite(origin.x + size.x*(i2+0.5+rand.nextDouble()), y);
			tri.insertSite(origin.x + size.x*(i2+0.5+rand.nextDouble()), y);
		}

		List<VoronoiPolygon> polygons = tri.getVoronoiPolygons();

		int numVertices = 0;
		for (VoronoiPolygon p : polygons) {
			numVertices += p.edges().size() - 2;
		}
		numVertices *= 3;

		IArray<float[]> data = null;
		try {
			data = arrayPools.getFloatArray(numVertices*4);
			float[] array = data.getArray();

			int k = 0;
			for (VoronoiPolygon p : polygons) {
				Iterator<QuadEdge> it = p.edges().iterator();
				if (!it.hasNext()) continue;

				Point p0 = it.next().getOrg();

				double gx = 0, gy = 0, ssum = 0;
				while (it.hasNext()) {
					QuadEdge qe = it.next();
					if (!it.hasNext()) break;	// 最後の辺は処理の必要が無いので。

					Point p1 = qe.getOrg();
					Point p2 = qe.getDest();

					Vector3d cross = new Vector3d();
					cross.cross(new Vector3d(p1.getX(), p1.getY(), 0),
								new Vector3d(p2.getX(), p2.getY(), 0));
					double s = cross.length() / 2;

					gx += s * (p0.getX() + p1.getX() + p2.getX()) / 3;
					gy += s * (p0.getY() + p1.getY() + p2.getY()) / 3;
					ssum += s;
				}
				gx /= ssum;
				gy /= ssum;

				it = p.edges().iterator();
				it.next();
				while (it.hasNext()) {
					QuadEdge qe = it.next();
					if (!it.hasNext()) break;

					Point p1 = qe.getOrg();
					Point p2 = qe.getDest();

					array[k++] = (float)(p0.getX() - gx);
					array[k++] = (float)(p0.getY() - gy);
					array[k++] = (float)gx;
					array[k++] = (float)gy;

					array[k++] = (float)(p1.getX() - gx);
					array[k++] = (float)(p1.getY() - gy);
					array[k++] = (float)gx;
					array[k++] = (float)gy;

					array[k++] = (float)(p2.getX() - gx);
					array[k++] = (float)(p2.getY() - gy);
					array[k++] = (float)gx;
					array[k++] = (float)gy;
				}
			}

			return createVBO(data, numVertices, token);

		} finally {
			if (data != null) data.release();
		}
	}

	private double[] rotate(VideoBounds bounds, Vec2d origin, double rad) {
		double[][] pt = new double[][] {
				{ bounds.x             , bounds.y               },
				{ bounds.x+bounds.width, bounds.y               },
				{ bounds.x+bounds.width, bounds.y+bounds.height },
				{ bounds.x             , bounds.y+bounds.height },
		};

		double cos = Math.cos(rad);
		double sin = Math.sin(rad);

		double[] result = new double[] {
			Double.POSITIVE_INFINITY, Double.POSITIVE_INFINITY,
			Double.NEGATIVE_INFINITY, Double.NEGATIVE_INFINITY
		};

		for (int i = 0; i < 4; ++i) {
			rotate(pt[i], origin, cos, sin);
			result[0] = Math.min(pt[i][0], result[0]);
			result[1] = Math.min(pt[i][1], result[1]);
			result[2] = Math.max(pt[i][0], result[2]);
			result[3] = Math.max(pt[i][1], result[3]);
		}

		return result;
	}

	private void rotate(double[] pt, Vec2d origin, double cos, double sin) {
		double x = pt[0], y = pt[1];
		double x0 = origin.x, y0 = origin.y;
		pt[0] = (x-x0)*cos - (y-y0)*sin + x0;
		pt[1] = (x-x0)*sin + (y-y0)*cos + y0;
	}

	private IVideoBuffer doShatter(
			IVideoBuffer input, final int vboId, final int numVertices,
			final int primitiveType, Vec2d patOrigin, double patDirRad) {

		final VideoBounds bounds = input.getBounds();
		final int[] viewport = new int[4];
		final double[] prjMatrix = new double[16];
		final double[] mvMatrix = new double[16];
		calcCameraParameters(bounds, viewport, prjMatrix, mvMatrix);

		IVideoBuffer buffer = null;
		try {
			Runnable operation = new Runnable() {
				public void run() {
					final GL2 gl = context.getGL().getGL2();

					gl.glViewport(viewport[0], viewport[1], viewport[2], viewport[3]);
					gl.glMatrixMode(GL2.GL_PROJECTION);
					gl.glLoadMatrixd(prjMatrix, 0);
					gl.glMatrixMode(GL2.GL_MODELVIEW);
					gl.glLoadMatrixd(mvMatrix, 0);

					int vertOffsetLoc = program.getAttributeLocation("vertOffset");
					int fragCenterLoc = program.getAttributeLocation("fragCenter");
					try {
						gl.glEnableVertexAttribArray(vertOffsetLoc);
						gl.glEnableVertexAttribArray(fragCenterLoc);
						gl.glBindBuffer(GL2.GL_ARRAY_BUFFER, vboId);

						gl.glVertexAttribPointer(fragCenterLoc, 2, GL2.GL_FLOAT, false, 4*4, 8);
														// 2: ひとつの頂点あたり配列要素2つ
														// 4*4: ストライドの要素数 x floatのバイト数
														// 8: ひとつの頂点あたり配列要素2つ x float のバイト数
	
						gl.glVertexAttribPointer(vertOffsetLoc, 2, GL2.GL_FLOAT, false, 4*4, 0);

						aaSupport.antiAlias(bounds.width, bounds.height, true, Color.COLORLESS_TRANSPARENT, new Runnable() {
							public void run() {
								gl.glDrawArrays(primitiveType, 0, numVertices);
							}
						});

					} finally {
						gl.glDisableVertexAttribArray(vertOffsetLoc);
						gl.glDisableVertexAttribArray(fragCenterLoc);
						gl.glBindBuffer(GL2.GL_ARRAY_BUFFER, 0);
					}
				}
			};

			buffer = context.createVideoBuffer(bounds);
			buffer.clear();

			support.useShaderProgram(program, prepareUniforms(bounds, patOrigin, patDirRad), operation, 0, buffer, input);

			IVideoBuffer result = buffer;
			buffer = null;
			return result;

		} finally {
			if (buffer != null) buffer.dispose();
		}
	}

	private void calcCameraParameters(VideoBounds bounds, int[] viewport, double[] prjMatrix, double[] mvMatrix) {
		CameraSystem cameraSystem = context.value(this.cameraSystem);
		switch (cameraSystem) {
			case LOCAL_CAMERA: {
				viewport[0] = 0;
				viewport[1] = 0;
				viewport[2] = bounds.width;
				viewport[3] = bounds.height;

				Resolution resolution = context.getVideoResolution();
				double cameraRotationX = context.value(this.cameraRotationX);
				double cameraRotationY = context.value(this.cameraRotationY);
				double cameraRotationZ = context.value(this.cameraRotationZ);
				Vec2d cameraPositionXY = resolution.scale(context.value(this.cameraPositionXY));
				double cameraPositionZ = resolution.scale(context.value(this.cameraPositionZ));
				double cameraZoom = resolution.scale(context.value(this.cameraZoom));
				double cameraFovy = Math.toDegrees(2 * Math.atan(bounds.height / (2*cameraZoom)));
				double cameraAspect = (double) bounds.width / bounds.height;
				double cameraNear = resolution.scale(context.value(this.cameraNear));
				double cameraFar = resolution.scale(context.value(this.cameraFar));

				GL2 gl = context.getGL().getGL2();

				gl.glMatrixMode(GL2.GL_PROJECTION);
				gl.glLoadIdentity();
				context.getGLU().gluPerspective(cameraFovy, cameraAspect, cameraNear, cameraFar);

				gl.glMatrixMode(GL2.GL_MODELVIEW);
				gl.glLoadIdentity();
				gl.glTranslatef((float)-cameraPositionXY.x, (float)-cameraPositionXY.y, (float)cameraPositionZ);
				gl.glTranslatef((float)(bounds.x+bounds.width/2.0), (float)(bounds.y+bounds.height/2.0), 0);
				gl.glRotatef((float)-cameraRotationZ, 0, 0, 1);
				gl.glRotatef((float)-cameraRotationY, 0, 1, 0);
				gl.glRotatef((float)-cameraRotationX, 1, 0, 0);
				gl.glTranslatef((float)-(bounds.x+bounds.width/2.0), (float)-(bounds.y+bounds.height/2.0), 0);
				gl.glScalef(1, 1, -1);

				gl.glGetDoublev(GL2.GL_PROJECTION_MATRIX, prjMatrix, 0);
				gl.glGetDoublev(GL2.GL_MODELVIEW_MATRIX, mvMatrix, 0);
				break;
			}

			case COMPOSITION_CAMERA: {
				ICamera camera = context.getCamera();

				Size2i vpSize = camera.getViewportSize();
				viewport[0] = (bounds.width-vpSize.width)/2;		// 2で割り切れないときは0.5ピクセル分ずれるが、しかたない。
				viewport[1] = (bounds.height-vpSize.height)/2;
				viewport[2] = vpSize.width;
				viewport[3] = vpSize.height;

				camera.getProjectionMatrix(prjMatrix);
				camera.getModelViewMatrix(mvMatrix);

				GL2 gl = context.getGL().getGL2();
				gl.glMatrixMode(GL2.GL_MODELVIEW);
				gl.glLoadMatrixd(mvMatrix, 0);
				gl.glTranslatef(-viewport[0], -viewport[1], 0);
				gl.glGetDoublev(GL2.GL_MODELVIEW_MATRIX, mvMatrix, 0);
				break;
			}

			default:
				throw new RuntimeException("unknown CameraSystem: " + cameraSystem);
		}
	}

	private Set<GLUniformData> prepareUniforms(VideoBounds bounds, Vec2d patOrigin, double patDirRad) {
		Resolution resolution = context.getVideoResolution();
		double patDirCos = Math.cos(patDirRad);
		double patDirSin = Math.sin(patDirRad);

		Set<GLUniformData> uniforms = new HashSet<GLUniformData>();

		uniforms.add(new GLUniformData("texture", 0));
		uniforms.add(new GLUniformData("texOffset", 2, toFloatBuffer(-bounds.x, -bounds.y)));
		uniforms.add(new GLUniformData("texSize", 2, toFloatBuffer(bounds.width, bounds.height)));
		uniforms.add(new GLUniformData("patOrigin", 2, toFloatBuffer(patOrigin.x, patOrigin.y)));
		uniforms.add(new GLUniformData("patDirMat", 2, 2, toFloatBuffer(patDirCos, patDirSin, -patDirSin, patDirCos)));

		double randomness = context.value(this.randomness);
		double rotationSpeed = context.value(this.rotationSpeed);
		double gravity = resolution.scale(context.value(this.gravity)) * 100;
		double gravityDirection = Math.toRadians(context.value(this.gravityDirection));
		double gravityInclination = Math.toRadians(context.value(this.gravityInclination));
		double gravityX = gravity * Math.cos(gravityInclination) * Math.sin(gravityDirection);
		double gravityY = -gravity * Math.cos(gravityInclination) * Math.cos(gravityDirection);
		double gravityZ = gravity * Math.sin(gravityInclination);

		uniforms.add(new GLUniformData("randomness", (float)randomness));
		uniforms.add(new GLUniformData("rotationSpeed", (float)rotationSpeed));
		uniforms.add(new GLUniformData("gravity", 3, toFloatBuffer(gravityX, gravityY, gravityZ)));

		double time;
		TimeBase timeBase = context.value(this.timeBase);
		switch (timeBase) {
			case LOCAL_TIME:
				time = context.value(this.time);
				break;
			case COMPOSITION_TIME:
				time = context.getTime().toSecond();
				break;
			default:
				throw new RuntimeException("unkown TimeBase: " + timeBase);
		}

		float[] forceTime = new float[3];
		float[] forcePosition = new float[2*3];
		float[] forceRadius = new float[3];
		float[] forceDelay = new float[3];
		float[] forceStrength = new float[3];
		float[] forceDecay = new float[3];
		float[] forceDepth = new float[3];

		int i = 0;

//		if (context.value(this.force1Enabled)) {
			double ftime = context.value(force1Time);
			Vec2d pos = resolution.scale(context.value(this.force1Position));
			double radius = resolution.scale(context.value(this.force1Radius));
			double delay = context.value(this.force1Delay);
			double strength = resolution.scale(context.value(this.force1Strength));
			double decay = resolution.scale(context.value(this.force1Decay));
			double spread = context.value(this.force1Spread);

			forceTime[i] = (float)(time - ftime);
			forcePosition[i*2] = (float)pos.x;
			forcePosition[i*2+1] = (float)pos.y;
			forceRadius[i] = (float)radius;
			forceDelay[i] = (float)delay;
			forceStrength[i] = (float)strength;
			forceDecay[i] = (float)(((strength < 0) ? -1 : 1) * decay);
			forceDepth[i] = spread == 0 ? -1 : (float)(radius/Math.tan(Math.toRadians(spread)));
			++i;
//		}
//		if (context.value(this.force2Enabled)) {
//			double ftime = context.value(force2Time);
//			Vec2d pos = resolution.scale(context.value(this.force2Position));
//			double radius = resolution.scale(context.value(this.force2Radius));
//			double delay = context.value(this.force2Delay);
//			double strength = resolution.scale(context.value(this.force2Strength));
//			double decay = resolution.scale(context.value(this.force2Decay));
//			double spread = context.value(this.force2Spread);
//
//			forceTime[i] = (float)(time - ftime);
//			forcePosition[i*2] = (float)pos.x;
//			forcePosition[i*2+1] = (float)pos.y;
//			forceRadius[i] = (float)radius;
//			forceDelay[i] = (float)delay;
//			forceStrength[i] = (float)strength;
//			forceDecay[i] = (float)(((strength < 0) ? -1 : 1) * decay);
//			forceDepth[i] = spread == 0 ? -1 : (float)(radius/Math.tan(Math.toRadians(spread)));
//			++i;
//		}
//		if (context.value(this.force3Enabled)) {
//			double ftime = context.value(force3Time);
//			Vec2d pos = resolution.scale(context.value(this.force3Position));
//			double radius = resolution.scale(context.value(this.force3Radius));
//			double delay = context.value(this.force3Delay);
//			double strength = resolution.scale(context.value(this.force3Strength));
//			double decay = resolution.scale(context.value(this.force3Decay));
//			double spread = context.value(this.force3Spread);
//
//			forceTime[i] = (float)(time - ftime);
//			forcePosition[i*2] = (float)pos.x;
//			forcePosition[i*2+1] = (float)pos.y;
//			forceRadius[i] = (float)radius;
//			forceDelay[i] = (float)delay;
//			forceStrength[i] = (float)strength;
//			forceDecay[i] = (float)(((strength < 0) ? -1 : 1) * decay);
//			forceDepth[i] = spread == 0 ? -1 : (float)(radius/Math.tan(Math.toRadians(spread)));
//			++i;
//		}

//		uniforms.add(new GLUniformData("numOfForces", i));
		uniforms.add(new GLUniformData("forceTime[0]", 1, FloatBuffer.wrap(forceTime)));
		uniforms.add(new GLUniformData("forcePosition[0]", 2, FloatBuffer.wrap(forcePosition)));
		uniforms.add(new GLUniformData("forceRadius[0]", 1, FloatBuffer.wrap(forceRadius)));
		uniforms.add(new GLUniformData("forceDelay[0]", 1, FloatBuffer.wrap(forceDelay)));
		uniforms.add(new GLUniformData("forceStrength[0]", 1, FloatBuffer.wrap(forceStrength)));
		uniforms.add(new GLUniformData("forceDecay[0]", 1, FloatBuffer.wrap(forceDecay)));
		uniforms.add(new GLUniformData("forceDepth[0]", 1, FloatBuffer.wrap(forceDepth)));

		uniforms.add(new GLUniformData("resolution", (float)resolution.scale));
		return uniforms;
	}

	private FloatBuffer toFloatBuffer(double... values) {
		float[] array = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			array[i] = (float)values[i];
		}
		return FloatBuffer.wrap(array);
	}

	@ShaderSource(type=ShaderType.VERTEX_SHADER, program=false)
	public static final String[] vertex_shader = {
		"const vec2 randVec = vec2(12.9898, 78.233);",
		"",
		"float rand1(vec2 coord, float seed)",
		"{",
		"	return fract(sin(dot(coord, randVec) + seed) * 43758.5453);",
		"}",
		"",
		"vec3 rand3(vec2 coord, float seed)",
		"{",
		"	float x = fract(sin(dot(coord, randVec) + seed) * 43758.5453);",
		"	float y = fract(sin(dot(coord, randVec*2.0) + seed) * 43758.5453);",
		"	float z = fract(sin(dot(coord, randVec*3.0) + seed) * 43758.5453);",
		"	return vec3(x, y, z);",
		"}",
		"",
		"mat3 selfRotMat(vec3 axis, float rad)",
		"{",
		"	float a = axis.x;",
		"	float b = axis.y;",
		"	float c = axis.z;",
		"	float cos = cos(rad);",
		"	float sin = sin(rad);",
		"	float omc = 1.0-cos;",
		"",
		"	float a2_omc = a*a*omc;",
		"	float ab_omc = a*b*omc;",
		"	float ac_omc = a*c*omc;",
		"	float b2_omc = b*b*omc;",
		"	float bc_omc = b*c*omc;",
		"	float c2_omc = c*c*omc;",
		"	float asin = a*sin;",
		"	float bsin = b*sin;",
		"	float csin = c*sin;",
		"",
		"	return mat3(",
		"		a2_omc+cos , ab_omc-csin, ac_omc+bsin,",
		"		ab_omc+csin, b2_omc+cos , bc_omc-asin,",
		"		ac_omc-bsin, bc_omc+asin, c2_omc+cos",
		"	);",
		"}",
		"",
		"attribute vec2 vertOffset;",
		"attribute vec2 fragCenter;",
		"",
		"uniform vec2 texOffset;",
		"uniform vec2 texSize;",
		"uniform vec2 patOrigin;",
		"uniform mat2 patDirMat;",
		"",
		"uniform float randomness;",
		"uniform float rotationSpeed;",
		"uniform vec3 gravity;",
		"",
//		"uniform int numOfForces;",
		"uniform float forceTime[3];",		// あらかじめ time プロパティから forceNTime プロパティを引いておいたもの
		"uniform vec2 forcePosition[3];",
		"uniform float forceRadius[3];",
		"uniform float forceDelay[3];",
		"uniform float forceStrength[3];",
		"uniform float forceDecay[3];",
		"uniform float forceDepth[3];",
		"",
		"uniform float resolution;",
		"",
		"void main(void)",
		"{",
		"	vec2 pos0 = patDirMat*(vertOffset+fragCenter-patOrigin)+patOrigin;",	// パターンの回転を適用した、破片の頂点
		"	vec2 fc = patDirMat*(fragCenter-patOrigin)+patOrigin;",					// パターンの回転を適用した、破片の中心
		"	vec2 rc = (fragCenter-patOrigin) / resolution;",						// rand1などの1つめの引数に使う値
		"",
		"	float t = 0.0;",		// フォースが到達してからの経過時間
		"	vec3 v = vec3(0.0);",	// フォース到達時の破片の初速
		"",
		"	/*for (int i = 0; i < numOfForces; ++i)*/ {",
		"		float ft  = forceTime[/*i*/0];",
		"		vec2  fp  = forcePosition[/*i*/0];",
		"		float fr  = forceRadius[/*i*/0];",
		"		float fdl = forceDelay[/*i*/0];",
		"		float fs  = forceStrength[/*i*/0];",
		"		float fdc = forceDecay[/*i*/0];",
		"		float fdp = forceDepth[/*i*/0];",
		"",
		"		vec2 fragCenterFromFp = fc - fp;",				// forcePosition から破片の中心へのベクトル
		"		float distance = length(fragCenterFromFp);",	// forcePosition から破片の中心までの距離
		"		float arrivalTime;",							// forcePosition の破壊開始時刻を基準にした、フォースの到達時刻 
		"		float arrivalStrength;",						// フォース到達時のフォースの強度
		"",
		"		if (distance > fr) {",
		"			arrivalTime = ft;", 
		"			arrivalStrength = 0.0;",
		"		} else {",
		"			float p = distance / fr;",
		"			float p3 = p*p*p;",
		"			arrivalTime = fdl * p3;",
		"			arrivalStrength = fs - fdc * p3;",
		"			if (arrivalStrength * fs < 0.0) {",
		"				arrivalStrength = 0.0;",
		"			}",
		"		}",
		"",
		"		float afterArrival = max(0.0, ft - arrivalTime);",
		"		if (afterArrival > t) {",
		"			t = afterArrival;",
		"			v = arrivalStrength * (fdp < 0.0 ? vec3(0.0, 0.0, -1.0) : normalize(vec3(fragCenterFromFp, -fdp)));",
		"		}",
		"	}",
		"",
		"	vec3 pos1 = vec3(pos0 - fc, 0.0);",			// 破片の中心を原点とした、破片の頂点の座標
		"",
		"	v += length(v)*randomness*normalize(4.0*rand3(rc, 1.0)-2.0);",
		"",
		"	vec3 selfRotAxis = cross(v, vec3(0.0, 0.0, -1.0));",
		"	if (length(selfRotAxis) > 0.0) {",
		"		float rs = rotationSpeed*(1.0+randomness*(4.0*rand1(rc, 2.0)-2.0));",
		"		pos1 = selfRotMat(normalize(selfRotAxis), t*rs) * pos1;",					// 自己回転
		"	}",
		"",
		"	vec3 pos2 = pos1 + vec3(fc, 0.0) + v*t + t*t*gravity;",
		"	gl_Position = gl_ModelViewProjectionMatrix * vec4(pos2, 1.0);",
		"",
		"	gl_TexCoord[0] = vec4((pos0 + texOffset) / texSize, 0.0, 1.0);",
		"}"
	};

	@ShaderSource(attach="vertex_shader")
	public static final String[] SHATTER = {
		"uniform sampler2D texture;",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2D(texture, gl_TexCoord[0].st);",
		"	if (color.a == 0.0) discard;",
		"	gl_FragColor = color;",
		"}"
	};

}
