/*
 * Copyright (c) 2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.stylize;

import java.nio.FloatBuffer;
import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.effects.TexCoords;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.FindEdges", category=Categories.STYLIZE)
public class FindEdges {

	@Property(value="0", min="0", max="100")
	private IAnimatableDouble threshold;

	@Property(value="100", min="0")
	private IAnimatableDouble strength;

	@Property(value="0", min="0", max="1")
	private IAnimatableDouble thickness;

	@Property("0,0,0")
	private IAnimatableColor edgeColor;

	@Property("1,1,1")
	private IAnimatableColor backgroundColor;

	@Property(value="100", min="0", max="100")
	private IAnimatableDouble backgroundOpacity;

	@Property(value="0", min="0", max="100")
	private IAnimatableDouble blendWithOriginal;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IShaderProgram luminosityProgram;

	private final IShaderProgram gradientProgram;

	private final IShaderProgram findEdgesProgram;

	@Inject
	public FindEdges(IVideoEffectContext context, IVideoRenderSupport support, IShaderRegistry shaders) {
		this.context = context;
		this.support = support;
		luminosityProgram = shaders.getProgram(FindEdges.class, "LUMINOSITY");
		gradientProgram = shaders.getProgram(FindEdges.class, "GRADIENT");
		findEdgesProgram = shaders.getProgram(FindEdges.class, "FIND_EDGES");
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer source = context.doPreviousEffect();

		VideoBounds bounds = source.getBounds();
		if (bounds.isEmpty()) {
			return source;
		}

		double threshold = context.value(this.threshold) / 100;
		double strength = context.value(this.strength) / 100;
		double thickness = context.value(this.thickness);
		Color edgeColor = context.value(this.edgeColor);
		Color bgColor = context.value(this.backgroundColor);
		double bgOpacity = context.value(this.backgroundOpacity) / 100;
		double blend = context.value(this.blendWithOriginal) / 100;

		IVideoBuffer luminosity = null;
		IVideoBuffer gradient = null;
		try {
			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();

			uniforms.add(new GLUniformData("source", 0));
			luminosity = support.useShaderProgram(luminosityProgram, uniforms, null, source);

			FloatBuffer neib = FloatBuffer.wrap(TexCoords.neighboringOffset8(bounds.width, bounds.height));

			uniforms.clear();
			uniforms.add(new GLUniformData("luminosity", 0));
			uniforms.add(new GLUniformData("neib[0]", 2, neib));
			gradient = support.useShaderProgram(gradientProgram, uniforms, null, luminosity);

			uniforms.clear();
			uniforms.add(new GLUniformData("gradient", 0));
			uniforms.add(new GLUniformData("source", 1));
			uniforms.add(new GLUniformData("neib[0]", 2, neib));
			uniforms.add(new GLUniformData("threshold", (float)threshold));
			uniforms.add(new GLUniformData("strength", (float)strength));
			uniforms.add(new GLUniformData("thickness", (float)thickness));
			uniforms.add(new GLUniformData("edgeColor", 4, toFloatBuffer(edgeColor.r, edgeColor.g, edgeColor.b, 1.0)));
			uniforms.add(new GLUniformData("bgColor", 4, toFloatBuffer(bgColor.r*bgOpacity, bgColor.g*bgOpacity, bgColor.b*bgOpacity, bgOpacity)));
			uniforms.add(new GLUniformData("blend", (float)blend));
			return support.useShaderProgram(findEdgesProgram, uniforms, null, gradient, source);

		} finally {
			if (gradient != null) gradient.dispose();
			if (luminosity != null) luminosity.dispose();
			source.dispose();
		}
	}

	private FloatBuffer toFloatBuffer(double... values) {
		float[] array = new float[values.length];
		for (int i = 0; i < values.length; ++i) {
			array[i] = (float)values[i];
		}
		return FloatBuffer.wrap(array);
	}

	@ShaderSource
	public static final String[] LUMINOSITY = {
		"uniform sampler2D source;",
		"",
		"const vec3 yvec = vec3(0.299, 0.587, 0.114);",
		"",
		"void main(void)",
		"{",
		"	vec4 src = texture2D(source, gl_TexCoord[0].st);",
		"	float y = (src.a != 0.0) ? dot(src.rgb/src.a, yvec) : 0.0;",
		"	gl_FragColor = vec4(y, y, y, src.a);",
		"}"
	};

	@ShaderSource
	public static final String[] GRADIENT = {
		"uniform sampler2D luminosity;",
		"uniform vec2 neib[8];",
		"",
		"void main(void)",
		"{",
		"	vec2 coord = gl_TexCoord[0].st;",
		"",
		"	vec4 luma0 = texture2D(luminosity, coord);",
		"	vec4 luma1 = texture2D(luminosity, coord + neib[1]);",
		"	vec4 luma2 = texture2D(luminosity, coord + neib[3]);",
		"	vec4 luma3 = texture2D(luminosity, coord + neib[4]);",
		"	vec4 luma4 = texture2D(luminosity, coord + neib[6]);",
		"",
		"	float tmp;",
		"	float grad = 0.0;",
		"	float alpha = 0.0;",
		"",
		"	tmp = abs(luma1.r - luma0.r);",
		"	if (tmp > grad) {",
		"		grad = tmp;",
		"		alpha = luma1.a;",
		"	}",
		"",
		"	tmp = abs(luma2.r - luma0.r);",
		"	if (tmp > grad) {",
		"		grad = tmp;",
		"		alpha = luma2.a;",
		"	}",
		"",
		"	tmp = abs(luma3.r - luma0.r);",
		"	if (tmp > grad) {",
		"		grad = tmp;",
		"		alpha = luma3.a;",
		"	}",
		"",
		"	tmp = abs(luma4.r - luma0.r);",
		"	if (tmp > grad) {",
		"		grad = tmp;",
		"		alpha = luma4.a;",
		"	}",
		"",
		"	alpha = max(alpha, luma0.a);",
		"",
		"	float sum = texture2D(luminosity, coord + neib[0]).r",
		"			  + luma1.r",
		"			  + texture2D(luminosity, coord + neib[2]).r",
		"			  + luma2.r",
		"			  - 8.0*luma0.r",
		"			  + luma3.r",
		"			  + texture2D(luminosity, coord + neib[5]).r",
		"			  + luma4.r",
		"			  + texture2D(luminosity, coord + neib[7]).r;",
		"",
		"	gl_FragColor = (sum > 0.0) ? vec4(grad, 0.0, 0.0, alpha) : vec4(0.0, grad, 0.0, alpha);",
		"}"
	};

	@ShaderSource
	public static final String[] FIND_EDGES = {
		"uniform sampler2D gradient;",
		"uniform sampler2D source;",
		"uniform vec2 neib[8];",
		"uniform float threshold;",
		"uniform float strength;",
		"uniform float thickness;",
		"uniform vec4 edgeColor;",
		"uniform vec4 bgColor;",
		"uniform float blend;",
		"",
		"void main(void)",
		"{",
		"	vec2 coord = gl_TexCoord[0].st;",
		"	vec4 grad = texture2D(gradient, coord);",
		"	float edge = 0.0;",
		"",
		"	if (grad.r != 0.0 && (",
		"			   texture2D(gradient, coord + neib[0]).g != 0.0",
		"			|| texture2D(gradient, coord + neib[1]).g != 0.0",
		"			|| texture2D(gradient, coord + neib[2]).g != 0.0",
		"			|| texture2D(gradient, coord + neib[3]).g != 0.0",
		"			|| texture2D(gradient, coord + neib[4]).g != 0.0",
		"			|| texture2D(gradient, coord + neib[5]).g != 0.0",
		"			|| texture2D(gradient, coord + neib[6]).g != 0.0",
		"			|| texture2D(gradient, coord + neib[7]).g != 0.0)) {",
		"",
		"		edge = (grad.r > threshold) ? grad.r : 0.0;",
		"	} else {",
		"		edge = (grad.g > threshold) ? grad.g*thickness : 0.0;",
		"	}",
		"",
		"	edge = min(edge*strength, 1.0);",
		"	vec4 color = edgeColor*edge + bgColor*(1.0-edge);",
		"",
		"	vec4 src = texture2D(source, coord);",
		"	gl_FragColor = color*grad.a*(1.0-blend) + src*blend;",
		"}"
	};

}
