/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.api;

/**
 * <p><code>Resolution</code> クラスはレンダリング解像度を表します。
 * エフェクトは、{@link ch.kuramo.javie.api.services.IEffectContext#getVideoResolution()} から返される
 * <code>Resolution</code> オブジェクトのレンダリング解像度に従って適切な処理を行う必要があります。</p>
 * 
 * <p><code>Resolution</code> オブジェクトは不変です。
 * <code>Resolution</code> オブジェクトは作成したあとに変更できないため、共用することができます。</p>
 */
public final class Resolution {

	/**
	 * フル解像度を表します。
	 */
	public static final Resolution FULL		= new Resolution(1);

	/**
	 * 1/2解像度を表します。
	 */
	public static final Resolution HALF		= new Resolution(1/2d);

	/**
	 * 1/3解像度を表します。
	 */
	public static final Resolution ONETHIRD	= new Resolution(1/3d);

	/**
	 * 1/4解像度を表します。
	 */
	public static final Resolution QUARTER	= new Resolution(1/4d);


	/**
	 * レンダリング解像度の値です。
	 */
	public final double scale;


	/**
	 * レンダリング解像度の値を指定して <code>Resolution</code> オブジェクトを生成します。
	 * 
	 * @param scale レンダリング解像度の値
	 */
	public Resolution(double scale) {
		this.scale = scale;
	}

	/**
	 * この <code>Resolution</code> オブジェクトのレンダリング解像度で引数 <code>d</code> をスケールします。
	 * 
	 * @param d スケールする値
	 * @return 引数 <code>d</code> をスケールした値
	 */
	public double scale(double d) {
		return d * scale;
	}

	/**
	 * この <code>Resolution</code> オブジェクトのレンダリング解像度で引数 <code>size</code> をスケールします。
	 * 
	 * @param size スケールする値
	 * @return 引数 <code>size</code> をスケールした値
	 */
	public Size2i scale(Size2i size) {
		return (scale == 1) ? size : new Size2i((int) (size.width * scale), (int) (size.height * scale));
	}

	/**
	 * この <code>Resolution</code> オブジェクトのレンダリング解像度で引数 <code>bounds</code> をスケールします。
	 * 
	 * @param bounds スケールする値
	 * @return 引数 <code>bounds</code> をスケールした値
	 */
	public VideoBounds scale(VideoBounds bounds) {
		return (scale == 1) ? bounds : new VideoBounds(bounds.x * scale, bounds.y * scale, 
											(int) (bounds.width * scale), (int) (bounds.height * scale));
	}

	/**
	 * この <code>Resolution</code> オブジェクトのレンダリング解像度で引数 <code>vec</code> をスケールします。
	 * 
	 * @param vec スケールする値
	 * @return 引数 <code>vec</code> をスケールした値
	 */
	public Vec2d scale(Vec2d vec) {
		return (scale == 1) ? vec : new Vec2d(vec.x * scale, vec.y * scale);
	}

	/**
	 * この <code>Resolution</code> オブジェクトのレンダリング解像度で引数 <code>vec</code> をスケールします。
	 * 
	 * @param vec スケールする値
	 * @return 引数 <code>vec</code> をスケールした値
	 */
	public Vec3d scale(Vec3d vec) {
		return (scale == 1) ? vec : new Vec3d(vec.x * scale, vec.y * scale, vec.z * scale);
	}

	/**
	 * この <code>Resolution</code> オブジェクトのレンダリング解像度で引数 <code>array</code> の各要素をスケールします。
	 * 
	 * @param array スケールする値
	 */
	public void scale(double[] array) {
		if (scale != 1) {
			for (int i = 0; i < array.length; ++i) {
				array[i] *= scale;
			}
		}
	}

	/**
	 * この <code>Resolution</code> オブジェクトのレンダリング解像度で引数 <code>array</code> の各要素をスケールします。
	 * 
	 * @param array スケールする値
	 */
	public void scale(float[] array) {
		if (scale != 1) {
			for (int i = 0; i < array.length; ++i) {
				array[i] *= scale;
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		long temp;
		temp = Double.doubleToLongBits(scale);
		result = prime * result + (int) (temp ^ (temp >>> 32));
		return result;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Resolution other = (Resolution) obj;
		if (Double.doubleToLongBits(scale) != Double.doubleToLongBits(other.scale))
			return false;
		return true;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		return "Resolution [" + scale + "]";
	}

}
