/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.actions;

import java.lang.reflect.InvocationTargetException;
import java.text.DateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.util.LocalSelectionTransfer;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.ITreeSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeColumnViewerLabelProvider;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.TreeViewerColumn;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.DragSourceEvent;
import org.eclipse.swt.dnd.DragSourceListener;
import org.eclipse.swt.dnd.DropTargetAdapter;
import org.eclipse.swt.dnd.DropTargetEvent;
import org.eclipse.swt.dnd.DropTargetListener;
import org.eclipse.swt.dnd.Transfer;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeColumn;
import org.eclipse.ui.PlatformUI;

import ch.kuramo.javie.app.BatchStore;
import ch.kuramo.javie.app.ColorUtil;
import ch.kuramo.javie.app.CommandIds;
import ch.kuramo.javie.app.BatchStore.BatchEntry;
import ch.kuramo.javie.app.BatchStore.BatchState;
import ch.kuramo.javie.app.misc.Twitter;
import ch.kuramo.javie.app.views.LocalSelectionWrapper;
import ch.kuramo.javie.app.widgets.FontUtil;
import ch.kuramo.javie.core.CompositionItem;
import ch.kuramo.javie.core.Item;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.output.Output;
import ch.kuramo.javie.core.output.ProgressMonitor;

public class BatchOutputAction extends Action {

	public BatchOutputAction() {
		super("バッチ出力...");
		setId(CommandIds.BATCH_OUTPUT);
		setActionDefinitionId(CommandIds.BATCH_OUTPUT);
		//setImageDescriptor(Activator.getImageDescriptor("/icons/batch_output.png"));
	}

	public void run() {
		Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
		BatchOutputDialog dialog = new BatchOutputDialog(shell);
		dialog.open();
	}

}

class BatchOutputDialog extends Dialog {

	private static final boolean COCOA = SWT.getPlatform().equals("cocoa");

	private static final int REMOVE_ID = IDialogConstants.CLIENT_ID + 1;
	private static final int PENDING_ID = IDialogConstants.CLIENT_ID + 2;
	private static final int RETRY_ID = IDialogConstants.CLIENT_ID + 3;


	private final List<BatchEntry> entries;

	private TreeViewer treeViewer;

	BatchOutputDialog(Shell parentShell) {
		super(parentShell);
		entries = BatchStore.listEntries();
	}

	protected Point getInitialSize() {
		Point size = super.getInitialSize();
		size.y = Math.max(size.y, 400);
		return size;
	}

	protected boolean isResizable() {
		return true;
	}

	protected void configureShell(Shell newShell) {
		super.configureShell(newShell);
		newShell.setText("バッチ出力");
	}

	protected Control createDialogArea(Composite parent) {
		Composite composite = (Composite) super.createDialogArea(parent);

		GridLayout layout = new GridLayout(1, false);
		layout.marginWidth = 10;
		layout.marginHeight = 0;
		layout.marginTop = 10;
		layout.verticalSpacing = 10;
		composite.setLayout(layout);


		treeViewer = new TreeViewer(composite, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER | SWT.FULL_SELECTION);

		treeViewer.setContentProvider(new ITreeContentProvider() {
			public void inputChanged(Viewer viewer, Object oldInput, Object newInput) { }
			public void dispose() { }
			public Object[] getElements(Object inputElement) { return ((List<?>)inputElement).toArray(); }
			public boolean hasChildren(Object element) { return false; }
			public Object getParent(Object element) { return null; }
			public Object[] getChildren(Object parentElement) { return null; }
		});

		Tree tree = treeViewer.getTree();
		tree.setHeaderVisible(true);
		tree.setLinesVisible(!COCOA);

		FontUtil.setCompatibleFont(tree);
		tree.setBackground(ColorUtil.tableBackground());

		if (COCOA) {
			final Color lineColor = ColorUtil.tableRowLine();

			tree.addListener(SWT.EraseItem, new Listener() {
				public void handleEvent(Event event) {
					int x = event.x;
					int y = event.y + event.height - 1;
					GC gc = event.gc;
					gc.setForeground(lineColor);
					gc.drawLine(x, y, x + event.width, y);
				}
			});
		}


		createViewerColumn( 65, "状況");
		createViewerColumn(200, "出力ファイル");
		createViewerColumn(100, "プロジェクト");
		createViewerColumn(100, "コンポジション");
		createViewerColumn(100, "登録");
		createViewerColumn(100, "開始");
		createViewerColumn(100, "完了");


		treeViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
				updateControls();
			}
		});

		initDragAndDrop();


		GridData gridData = new GridData();
		gridData.horizontalAlignment = SWT.FILL;
		gridData.verticalAlignment = SWT.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		tree.setLayoutData(gridData);


		treeViewer.setInput(entries);
		return composite;
	}

	private static class BatchLabelProvider extends LabelProvider implements ITableLabelProvider {
		public Image getColumnImage(Object element, int columnIndex) {
			return null;
		}

		public String getColumnText(Object element, int columnIndex) {
			BatchEntry entry = (BatchEntry) element;
			switch (columnIndex) {
				case 0: return getState(entry);
				case 1: return entry.file.getName();
				case 2: return entry.projectName;
				case 3: return entry.compositionName;
				case 4: return formatDate(entry.registeredTime);
				case 5: return formatDate(entry.startTime);
				case 6: return formatDate(entry.endTime);
			}
			return null;
		}

		private String getState(BatchEntry entry) {
			switch (entry.state) {
				case READY: return "";
				case IN_PROGRESS: return "出力中";
				case CANCELED: return "中止";
				case PENDING: return "保留";
				case ERROR: return "エラー";
				case DONE: return "完了";
				case REMOVED: return "削除";
				default: return entry.state.name();
			}
		}

		private static final DateFormat df = DateFormat.getDateTimeInstance(DateFormat.SHORT, DateFormat.SHORT);

		private String formatDate(Date date) {
			return date != null ? df.format(date) : "";
		}
	}

	private TreeViewerColumn createViewerColumn(int width, String name) {
		TreeViewerColumn viewerColumn = new TreeViewerColumn(treeViewer, SWT.NONE);
		TreeColumn column = viewerColumn.getColumn();
		column.setWidth(width);
		column.setText(name);
		column.setMoveable(true);

		viewerColumn.setLabelProvider(new TreeColumnViewerLabelProvider(new BatchLabelProvider()));

		return viewerColumn;
	}

	private void initDragAndDrop() {
		final LocalSelectionTransfer selectionTransfer = LocalSelectionTransfer.getTransfer();

		DragSourceListener sourceListener = new DragSourceListener() {
			public void dragStart(DragSourceEvent event) {
				ISelection selection = new LocalSelectionWrapper(treeViewer);
				if (selection.isEmpty()) {
					event.doit = false;
				} else {
					selectionTransfer.setSelection(selection);
					selectionTransfer.setSelectionSetTime(event.time & 0xFFFFFFFFL);
					event.doit = true;
				}
			}

			public void dragSetData(DragSourceEvent event) {
				event.data = selectionTransfer.getSelection();
			}

			public void dragFinished(DragSourceEvent event) {
				selectionTransfer.setSelection(null);
				selectionTransfer.setSelectionSetTime(0);
			}
		};

		DropTargetListener targetListener = new DropTargetAdapter() {
			private boolean isAcceptable(DropTargetEvent event) {
				if (!selectionTransfer.isSupportedType(event.currentDataType)) {
					return false;
				}
				ITreeSelection selection = LocalSelectionWrapper.getSelection(selectionTransfer, treeViewer);
				if (selection == null) {
					return false;
				}
				return true;
			}

			public void dragOver(DropTargetEvent event) {
				if (!isAcceptable(event)) {
					event.detail = DND.DROP_NONE;
					event.feedback = DND.FEEDBACK_NONE;
					return;
				}
				event.detail = event.operations & DND.DROP_MOVE;
				event.feedback = (event.feedback | DND.FEEDBACK_INSERT_BEFORE)
								& ~DND.FEEDBACK_SELECT & ~DND.FEEDBACK_EXPAND;
			}

			public void dropAccept(DropTargetEvent event) {
				dragOver(event);
			}

			public void drop(DropTargetEvent event) {
				ITreeSelection selection = LocalSelectionWrapper.getSelection(selectionTransfer, treeViewer);
				if (selection == null) {
					event.detail = DND.DROP_NONE;
					return;
				}

				@SuppressWarnings("unchecked")
				List<BatchEntry> srcList = selection.toList();

				if (event.item == null) {
					entries.removeAll(srcList);
					entries.addAll(srcList);
				} else {
					Object target = event.item.getData();
					List<BatchEntry> srcList2 = Util.newList(srcList);
					boolean targetIsSrc = srcList2.remove(target);
					entries.removeAll(srcList2);
					int targetIndex = entries.indexOf(target);
					if (targetIsSrc) {
						entries.remove(target);
					}
					entries.addAll(targetIndex, srcList);
				}
				BatchStore.saveEntries(entries);
				treeViewer.refresh();
			}
		};

		treeViewer.addDragSupport(DND.DROP_MOVE, new Transfer[] { selectionTransfer }, sourceListener);
		treeViewer.addDropSupport(DND.DROP_MOVE, new Transfer[] { selectionTransfer }, targetListener);
	}

	protected void createButtonsForButtonBar(Composite parent) {
		((GridLayout)parent.getLayout()).makeColumnsEqualWidth = false;
		((GridData)parent.getLayoutData()).horizontalAlignment = GridData.FILL;

		createButton(parent, REMOVE_ID, "削除", false);
		createButton(parent, PENDING_ID, "保留", false);
		createButton(parent, RETRY_ID, "再開", false);

		Button dummy = createButton(parent, IDialogConstants.CLIENT_ID + 0, "", false);
		((GridData)dummy.getLayoutData()).grabExcessHorizontalSpace = true;
		dummy.setVisible(false);

		createButton(parent, IDialogConstants.OK_ID, "出力開始", true);
		createButton(parent, IDialogConstants.CANCEL_ID, "閉じる", false);
	}

	public void create() {
		super.create();
		updateControls();
	}

	private void updateControls() {
		Button removeButton = getButton(REMOVE_ID);
		Button pendingButton = getButton(PENDING_ID);
		Button retryButton = getButton(RETRY_ID);
		Button okButton = getButton(IDialogConstants.OK_ID);

		ITreeSelection selection = (ITreeSelection) treeViewer.getSelection();

		boolean removeButtonEnabled = false;
		boolean pendingButtonEnabled = false;
		boolean retryButtonEnabled = false;
		for (Iterator<?> it = selection.iterator(); it.hasNext(); ) {
			switch (((BatchEntry) it.next()).state) {
				case READY:
					removeButtonEnabled = true;
					pendingButtonEnabled = true;
					break;

				case CANCELED:
				case PENDING:
				case ERROR:
					removeButtonEnabled = true;
					retryButtonEnabled = true;
					break;

				case DONE:
					retryButtonEnabled = true;
					break;

				case REMOVED:
					pendingButtonEnabled = true;
					retryButtonEnabled = true;
					break;
			}
		}
		removeButton.setEnabled(removeButtonEnabled);
		pendingButton.setEnabled(pendingButtonEnabled);
		retryButton.setEnabled(retryButtonEnabled);


		boolean okButtonEnabled = false;
		for (BatchEntry entry : entries) {
			if (entry.state == BatchState.READY) {
				okButtonEnabled = true;
				break;
			}
		}
		okButton.setEnabled(okButtonEnabled);
	}

	protected void buttonPressed(int buttonId) {
		if (buttonId < IDialogConstants.CLIENT_ID) {
			super.buttonPressed(buttonId);
			return;
		}

		ITreeSelection selection = (ITreeSelection) treeViewer.getSelection();

		for (Iterator<?> it = selection.iterator(); it.hasNext();) {
			BatchEntry entry = (BatchEntry) it.next();
			switch (buttonId) {
				case REMOVE_ID: entry.state = BatchState.REMOVED; break;
				case PENDING_ID: entry.state = BatchState.PENDING; break;
				case RETRY_ID:
					entry.state = BatchState.READY;
					entry.startTime = null;
					entry.endTime = null;
					break;
			}
		}

		BatchStore.saveEntries(entries);
		treeViewer.update(selection.toArray(), null);
		updateControls();
	}

	protected void cancelPressed() {
		super.cancelPressed();
		BatchStore.cleanup();
	}

	protected void okPressed() {
		long start = System.currentTimeMillis();

		for (final BatchEntry entry : entries) {
			if (entry.state != BatchState.READY) continue;

			Project project = null;
			try {
				entry.state = BatchState.IN_PROGRESS;
				entry.startTime = new Date();
				treeViewer.update(entry, null);
				treeViewer.setSelection(new TreeSelection(new TreePath(new Object[] { entry })), true);

				Object[] data = BatchStore.loadData(entry);

				project = (Project) data[0];
				final Output output = (Output) data[1];

				CompositionItem compItem = (CompositionItem) project.getItem(entry.compositionItemId);
				output.setComposition(compItem.getComposition());

				ProgressMonitorDialog dialog = new ProgressMonitorDialog(getShell());
				dialog.create();
				dialog.getShell().setText("書き出し");
				dialog.run(true, true, new IRunnableWithProgress() {
					public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
						output.doOutput(createProgressMonitor(monitor, entry.compositionName, output.getFile().getName()));
						if (monitor.isCanceled()) throw new InterruptedException();
					}
				});

				entry.state = BatchState.DONE;
				entry.endTime = new Date();

				Twitter.finishComp(compItem.getName(), output.getFile().getName(),
						(entry.endTime.getTime() - entry.startTime.getTime()) / 1000);

			} catch (InvocationTargetException e) {
				entry.state = BatchState.ERROR;
				e.printStackTrace();
				
			} catch (InterruptedException e) {
				entry.state = BatchState.CANCELED;

			} finally {
				if (project != null) {
					for (Item item : project.getItems()) {
						item.dispose();
					}
				}
			}

			BatchStore.saveEntries(entries);
			treeViewer.update(entry, null);
			updateControls();

			if (entry.state == BatchState.CANCELED) {
				break;
			}
		}

		Twitter.finishBatch((System.currentTimeMillis() - start) / 1000);
	}

	private ProgressMonitor createProgressMonitor(
			final IProgressMonitor monitor, final String compName, final String fileName) {

		return new ProgressMonitor() {
			private int totalWork;
			private int currentWork;

			private final long startTime = System.currentTimeMillis();
			private final int tweetInterval = Twitter.getCompProgressInterval();
			private int tweetCount = 0;

			public void beginTask(int totalWork) {
				monitor.beginTask(String.format("書き出し: %s => %s", compName, fileName), totalWork);
				this.totalWork = totalWork;
			}

			public void currentWork(int currentWork) {
				if (currentWork > this.currentWork) {
					monitor.worked(currentWork - this.currentWork);
					monitor.subTask(String.format("%.1f%%", 100.0 * currentWork / totalWork));
					this.currentWork = currentWork;
				}

				long time = (System.currentTimeMillis() - startTime) / 60000;
				if (time >= tweetInterval * (tweetCount + 1)) {
					Twitter.compProgress(compName, fileName, 100.0 * currentWork / totalWork);
					++tweetCount;
				}
			}

			public void done() {
				monitor.done();
			}

			public boolean isCanceled() {
				return monitor.isCanceled();
			}
		};
	}

}
