/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.misc;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import net.arnx.jsonic.JSON;

import org.eclipse.core.runtime.preferences.AbstractPreferenceInitializer;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.browser.IWebBrowser;
import org.eclipse.ui.browser.IWorkbenchBrowserSupport;
import org.scribe.builder.ServiceBuilder;
import org.scribe.builder.api.TwitterApi;
import org.scribe.model.OAuthRequest;
import org.scribe.model.Response;
import org.scribe.model.Token;
import org.scribe.model.Verb;
import org.scribe.model.Verifier;
import org.scribe.oauth.OAuthService;

import ch.kuramo.javie.app.Activator;
import ch.kuramo.javie.app.widgets.GridBuilder;

public class Twitter {

	private static final String TWITTER_TOKEN					= "TWITTER_TOKEN";
	private static final String TWITTER_SECRET					= "TWITTER_SECRET";
	private static final String TWITTER_COMP_PROGRESS_ENABLED	= "TWITTER_COMP_PROGRESS_ENABLED";
	private static final String TWITTER_COMP_PROGRESS_TEMPLATE	= "TWITTER_COMP_PROGRESS_TEMPLATE";
	private static final String TWITTER_COMP_PROGRESS_INTERVAL	= "TWITTER_COMP_PROGRESS_INTERVAL";
	private static final String TWITTER_FINISH_COMP_ENABLED		= "TWITTER_FINISH_COMP_ENABLED";
	private static final String TWITTER_FINISH_COMP_TEMPLATE	= "TWITTER_FINISH_COMP_TEMPLATE";
	private static final String TWITTER_FINISH_BATCH_ENABLED	= "TWITTER_FINISH_BATCH_ENABLED";
	private static final String TWITTER_FINISH_BATCH_TEMPLATE	= "TWITTER_FINISH_BATCH_TEMPLATE";

	private static IPreferenceStore prefStore;

	private static OAuthService oauthService;

	private static ExecutorService executorService;

	private static OAuthService getOAuthService() {
		synchronized (Twitter.class) {
			if (oauthService == null) {
				oauthService = new ServiceBuilder()
						.provider(TwitterApi.class)
						.apiKey("lGTFxMSmzFATgw1Qf1Mnq")
						.apiSecret("kMuOAuAse8jJFMOPqyUktojMyRfTf0qEGoYw1L4FvS")
						.build();
			}
		}
		return oauthService;
	}

	private static ExecutorService getExecutorService() {
		synchronized (Twitter.class) {
			if (executorService == null) {
				executorService = Executors.newSingleThreadExecutor();
			}
		}
		return executorService;
	}

	public static class TwitterPreferenceInitializer extends AbstractPreferenceInitializer {
		@Override
		public void initializeDefaultPreferences() {
			prefStore = Activator.getDefault().getPreferenceStore();
			prefStore.setDefault(TWITTER_TOKEN, "");
			prefStore.setDefault(TWITTER_SECRET, "");
			prefStore.setDefault(TWITTER_COMP_PROGRESS_ENABLED, true);
			prefStore.setDefault(TWITTER_COMP_PROGRESS_TEMPLATE, "出力中：$comp_name => $file_name ($percent%) #JavieOutput");
			prefStore.setDefault(TWITTER_COMP_PROGRESS_INTERVAL, 10);
			prefStore.setDefault(TWITTER_FINISH_COMP_ENABLED, true);
			prefStore.setDefault(TWITTER_FINISH_COMP_TEMPLATE, "出力完了：$comp_name => $file_name ($time) #JavieOutput");
			prefStore.setDefault(TWITTER_FINISH_BATCH_ENABLED, true);
			prefStore.setDefault(TWITTER_FINISH_BATCH_TEMPLATE, "バッチ出力完了 ($time) #JavieOutput");
		}
	}

	public static class TwitterPreferencePage extends PreferencePage implements IWorkbenchPreferencePage {

		private Button enabledButton;
		private Button progButton;
		private Text progText;
		private Label intervalLabel1;
		private Combo intervalCombo;
		private Label intervalLabel2;
		private Button compButton;
		private Text compText;
		private Button batchButton;
		private Text batchText;

		private String token;
		private String secret;

		public void init(IWorkbench workbench) {
			// nothing to do
		}

		@Override
		protected Control createContents(Composite parent) {
			GridBuilder gb1 = new GridBuilder(parent, 1, false);
			((GridLayout) gb1.getComposite().getLayout()).verticalSpacing = 0;

			enabledButton = gb1.hAlign(SWT.LEFT).button(SWT.CHECK, "Twitter連携を使用する");
			Group group = gb1.hAlign(SWT.FILL).hGrab().group(SWT.NONE, null);

			FillLayout layout = new FillLayout(SWT.HORIZONTAL);
			group.setLayout(layout);
			GridBuilder gb2 = new GridBuilder(group, 4, false);

			progButton =	gb2.hSpan(1).hAlign(SWT.LEFT).button(SWT.CHECK, "コンポジション出力中:");
			progText =		gb2.hSpan(3).hAlign(SWT.FILL).hGrab().text(SWT.BORDER, "");
							gb2.hSpan(1).size(10, 10).composite(SWT.NULL);
			intervalLabel1= gb2.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "投稿間隔:");
			intervalCombo =	gb2.hSpan(1).hAlign(SWT.FILL).combo(SWT.READ_ONLY, 6, new String[]{"10","20","30","40","50","60"}, "10");
			intervalLabel2= gb2.hSpan(1).hAlign(SWT.LEFT).hGrab().label(SWT.NULL, "分");
			compButton = 	gb2.hSpan(1).hAlign(SWT.LEFT).button(SWT.CHECK, "コンポジション出力完了時:");
			compText =		gb2.hSpan(3).hAlign(SWT.FILL).hGrab().text(SWT.BORDER, "");
			batchButton =	gb2.hSpan(1).hAlign(SWT.LEFT).button(SWT.CHECK, "バッチ出力完了時:");
			batchText =		gb2.hSpan(3).hAlign(SWT.FILL).hGrab().text(SWT.BORDER, "");

			performReset();

			enabledButton.addSelectionListener(new SelectionAdapter() {
				public void widgetSelected(SelectionEvent e) {
					boolean enabled = enabledButton.getSelection();
					if (enabled && (token.length() == 0 || secret.length() == 0) && !authorize()) {
						enabled = false;
						enabledButton.setSelection(false);
					}
					enableControls(enabled);
				}
			});

			Composite grid2 = gb2.getComposite();
			grid2.setTabList(gb2.getTabList());
			Composite grid1 = gb1.getComposite();
			grid1.setTabList(gb1.getTabList());
			return grid1;
		}

		private void enableControls(boolean enabled) {
			progButton.setEnabled(enabled);
			progText.setEnabled(enabled);
			intervalLabel1.setEnabled(enabled);
			intervalCombo.setEnabled(enabled);
			intervalLabel2.setEnabled(enabled);
			compButton.setEnabled(enabled);
			compText.setEnabled(enabled);
			batchButton.setEnabled(enabled);
			batchText.setEnabled(enabled);
		}

		private void performReset() {
			token = prefStore.getString(TWITTER_TOKEN);
			secret = prefStore.getString(TWITTER_SECRET);

			boolean enabled = (token.length() != 0 && secret.length() != 0);
			enabledButton.setSelection(enabled);
			enableControls(enabled);

			progButton.setSelection(prefStore.getBoolean(TWITTER_COMP_PROGRESS_ENABLED));
			progText.setText(prefStore.getString(TWITTER_COMP_PROGRESS_TEMPLATE));
			intervalCombo.select(prefStore.getInt(TWITTER_COMP_PROGRESS_INTERVAL) / 10 - 1);
			compButton.setSelection(prefStore.getBoolean(TWITTER_FINISH_COMP_ENABLED));
			compText.setText(prefStore.getString(TWITTER_FINISH_COMP_TEMPLATE));
			batchButton.setSelection(prefStore.getBoolean(TWITTER_FINISH_BATCH_ENABLED));
			batchText.setText(prefStore.getString(TWITTER_FINISH_BATCH_TEMPLATE));
		}

		@Override
		protected void performDefaults() {
			//token = "";
			//secret = "";

			enabledButton.setSelection(false);
			enableControls(false);

			progButton.setSelection(prefStore.getDefaultBoolean(TWITTER_COMP_PROGRESS_ENABLED));
			progText.setText(prefStore.getDefaultString(TWITTER_COMP_PROGRESS_TEMPLATE));
			intervalCombo.select(prefStore.getDefaultInt(TWITTER_COMP_PROGRESS_INTERVAL) / 10 - 1);
			compButton.setSelection(prefStore.getDefaultBoolean(TWITTER_FINISH_COMP_ENABLED));
			compText.setText(prefStore.getDefaultString(TWITTER_FINISH_COMP_TEMPLATE));
			batchButton.setSelection(prefStore.getDefaultBoolean(TWITTER_FINISH_BATCH_ENABLED));
			batchText.setText(prefStore.getDefaultString(TWITTER_FINISH_BATCH_TEMPLATE));
		}

		@Override
		public boolean performOk() {
			if (enabledButton.getSelection()) {
				prefStore.setValue(TWITTER_TOKEN, token);
				prefStore.setValue(TWITTER_SECRET, secret);
			} else {
				if (token.length() != 0 && secret.length() != 0) {
					// TODO APIで「許可の取り消し」をする方法がわからなかった。するならここで。
				}
				token = "";
				secret = "";
				prefStore.setValue(TWITTER_TOKEN, "");
				prefStore.setValue(TWITTER_SECRET, "");
			}
			prefStore.setValue(TWITTER_COMP_PROGRESS_ENABLED, progButton.getSelection());
			prefStore.setValue(TWITTER_COMP_PROGRESS_TEMPLATE, progText.getText().trim());
			prefStore.setValue(TWITTER_COMP_PROGRESS_INTERVAL, (intervalCombo.getSelectionIndex() + 1) * 10);
			prefStore.setValue(TWITTER_FINISH_COMP_ENABLED, compButton.getSelection());
			prefStore.setValue(TWITTER_FINISH_COMP_TEMPLATE, compText.getText().trim());
			prefStore.setValue(TWITTER_FINISH_BATCH_ENABLED, batchButton.getSelection());
			prefStore.setValue(TWITTER_FINISH_BATCH_TEMPLATE, batchText.getText().trim());
			return true;
		}

		private boolean authorize() {
			token = "";
			secret = "";

			OAuthService oauth = getOAuthService();
			final Token requestToken = oauth.getRequestToken();
			String authUrl = oauth.getAuthorizationUrl(requestToken);

			IWorkbenchBrowserSupport browserSupport = PlatformUI.getWorkbench().getBrowserSupport();
			try {
				IWebBrowser browser = browserSupport.getExternalBrowser();
				browser.openURL(new URL(authUrl));
			} catch (PartInitException e) {
				e.printStackTrace();
				return false;
			} catch (MalformedURLException e) {
				e.printStackTrace();
				return false;
			}

			Dialog dialog = new Dialog(getShell()) {
				private String pin = "";

				@Override
				protected void configureShell(Shell newShell) {
					super.configureShell(newShell);
					newShell.setText("Twitter / アプリケーション認証");
				}

				@Override
				protected Control createDialogArea(Composite parent) {
					Composite composite = (Composite) super.createDialogArea(parent);
					GridBuilder gb = new GridBuilder(composite, 2, false);

					String msg = "ブラウザに表示された内容をよくお読みになり、Javieがあなたのアカウント\n" +
								 "を利用することを許可する場合は「連携アプリを認証」を押してください。\n" +
								 "そして、表示されたPINを以下に入力しOKボタンを押してください。\n";
					gb.hSpan(2).hAlign(SWT.FILL).label(SWT.NULL, msg);
					gb.hSpan(1).hAlign(SWT.RIGHT).label(SWT.NULL, "PIN:");
					final Text pinText = gb.hSpan(1).hAlign(SWT.FILL).text(SWT.BORDER, "");
					pinText.addModifyListener(new ModifyListener() {
						public void modifyText(ModifyEvent e) {
							pin = pinText.getText().trim();
						}
					});

					Composite grid = gb.getComposite();
					grid.setTabList(gb.getTabList());
					parent.pack();
					return composite;
				}

				@Override
				protected void okPressed() {
					if (pin.length() == 0) {
						Display.getCurrent().beep();
						return;
					}

					try {
						Token accessToken = getOAuthService().getAccessToken(requestToken, new Verifier(pin));
						token = accessToken.getToken();
						secret = accessToken.getSecret();
					} catch (Exception e) {
						e.printStackTrace();
						MessageDialog.openError(getShell(), "エラー", "アプリケーション認証に失敗しました。");
					}

					super.okPressed();
				}
			};

			dialog.open();
			return (token.length() != 0 && secret.length() != 0);
		}
	}

	public static int getCompProgressInterval() {
		return prefStore.getInt(TWITTER_COMP_PROGRESS_INTERVAL);
	}

	public static void compProgress(String compName, String fileName, double progressInPercent) {
		if (compName == null) throw new NullPointerException();
		if (fileName == null) throw new NullPointerException();
		updateStatus(compName, fileName, progressInPercent);
	}

	public static void finishComp(String compName, String fileName, long timeInSeconds) {
		if (compName == null) throw new NullPointerException();
		if (fileName == null) throw new NullPointerException();
		updateStatus(compName, fileName, timeInSeconds);
	}

	public static void finishBatch(long timeInSeconds) {
		updateStatus(null, null, timeInSeconds);
	}

	public static void updateStatus(final String compName, final String fileName, final Number timeOrPercent) {
		getExecutorService().submit(new Runnable() {
			public void run() {
				try {

					String token = prefStore.getString(TWITTER_TOKEN);
					String secret = prefStore.getString(TWITTER_SECRET);

					if (token.length() == 0 || secret.length() == 0) {
						return;
					}

					boolean prog = (timeOrPercent instanceof Double);
					boolean comp = (compName != null);

					if (!prefStore.getBoolean(prog ? TWITTER_COMP_PROGRESS_ENABLED :
											  comp ? TWITTER_FINISH_COMP_ENABLED : TWITTER_FINISH_BATCH_ENABLED)) {
						return;
					}

					OAuthService oauth = getOAuthService();
					Token accessToken = new Token(token, secret);

					String status = prefStore.getString(prog ? TWITTER_COMP_PROGRESS_TEMPLATE :
														comp ? TWITTER_FINISH_COMP_TEMPLATE : TWITTER_FINISH_BATCH_TEMPLATE);
					if (status.contains("$screen_name")) {
						OAuthRequest request = new OAuthRequest(Verb.GET, "http://api.twitter.com/1/account/verify_credentials.json");
						oauth.signRequest(accessToken, request);
						Response response = request.send();
						if (!response.isSuccessful()) {
							return;
						}
						@SuppressWarnings("unchecked")
						Map<String, Object> map = (Map) JSON.decode(response.getBody());
						status = status.replaceAll("\\$screen_name", map.get("screen_name").toString());
					}

					if (prog) {
						status = status.replaceAll("\\$percent",
								String.format("%.1f", timeOrPercent.doubleValue()));
					} else {
						long s = timeOrPercent.longValue();
						long h = s / 3600; s %= 3600;
						long m = s / 60; s %= 60;
						status = status.replaceAll("\\$time", h > 0 ? String.format("%d:%02d:%02d", h, m, s)
								   : String.format("%d:%02d", m, s));
					}

					if (comp) {
						status = status.replaceAll("\\$comp_name", compName);
						status = status.replaceAll("\\$file_name", fileName);
					}

					OAuthRequest request = new OAuthRequest(Verb.POST, "https://api.twitter.com/1/statuses/update.json");
					request.addBodyParameter("status", status);
					oauth.signRequest(accessToken, request);
					Response response = request.send();
					if (!response.isSuccessful()) {
						// ignore
					}

				} catch (Throwable t) {
					t.printStackTrace();
				}
			}
		});
	}

}
