/*
 * Copyright (c) 2009 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.player;

import java.util.concurrent.atomic.AtomicReference;

import javax.sound.sampled.DataLine;

import ch.kuramo.javie.api.Time;

public class TimeKeeper {

	private final Clock clock;

	private final AtomicReference<Time[]> timesRef = new AtomicReference<Time[]>();

	private volatile boolean paused = true;


	public TimeKeeper(Clock clock) {
		this.clock = clock;
		resetTime();
	}

	private Time[] getTimes() {
		Time[] times = timesRef.get();

		if (!paused) {
			Time now = new Time(clock.getValue() - times[1].timeValue, times[1].timeScale);

			// pause中にsetTimeで時刻指定したのちresumeすると、resume直後のgetTimeの呼び出しでは
			// このnowの値がsetTimeで指定した時刻より僅かに過去の時刻となることがある。
			// これは、setTimeで指定した時刻のスケールとクロックのスケールの違いからくる計算誤差が原因である。
			// (setTime内での timeValueInClockScale の計算誤差)
			//
			// 最後にgetTimeしたときの時刻またはsetTimeで指定した時刻(=times[0]の値)よりも過去の時刻を
			// 返してはならないので、now <= times[0] の場合は timesRef を更新せず、times[0] をそのまま返す。

			if (now.after(times[0])) {
				Time[] newTimes = new Time[] { now, times[1] };
				if (timesRef.compareAndSet(times, newTimes)) {
					times = newTimes;
				} else {
					// 更新失敗＝他のスレッドがほぼ同時に更新ということなので、その値をそのまま使えばよい。
					times = timesRef.get();
				}
			}
		}

		return times;
	}

	public Time[] getTimeAndBase() {
		return getTimes().clone();
	}

	public Time getTime() {
		return getTimes()[0];
	}

	public void setTime(Time time) {
		// puase中にsetTimeした場合、正確に同じ時刻がgetTimeで取り出せる必要があるので、
		// 引数に渡された時刻はタイムスケールの変換をせずそのまま timesRef に格納する。
		// 一方、基準時刻はclockのタイムスケールである必要があるので、
		// 基準時刻を逆算するためには引数に渡された時刻のtimeValueをclockのスケールに変換する必要がある。

		int clockScale = clock.getScale();
		long timeValueInClockScale;

		if (time.timeScale == clockScale) {
			timeValueInClockScale = time.timeValue;
		} else {
			timeValueInClockScale = Math.round((double) clockScale * time.timeValue / time.timeScale);
		}

		Time baseTime = new Time(clock.getValue() - timeValueInClockScale, clockScale);
		timesRef.set(new Time[] { time, baseTime });
	}

	public void resetTime() {
		int clockScale = clock.getScale();

		timesRef.set(new Time[] {
				new Time(0, clockScale),
				new Time(clock.getValue(), clockScale)
		});

		// このTime配列の内容
		//	0: 最後にgetTimeしたときの時刻、またはsetTimeでセットされた時刻。
		//	1: 計時の基準となる時刻で、timeScaleはclockのscale。
	}

	/**
	 * pause() と resume() は単一のスレッドから操作する必要がある。
	 */
	public Time pause() {
		Time time = getTime();
		paused = true;
		return time;
	}

	/**
	 * pause() と resume() は単一のスレッドから操作する必要がある。
	 */
	public Time resume(Time time) {
		if (paused && time == null) {
			time = getTime();
		}
		if (time != null) {
			setTime(time);
		}
		paused = false;
		return time;
	}

	public boolean isPaused() {
		return paused;
	}


	public static TimeKeeper fromSystemTime() {
		return new TimeKeeper(new AbstractClock(1000000000) {
			protected long rawValue() {
				return System.nanoTime();
			}
		});
	}

	public static TimeKeeper fromAudioLine(final DataLine line) {
		// TODO getLongFramePosition が返す値は「ラインが開かれてから描画されたサンプルフレーム数」なので、
		//      ラインが閉じられることを考慮するなら以下では不十分なはず。(今のところ考慮する必要はないが)

		// Windowsの場合、複数のラインを同時に再生すると、なぜか getLongFramePosition() の値がときどき逆戻りする。

		return new TimeKeeper(new AbstractClock((int) line.getFormat().getFrameRate()) {
			protected long rawValue() {
				return line.getLongFramePosition();
			}
		});
	}

}
