/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.app.InjectorHolder;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.TASelector;
import ch.kuramo.javie.core.TextAnimator;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.services.ProjectDecoder;
import ch.kuramo.javie.core.services.ProjectEncoder;

import com.google.inject.Inject;

public class DuplicateTASelectorsOperation extends ProjectOperation {

	private final String _compId;

	private final List<Object[]> _srcData;

	private final String _dstLayerId;

	private final int _dstAnimatorIndex;

	private final int _dstIndex;

	@Inject
	private ProjectEncoder _encoder;

	@Inject
	private ProjectDecoder _decoder;


	public DuplicateTASelectorsOperation(ProjectManager pm,
			List<Object[]> selectors, TextLayer dstLayer, TextAnimator dstAnimator, int insertBefore) {

		super(pm, "セレクタの複製");
		InjectorHolder.getInjector().injectMembers(this);

		if (selectors.isEmpty()) {
			_noEffect = true;
			_compId = null;
			_srcData = null;
			_dstLayerId = null;
			_dstAnimatorIndex = 0;
			_dstIndex = 0;
			return;
		}

		LayerComposition comp = pm.checkLayer((TextLayer) selectors.get(0)[0]);
		if (dstLayer != null && pm.checkLayer(dstLayer) != comp) {
			throw new IllegalArgumentException();
		}

		_compId = comp.getId();
		_srcData = Util.newList();

		if (dstLayer != null) {
			_dstLayerId = dstLayer.getId();
			_dstAnimatorIndex = dstLayer.getTextAnimators().indexOf(dstAnimator);
			if (_dstAnimatorIndex == -1) {
				throw new IllegalArgumentException();
			}

			int dstSelectorCount = dstAnimator.getSelectors().size();
			if (insertBefore < 0 || insertBefore > dstSelectorCount) {
				_dstIndex = dstSelectorCount;
			} else {
				_dstIndex = insertBefore;
			}
		} else {
			_dstLayerId = null;
			_dstAnimatorIndex = 0;
			_dstIndex = 0;
		}

		for (Object[] o : selectors) {
			TextLayer srcLayer = (TextLayer) o[0];
			TextAnimator animator = (TextAnimator) o[1];
			TASelector selector = (TASelector) o[2];

			int layerIndex = comp.getLayers().indexOf(srcLayer);
			if (layerIndex == -1) {
				throw new IllegalArgumentException();
			}

			int animatorIndex = srcLayer.getTextAnimators().indexOf(animator);
			if (animatorIndex == -1) {
				throw new IllegalArgumentException();
			}

			int selectorIndex = animator.getSelectors().indexOf(selector);
			if (selectorIndex == -1) {
				throw new IllegalArgumentException();
			}

			_srcData.add(new Object[] { layerIndex, animatorIndex, selectorIndex,
					_encoder.encodeElement(selector) });
		}

		Collections.sort(_srcData, new Comparator<Object[]>() {
			public int compare(Object[] o1, Object[] o2) {
				int compare = (Integer)o1[0] - (Integer)o2[0];
				if (compare != 0) return compare;

				compare = (Integer)o1[1] - (Integer)o2[1];
				if (compare != 0) return compare;

				return (Integer)o2[2] - (Integer)o1[2];	// セレクタは逆順
			}
		});
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);

		List<Object[]> data = null;
		if (pm != null) {
			data = Util.newList();
		}

		Set<TASelector> newSelectors = Util.newSet();
		try {
			if (_dstLayerId != null) {
				TextLayer dstLayer = (TextLayer) comp.getLayer(_dstLayerId);
				TextAnimator dstAnimator = dstLayer.getTextAnimators().get(_dstAnimatorIndex);

				for (Object[] o : _srcData) {
					TASelector selector = _decoder.decodeElement((String)o[3], TASelector.class);
					// TASelector に afterDecode は無い。
					//selector.afterDecode(project);

					dstAnimator.getSelectors().add(_dstIndex, selector);
					newSelectors.add(selector);

					if (pm != null) {
						data.add(new Object[] { dstLayer, _dstAnimatorIndex, _dstIndex + data.size() });
						// FIXME 結果的にできるデータは正しいが _dstIndex + data.size() の部分がおかしい？
					}
				}

			} else {
				for (Object[] o : _srcData) {
					TASelector selector = _decoder.decodeElement((String)o[3], TASelector.class);
					// TASelector に afterDecode は無い。
					//selector.afterDecode(project);

					TextLayer layer = (TextLayer) comp.getLayers().get((Integer)o[0]);
					TextAnimator animator = layer.getTextAnimators().get((Integer)o[1]);
					animator.getSelectors().add((Integer)o[2]+1, selector);
					newSelectors.add(selector);

					if (pm != null) {
						data.add(new Object[] { layer, o[1], selector });
					}
				}
				if (pm != null) {
					for (Object[] o : data) {
						TextAnimator animator = ((TextLayer) o[0]).getTextAnimators().get((Integer)o[1]);
						o[2] = animator.getSelectors().indexOf(o[2]);
					}
				}
			}
		} catch (ProjectDecodeException e) {
			// 元の状態に戻す
			for (Layer layer : comp.getLayers()) {
				if (layer instanceof TextLayer) {
					for (TextAnimator animator : ((TextLayer) layer).getTextAnimators()) {
						animator.getSelectors().removeAll(newSelectors);
					}
				}
			}
			throw new ExecutionException("error decoding TASelector data", e);
		}

		if (pm != null) {
			pm.fireTASelectorsAdd(comp, data.toArray(new Object[data.size()][]));
		}

		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);

		if (_dstLayerId != null) {
			TextLayer dstLayer = (TextLayer) comp.getLayer(_dstLayerId);
			TextAnimator dstAnimator = dstLayer.getTextAnimators().get(_dstAnimatorIndex);
			dstAnimator.getSelectors().subList(_dstIndex, _dstIndex + _srcData.size()).clear();

			if (pm != null) {
				pm.fireTASelectorsRemove(comp, new Object[][] {
						new Object[] { dstLayer, _dstAnimatorIndex, -1 } });
			}

		} else {
			List<Object[]> data = null;
			if (pm != null) {
				data = Util.newList();
			}

			for (ListIterator<Object[]> it = _srcData.listIterator(_srcData.size()); it.hasPrevious(); ) {
				Object[] o = it.previous();

				TextLayer layer = (TextLayer) comp.getLayers().get((Integer)o[0]);
				TextAnimator animator = layer.getTextAnimators().get((Integer)o[1]);
				animator.getSelectors().remove((Integer)o[2]+1);

				if (pm != null) {
					data.add(new Object[] { layer, o[1], -1 });
				}
			}

			if (pm != null) {
				pm.fireTASelectorsRemove(comp, data.toArray(new Object[data.size()][]));
			}
		}

		return Status.OK_STATUS;
	}

}
