/*
 * Copyright (c) 2009,2010 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.app.project;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.ListIterator;

import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import ch.kuramo.javie.core.Effect;
import ch.kuramo.javie.core.EffectableLayer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.Util;

public class MoveEffectsOperation extends ProjectOperation {

	private final String _compId;

	private final List<int[]> _srcData;

	private final String _dstLayerId;

	private final int _dstIndex;


	public MoveEffectsOperation(ProjectManager pm,
			List<EffectableLayer> srcLayers, List<Effect> effects,
			EffectableLayer dstLayer, int moveBefore) {

		super(pm, "エフェクトの移動");

		if (srcLayers.size() != effects.size()) {
			throw new IllegalArgumentException();
		}

		if (srcLayers.isEmpty()) {
			_noEffect = true;
			_compId = null;
			_srcData = null;
			_dstLayerId = null;
			_dstIndex = 0;
			return;
		}

		LayerComposition comp = pm.checkLayer(srcLayers.get(0));
		if (pm.checkLayer(dstLayer) != comp) {
			throw new IllegalArgumentException();
		}

		_compId = comp.getId();
		_srcData = Util.newList();
		_dstLayerId = dstLayer.getId();

		int dstEffectCount = dstLayer.getEffects().size();
		int dstIndex;
		if (moveBefore < 0 || moveBefore > dstEffectCount) {
			dstIndex = moveBefore = dstEffectCount;
		} else {
			dstIndex = moveBefore;
		}

		for (ListIterator<EffectableLayer> it = srcLayers.listIterator(); it.hasNext(); ) {
			int i = it.nextIndex();
			EffectableLayer srcLayer = it.next();

			int layerIndex = comp.getLayers().indexOf(srcLayer);
			if (layerIndex == -1) {
				throw new IllegalArgumentException();
			}

			int effectIndex = srcLayer.getEffects().indexOf(effects.get(i));
			if (effectIndex == -1) {
				throw new IllegalArgumentException();
			}

			_srcData.add(new int[] { layerIndex, effectIndex });
			if (srcLayer == dstLayer && effectIndex < moveBefore) {
				--dstIndex;
			}
		}

		_dstIndex = dstIndex;

		Collections.sort(_srcData, new Comparator<int[]>() {
			public int compare(int[] o1, int[] o2) {
				int compare = o1[0] - o2[0];
				return (compare != 0) ? compare : o2[1] - o1[1]; // エフェクトは逆順
			}
		});

		List<Effect> tmp = Util.newList(dstLayer.getEffects());
		tmp.removeAll(effects);
		tmp.addAll(_dstIndex, effects);
		_noEffect = dstLayer.getEffects().equals(tmp);
	}

	@Override
	protected IStatus executeOrRedo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);
		EffectableLayer dstLayer = (EffectableLayer) comp.getLayer(_dstLayerId);

		List<Object[]> data1 = null, data2 = null;
		if (pm != null) {
			data1 = Util.newList();
			data2 = Util.newList();
		}

		List<Effect> effects = Util.newList();

		for (int[] srcData : _srcData) {
			EffectableLayer srcLayer = (EffectableLayer) comp.getLayers().get(srcData[0]);
			effects.add(0, srcLayer.getEffects().remove(srcData[1]));
			if (pm != null) {
				data1.add(new Object[] { srcLayer, -1 } );
				data2.add(new Object[] { dstLayer, _dstIndex + data2.size() });
			}
		}

		if (pm != null) {
			pm.fireEffectsRemove(comp, data1.toArray(new Object[data1.size()][]));
		}

		dstLayer.getEffects().addAll(_dstIndex, effects);
		if (pm != null) {
			pm.fireEffectsAdd(comp, data2.toArray(new Object[data2.size()][]));
		}

		return Status.OK_STATUS;
	}

	@Override
	protected IStatus undo(IProgressMonitor monitor, IAdaptable info,
			Project project, ProjectManager pm) throws ExecutionException {

		LayerComposition comp = project.getComposition(_compId);
		EffectableLayer dstLayer = (EffectableLayer) comp.getLayer(_dstLayerId);

		List<Object[]> data = null;
		if (pm != null) {
			data = Util.newList();
		}

		List<Effect> sublist = dstLayer.getEffects().subList(_dstIndex, _dstIndex + _srcData.size());
		List<Effect> effects = Util.newList(sublist);
		sublist.clear();

		if (pm != null) {
			pm.fireEffectsRemove(comp, new Object[][] { new Object[] { dstLayer, -1 } });
		}

		for (ListIterator<int[]> it = _srcData.listIterator(_srcData.size()); it.hasPrevious(); ) {
			int[] srcData = it.previous();
			EffectableLayer srcLayer = (EffectableLayer) comp.getLayers().get(srcData[0]);
			srcLayer.getEffects().add(srcData[1], effects.remove(0));
			if (pm != null) {
				data.add(new Object[] { srcLayer, srcData[1] });
			}
		}
		if (pm != null) {
			pm.fireEffectsAdd(comp, data.toArray(new Object[data.size()][]));
		}

		return Status.OK_STATUS;
	}

}
