/*
 * Copyright (c) 2009-2012 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.core.internal;

import java.util.List;
import java.util.ListIterator;

import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IAudioBuffer;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Time;
import ch.kuramo.javie.core.AbstractComposition;
import ch.kuramo.javie.core.AbstractLayer;
import ch.kuramo.javie.core.CoreContext;
import ch.kuramo.javie.core.ExpressionScope;
import ch.kuramo.javie.core.Layer;
import ch.kuramo.javie.core.LayerComposition;
import ch.kuramo.javie.core.LayerNature;
import ch.kuramo.javie.core.MediaItemLayer;
import ch.kuramo.javie.core.MediaLayer;
import ch.kuramo.javie.core.Project;
import ch.kuramo.javie.core.ProjectDecodeException;
import ch.kuramo.javie.core.SuperSampling;
import ch.kuramo.javie.core.TextLayer;
import ch.kuramo.javie.core.TrackMatte;
import ch.kuramo.javie.core.Util;
import ch.kuramo.javie.core.VideoLayerRenderer;
import ch.kuramo.javie.core.WrappedOperation;
import ch.kuramo.javie.core.AbstractLayer.LayerExpressionElement;
import ch.kuramo.javie.core.VideoLayerRenderer.AbstractMatteLayerRenderers;
import ch.kuramo.javie.core.VideoLayerRenderer.CRLayerRenderer;
import ch.kuramo.javie.core.VideoLayerRenderer.GeneralLayerRenderer;
import ch.kuramo.javie.core.VideoLayerRenderer.MatteLayerRenderers;
import ch.kuramo.javie.core.VideoLayerRenderer.NormalLayerRenderer;
import ch.kuramo.javie.core.annotations.ProjectElement;
import ch.kuramo.javie.core.services.AudioRenderSupport;
import ch.kuramo.javie.core.services.RenderContext;

import com.google.inject.Inject;
import com.google.inject.Injector;

@ProjectElement("layerComposition")
public class LayerCompositionImpl extends AbstractComposition implements LayerComposition {

	private boolean _shyEnabled = false;

	private boolean _frameBlendEnabled = false;

	private boolean _motionBlurEnabled = false;

	private double _motionBlurShutterAngle = 180;

	private double _motionBlurShutterPhase = -90;

	private int _motionBlurSamples = 16;

	private SuperSampling _superSampling = SuperSampling.SS_NONE;

	private List<Layer> _layers = Util.newList();

	@Inject
	private RenderContext _context;

	@Inject
	private AudioRenderSupport _arSupport;

	@Inject
	private Injector _injector;


	public LayerCompositionImpl() {
		super();
	}

	public boolean isShyEnabled() {
		return _shyEnabled;
	}

	public void setShyEnabled(boolean shyEnabled) {
		_shyEnabled = shyEnabled;
	}

	public boolean isFrameBlendEnabled() {
		return _frameBlendEnabled;
	}

	public void setFrameBlendEnabled(boolean frameBlendEnabled) {
		_frameBlendEnabled = frameBlendEnabled;
	}

	public boolean isMotionBlurEnabled() {
		return _motionBlurEnabled;
	}

	public void setMotionBlurEnabled(boolean motionBlurEnabled) {
		_motionBlurEnabled = motionBlurEnabled;
	}

	public double getMotionBlurShutterAngle() {
		return _motionBlurShutterAngle;
	}

	public void setMotionBlurShutterAngle(double motionBlurShutterAngle) {
		_motionBlurShutterAngle = motionBlurShutterAngle;
	}

	public double getMotionBlurShutterPhase() {
		return _motionBlurShutterPhase;
	}

	public void setMotionBlurShutterPhase(double motionBlurShutterPhase) {
		_motionBlurShutterPhase = motionBlurShutterPhase;
	}

	public int getMotionBlurSamples() {
		return _motionBlurSamples;
	}

	public void setMotionBlurSamples(int motionBlurSamples) {
		_motionBlurSamples = motionBlurSamples;
	}

	public SuperSampling getSuperSampling() {
		return _superSampling;
	}

	public void setSuperSampling(SuperSampling superSampling) {
		_superSampling = superSampling;
	}

	public List<Layer> getLayers() {
		return _layers;
	}

	public void setLayers(List<Layer> layers) {
		_layers = layers;
	}

	public Layer getLayer(String id) {
		for (Layer layer : _layers) {
			if (layer.getId().equals(id)) {
				return layer;
			}
		}
		return null;
	}

	public Layer getParentLayer(Layer layer) {
		String parentId = layer.getParentId();
		return (parentId != null) ? getLayer(parentId) : null;
	}

	public void setParentLayer(Layer layer, Layer parent) {
		// 引数のレイヤーと親がどちらもこのコンポジションに属しているかどうかのチェックをすべきか？
		layer.setParentId((parent != null) ? parent.getId() : null);
	}

	public void afterDecode(Project p) throws ProjectDecodeException {
		for (Layer layer : _layers) {
			layer.afterDecode(p, this);
		}
	}

	public void prepareExpression(ExpressionScope scope) {
		scope.putExpressionElement("thisComp", this);

		for (Layer layer : _layers) {
			layer.prepareExpression(scope.clone());
		}
	}

	public IVideoBuffer renderVideoFrame() {
		_context.setComposition(this);

		ColorMode colorMode = getColorMode();
		_context.setColorMode(colorMode);

		Time time = _context.getTime();

		if (isFrameDurationPreserved()) {
			Time frameDuration = getFrameDuration();
			time = Time.fromFrameNumber(time.toFrameNumber(frameDuration), frameDuration);
			_context.setTime(time);
			_context.setVideoFrameDuration(frameDuration);
		}

		_context.setCamera(_injector.getInstance(CameraImpl.class));
		_context.setLights(_injector.getInstance(LightSet.class).getLights());

		VideoLayerComposer composer = _injector.getInstance(VideoLayerComposer.class);

		for (int i = 0, n = _layers.size(); i < n; ++i) {
			Layer layer = _layers.get(i);

			if (!LayerNature.isVideoEnabled(layer)) {
				continue;
			}

			if (layer.getInPoint().after(time) || !layer.getOutPoint().after(time)) {
				continue;
			}

			if (layer instanceof MediaLayer) {
				MediaLayer mediaLayer = (MediaLayer) layer;
				MediaLayer matteLayer = null;

				if (mediaLayer.getTrackMatte() != TrackMatte.NONE && i+1 < n) {
					Layer nextLayer = _layers.get(i+1);
					if (nextLayer instanceof MediaLayer) {
						matteLayer = (MediaLayer) nextLayer;
					}
				}

				composer.addLayerRenderers(createVideoRenderers(mediaLayer, matteLayer));
			}
		}

		return composer.compose();
	}

	public List<VideoLayerRenderer> createCollapsedRenderers() {
		List<VideoLayerRenderer> list = Util.newList();

		_context.setComposition(this);

		Time time = _context.getTime();

		for (int i = 0, n = _layers.size(); i < n; ++i) {
			Layer layer = _layers.get(i);

			if (!LayerNature.isVideoEnabled(layer)) {
				continue;
			}

			if (layer.getInPoint().after(time) || !layer.getOutPoint().after(time)) {
				continue;
			}

			if (layer instanceof MediaLayer) {
				MediaLayer mediaLayer = (MediaLayer) layer;
				MediaLayer matteLayer = null;

				if (mediaLayer.getTrackMatte() != TrackMatte.NONE && i+1 < n) {
					Layer nextLayer = _layers.get(i+1);
					if (nextLayer instanceof MediaLayer) {
						matteLayer = (MediaLayer) nextLayer;
					}
				}

				list.addAll(createVideoRenderers(mediaLayer, matteLayer));
			}
		}

		return list;
	}

	private List<VideoLayerRenderer> createVideoRenderers(final MediaLayer layer, MediaLayer matteLayer) {
		List<VideoLayerRenderer> renderers = null;

		if (layer instanceof MediaItemLayer) {
			final MediaItemLayer miLayer = (MediaItemLayer) layer;

			if (LayerNature.isCTCR(miLayer) && miLayer.isPrecompositionLayer()) {
				renderers = _context.saveAndExecute(new WrappedOperation<List<VideoLayerRenderer>>() {
					public List<VideoLayerRenderer> execute() {
						return miLayer.createCollapsedRenderers();
					}
				});
			}
		} else if (layer instanceof TextLayer) {
			final TextLayer textLayer = (TextLayer) layer;

			if (LayerNature.isThreeD(textLayer) && textLayer.isPerCharacter3D()) {
				renderers = _context.saveAndExecute(new WrappedOperation<List<VideoLayerRenderer>>() {
					public List<VideoLayerRenderer> execute() {
						return textLayer.createPerCharacter3DRenderers();
					}
				});
			}
		}

		if (renderers == null) {
			GeneralLayerRenderer r = _context.saveAndExecute(new WrappedOperation<GeneralLayerRenderer>() {
				public GeneralLayerRenderer execute() {
					return layer.createVideoRenderer();
				}
			});

			// このリストは、コラップスした場合にListIterator経由でsetされるので、
			// Collections.singletonList/emptyList を使ってはいけない。
			renderers = Util.newList();
			if (r != null) {
				renderers.add(r);
			}
		}

		if (!renderers.isEmpty() && matteLayer != null) {
			final List<VideoLayerRenderer> matteRenderers = createVideoRenderers(matteLayer, null);
			if (!matteRenderers.isEmpty()) {
				MatteLayerRenderers mlr = new AbstractMatteLayerRenderers() {
					public MediaLayer getLayer() { return layer; }
					public List<VideoLayerRenderer> getRenderers() { return matteRenderers; }
				};

				for (VideoLayerRenderer r : renderers) {
					if (r instanceof GeneralLayerRenderer) {
						((GeneralLayerRenderer) r).addMatteId(mlr.getMatteId());
					}
				}

				renderers.add(mlr);
			}
		}

		return renderers;
	}

	public IVideoBuffer getLayerVideoFrame(Layer layer, boolean rawSource) {
		if (_context.getComposition() != this) {
			throw new IllegalStateException("this composition is not the context composition.");
		}

		if (layer == null) {
			throw new NullPointerException();
		}

		if (!_layers.contains(layer)) {
			throw new IllegalArgumentException("no such layer found in this LayerComposition");
		}

		if (!LayerNature.isVideoNature(layer)) {
			return null;
		}

		if (rawSource) {
			if (layer instanceof MediaLayer) {
				final MediaLayer ml = (MediaLayer) layer;
				
				return _context.saveAndExecute(new WrappedOperation<IVideoBuffer>() {
					public IVideoBuffer execute() {
						GeneralLayerRenderer r = ml.createVideoRenderer();
						if (r instanceof NormalLayerRenderer) {
							return ((NormalLayerRenderer) r).render(false, _frameBlendEnabled);
						} else {
							return ((CRLayerRenderer) r).render();
						}
					}
				});				
			}
		} else {
			Time time = _context.getTime();
			if (layer.getInPoint().after(time) || !layer.getOutPoint().after(time)) {
				return null;
			}

			if (layer instanceof MediaItemLayer) {
				final MediaItemLayer miLayer = (MediaItemLayer) layer;

				if (LayerNature.isCTCR(miLayer) && miLayer.isPrecompositionLayer()) {
					return _context.saveAndExecute(new WrappedOperation<IVideoBuffer>() {
						public IVideoBuffer execute() {
							_context.setCamera(_injector.getInstance(CameraImpl.class));

							VideoLayerComposer composer = _injector.getInstance(VideoLayerComposer.class);
							composer.addLayerRenderers(miLayer.createCollapsedRenderers());
							return composer.compose();
						}
					});
				}
			} else if (layer instanceof TextLayer) {
				final TextLayer textLayer = (TextLayer) layer;

				if (LayerNature.isThreeD(textLayer) && textLayer.isPerCharacter3D()) {
					return _context.saveAndExecute(new WrappedOperation<IVideoBuffer>() {
						public IVideoBuffer execute() {
							_context.setCamera(_injector.getInstance(CameraImpl.class));

							VideoLayerComposer composer = _injector.getInstance(VideoLayerComposer.class);
							composer.addLayerRenderers(textLayer.createPerCharacter3DRenderers());
							return composer.compose();
						}
					});
				}
			}

			if (layer instanceof MediaLayer) {
				final MediaLayer ml = (MediaLayer) layer;

				return _context.saveAndExecute(new WrappedOperation<IVideoBuffer>() {
					public IVideoBuffer execute() {
						_context.setCamera(_injector.getInstance(CameraImpl.class));

						VideoLayerComposer composer = _injector.getInstance(VideoLayerComposer.class);
						composer.addLayerRenderer(ml.createVideoRenderer());
						return composer.compose();
					}
				});
			}
		}

		return null;
	}

	public IAudioBuffer renderAudioChunk() {
		_context.setComposition(this);

		ColorMode colorMode = getColorMode();
		_context.setColorMode(colorMode);

		if (isFrameDurationPreserved()) {
			_context.setVideoFrameDuration(getFrameDuration());
		}

		IAudioBuffer ab = null;
		Time time = _context.getTime();
		Time chunkDuration = Time.fromFrameNumber(
				_context.getAudioFrameCount(), _context.getAudioMode().sampleDuration);

		for (Layer layer : _layers) {
			if (!LayerNature.isAudioEnabled(layer)) {
				continue;
			}

			if (layer.getInPoint().after(time.add(chunkDuration)) || !layer.getOutPoint().after(time)) {
				continue;
			}

			if (layer instanceof MediaLayer) {
				final MediaLayer al = (MediaLayer) layer;

				IAudioBuffer alab = _context.saveAndExecute(new WrappedOperation<IAudioBuffer>() {
					public IAudioBuffer execute() {
						return al.renderAudioChunk(false);
					}
				});

				if (alab != null) {
					if (ab == null) {
						ab = alab;
					} else {
						IAudioBuffer abOld = ab;
						try {
							ab = _arSupport.mix(ab, alab);
						} finally {
							if (ab != abOld) {
								abOld.dispose();
							}
							if (ab != alab) {
								alab.dispose();
							}
						}
					}
				}
			}
		}

		return ab;
	}

	public IAudioBuffer getLayerAudioChunk(Layer layer, final boolean rawSource) {
		if (_context.getComposition() != this) {
			throw new IllegalStateException("this composition is not the context composition.");
		}

		if (layer == null) {
			throw new NullPointerException();
		}

		if (!_layers.contains(layer)) {
			throw new IllegalArgumentException("no such layer found in this LayerComposition");
		}

		if (!LayerNature.isAudioNature(layer)) {
			return null;
		}

		if (!rawSource) {
			Time time = _context.getTime();
			Time chunkDuration = Time.fromFrameNumber(
					_context.getAudioFrameCount(), _context.getAudioMode().sampleDuration);
			if (layer.getInPoint().after(time.add(chunkDuration)) || !layer.getOutPoint().after(time)) {
				return null;
			}
		}

		if (layer instanceof MediaLayer) {
			final MediaLayer ml = (MediaLayer) layer;

			return _context.saveAndExecute(new WrappedOperation<IAudioBuffer>() {
				public IAudioBuffer execute() {
					return ml.renderAudioChunk(rawSource);
				}
			});
		}

		return null;
	}

	public Object createExpressionElement(CoreContext context) {
		return new LayerCompositionExpressionElement(context);
	}

	public class LayerCompositionExpressionElement {

		protected final CoreContext context;

		public LayerCompositionExpressionElement(CoreContext context) {
			this.context = context;
		}

		public Object layer(int index) {
			Layer layer = _layers.get(_layers.size() - index);
			return context.getExpressionElement(layer);
		}

		public Object layer(String name) {
			for (ListIterator<Layer> it = _layers.listIterator(_layers.size()); it.hasPrevious(); ) {
				Layer layer = it.previous();
				if (layer.getName().equals(name)) {
					return context.getExpressionElement(layer);
				}
			}
			return null;
		}

		public int layerIndex(LayerExpressionElement layerExprElem) {
			int index = _layers.indexOf(AbstractLayer.getLayer(layerExprElem));
			return _layers.size() - index;
		}

		public int getWidth() {
			return getSize().width;
		}

		public int getHeight() {
			return getSize().height;
		}

		public double getFrameDuration() {
			return LayerCompositionImpl.this.getFrameDuration().toSecond();
		}

		public double getDuration() {
			return LayerCompositionImpl.this.getDuration().toSecond();
		}

	}

}
