/*
 * Copyright (c) 2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "stdafx.h"
#include "MediaFoundationSource.h"

#ifndef INITGUID
#define INITGUID
#endif
#include <Guiddef.h>
#include <evr.h>


#define RELEASE(p)						do { if (p) { p->Release(); p = NULL; } } while (false)
#define RETURN_HRESULT_IF_FAILED(hr)	do { HRESULT _hr = (hr); if (FAILED(_hr)) return _hr; } while (false)
#define BAIL_IF_FAILED(label, hr)		do { if (FAILED((hr))) goto label; } while (false)


IDirect3DDeviceManager9* VideoSource::sD3DManager;
UINT VideoSource::sResetToken;

int VideoSource::InitDXVA2(HWND hwnd)
{
	if (sD3DManager) return S_OK;

	IDirect3D9* d3d = NULL;
	IDirect3DDevice9* d3dDevice = NULL;
	UINT resetToken;
	IDirect3DDeviceManager9* d3dManager = NULL;

	d3d = Direct3DCreate9(D3D_SDK_VERSION);
	if (d3d == NULL) return E_FAIL;

	D3DPRESENT_PARAMETERS d3dpp;
	ZeroMemory(&d3dpp, sizeof(d3dpp));
	d3dpp.Windowed = TRUE;
	d3dpp.SwapEffect = D3DSWAPEFFECT_DISCARD;
	d3dpp.BackBufferFormat = D3DFMT_UNKNOWN;

	HRESULT hr;

	BAIL_IF_FAILED(bail,
		hr = d3d->CreateDevice(D3DADAPTER_DEFAULT, D3DDEVTYPE_HAL, hwnd,
			D3DCREATE_MIXED_VERTEXPROCESSING, &d3dpp, &d3dDevice));

	BAIL_IF_FAILED(bail,
		hr = DXVA2CreateDirect3DDeviceManager9(&resetToken, &d3dManager));

	BAIL_IF_FAILED(bail,
		hr = d3dManager->ResetDevice(d3dDevice, resetToken));

	d3dManager->AddRef();
	sD3DManager = d3dManager;
	sResetToken = resetToken;

	hr = S_OK;

bail:
	RELEASE(d3dManager);
	RELEASE(d3dDevice);
	RELEASE(d3d);
	return hr;
}

VideoSource::VideoSource(BOOL sysMem)
	: mWidth(0), mHeight(0), mTopDown(FALSE), mPixAspectRatio(),
	  mDuration(0), mFrameRate(), mReader(NULL), mDevHandle(0), mSysMem(sysMem)
{
}

VideoSource::~VideoSource()
{
	if (mDevHandle != 0) {
		HRESULT hr = sD3DManager->TestDevice(mDevHandle);
		if (hr == S_OK || hr == DXVA2_E_NEW_VIDEO_DEVICE) {
			sD3DManager->CloseDeviceHandle(mDevHandle);
			mDevHandle = 0;
		}
	}

	RELEASE(mReader);
}

HRESULT VideoSource::InitWithFile(LPCWSTR file)
{
	HRESULT hr;

	IMFAttributes* attrs = NULL;
	IMFSourceReader* reader = NULL;
	IMFMediaType* mediaType = NULL;

	PROPVARIANT var;
	PropVariantInit(&var);


	// SourceReader̍쐬
	BAIL_IF_FAILED(bail,
		hr = MFCreateAttributes(&attrs, 2));

	BAIL_IF_FAILED(bail,
		hr = attrs->SetUnknown(MF_SOURCE_READER_D3D_MANAGER, sD3DManager));

	BAIL_IF_FAILED(bail,
		hr = attrs->SetUINT32(MF_SOURCE_READER_DISABLE_DXVA, FALSE));

	BAIL_IF_FAILED(bail,
		hr = MFCreateSourceReaderFromURL(file, attrs, &reader));


	// V[N\ǂ
	BAIL_IF_FAILED(bail,
		hr = reader->GetPresentationAttribute(MF_SOURCE_READER_MEDIASOURCE,
							MF_SOURCE_READER_MEDIASOURCE_CHARACTERISTICS, &var));
	ULONG flags;
	BAIL_IF_FAILED(bail,
		hr = PropVariantToUInt32(var, &flags));
	if (!(flags & MFMEDIASOURCE_CAN_SEEK) || (flags & MFMEDIASOURCE_HAS_SLOW_SEEK)) {
		hr = E_FAIL;
		goto bail;
	}


	// ŏ̃rfIXg[݂̂I
	BAIL_IF_FAILED(bail,
		hr = reader->SetStreamSelection(MF_SOURCE_READER_ALL_STREAMS, FALSE));

	BAIL_IF_FAILED(bail,
		hr = reader->SetStreamSelection(MF_SOURCE_READER_FIRST_VIDEO_STREAM, TRUE));


	// fBA^Cv̐ݒ
	BAIL_IF_FAILED(bail,
		hr = MFCreateMediaType(&mediaType));

	BAIL_IF_FAILED(bail,
		hr = mediaType->SetGUID(MF_MT_MAJOR_TYPE, MFMediaType_Video));

	BAIL_IF_FAILED(bail,
		hr = mediaType->SetGUID(MF_MT_SUBTYPE, MFVideoFormat_YUY2));

	BAIL_IF_FAILED(bail,
		hr = reader->SetCurrentMediaType(MF_SOURCE_READER_FIRST_VIDEO_STREAM, NULL, mediaType));


	RELEASE(mediaType);
	BAIL_IF_FAILED(bail,
		hr = reader->GetCurrentMediaType(MF_SOURCE_READER_FIRST_VIDEO_STREAM, &mediaType));

	// ƍ
	BAIL_IF_FAILED(bail,
		hr = MFGetAttributeSize(mediaType, MF_MT_FRAME_SIZE, &mWidth, &mHeight));

	// ㉺]Ă邩ǂ
	LONG stride = (LONG)MFGetAttributeUINT32(mediaType, MF_MT_DEFAULT_STRIDE, 1);
	mTopDown = (stride > 0);

	// sNZAXyNg
	hr = MFGetAttributeRatio(mediaType, MF_MT_PIXEL_ASPECT_RATIO,
							(UINT32*)&mPixAspectRatio.Numerator, (UINT32*)&mPixAspectRatio.Denominator);
	if (FAILED(hr)) {
		mPixAspectRatio.Numerator = 1;
		mPixAspectRatio.Denominator = 1;
	}

	// t[[g
	BAIL_IF_FAILED(bail,
		hr = MFGetAttributeRatio(mediaType, MF_MT_FRAME_RATE,
								(UINT32*)&mFrameRate.Numerator, (UINT32*)&mFrameRate.Denominator));

	// f[V
	BAIL_IF_FAILED(bail,
		hr = reader->GetPresentationAttribute(MF_SOURCE_READER_MEDIASOURCE, MF_PD_DURATION, &var));
	if (var.vt != VT_UI8) {
		hr = E_FAIL;
		goto bail;
	}
	mDuration = var.hVal.QuadPart;


	// ŏ̃rfIXg[xI
	// iTv\[Xł͂̂悤ɂĂ邪A̕Kv͂悭킩Ȃj
	BAIL_IF_FAILED(bail,
		hr = reader->SetStreamSelection(MF_SOURCE_READER_FIRST_VIDEO_STREAM, TRUE));


	reader->AddRef();
	mReader = reader;

	hr = S_OK;

bail:
	PropVariantClear(&var);
	RELEASE(mediaType);
	RELEASE(reader);
	RELEASE(attrs);
	return hr;
}

HRESULT VideoSource::FrameSurfaceAtTime(LONGLONG* ioTime, LONGLONG limitTime, IDirect3DSurface9** outSurface)
{
	HRESULT hr;

	IMFSample* sample = NULL;
	IMFMediaBuffer* medbuf = NULL;
	IDirect3DSurface9* surface1 = NULL;
	IDirect3DSurface9* surface2 = NULL;
	IDirect3DSurface9* surface3 = NULL;
	IDirect3DDevice9* d3dDevice = NULL;

	BAIL_IF_FAILED(bail,
		hr = LockDevice(&d3dDevice));

	if (*ioTime != 0x8000000000000000LL) {
		PROPVARIANT var;
		PropVariantInit(&var);

		var.vt = VT_I8;
		var.hVal.QuadPart = *ioTime;
		hr = mReader->SetCurrentPosition(GUID_NULL, var);

		PropVariantClear(&var);
		BAIL_IF_FAILED(bail, hr);
	}

	DWORD flags;
	LONGLONG actualTime;
	BAIL_IF_FAILED(bail,
		hr = mReader->ReadSample(MF_SOURCE_READER_FIRST_VIDEO_STREAM, 
								0, NULL, &flags, &actualTime, &sample));
	if (flags & MF_SOURCE_READERF_ENDOFSTREAM) {
		hr = E_FAIL;
		goto bail;
	}

	if (actualTime < limitTime || outSurface == NULL) {
		if (outSurface != NULL) {
			*outSurface = NULL;
		}
		*ioTime = actualTime;
		hr = S_FALSE;
		goto bail;
	}

	BAIL_IF_FAILED(bail,
		hr = sample->GetBufferByIndex(0, &medbuf));

	BAIL_IF_FAILED(bail,
		hr = MFGetService(medbuf, MR_BUFFER_SERVICE, IID_PPV_ARGS(&surface1)));

	if (mSysMem) {
		BAIL_IF_FAILED(bail,
			hr = d3dDevice->CreateRenderTarget(
						mWidth, mHeight, D3DFMT_A8R8G8B8, D3DMULTISAMPLE_NONE, 0, FALSE, &surface2, NULL));
	} else {
		BAIL_IF_FAILED(bail,
			hr = d3dDevice->CreateOffscreenPlainSurface(
						mWidth, mHeight, D3DFMT_A8R8G8B8, D3DPOOL_DEFAULT, &surface2, NULL));
	}

	RECT rect;
	SetRect(&rect, 0, 0, mWidth, mHeight);
	BAIL_IF_FAILED(bail,
		hr = d3dDevice->StretchRect(surface1, &rect, surface2, &rect, D3DTEXF_POINT));

	if (mSysMem) {
		BAIL_IF_FAILED(bail,
			hr = d3dDevice->CreateOffscreenPlainSurface(
						mWidth, mHeight, D3DFMT_A8R8G8B8, D3DPOOL_SYSTEMMEM, &surface3, NULL));

		BAIL_IF_FAILED(bail,
			hr = d3dDevice->GetRenderTargetData(surface2, surface3));

	} else {
		surface3 = surface2;
		surface2 = NULL;
	}

	surface3->AddRef();
	*outSurface = surface3;
	*ioTime = actualTime;

	hr = S_OK;

bail:
	RELEASE(surface3);
	RELEASE(surface2);
	RELEASE(surface1);
	RELEASE(medbuf);
	RELEASE(sample);
	UnlockDevice(d3dDevice);
	return hr;
}

HRESULT VideoSource::LockSurface(IDirect3DSurface9* surface, D3DLOCKED_RECT* outLockedRect)
{
	// foCXbNȂƕVideoSourceiʃXbhŁjgpƂɃNbVB

	HRESULT hr;
	IDirect3DDevice9* d3dDevice = NULL;

	BAIL_IF_FAILED(bail,
		hr = LockDevice(&d3dDevice));

	BAIL_IF_FAILED(bail,
		hr = surface->LockRect(outLockedRect, NULL, D3DLOCK_READONLY | D3DLOCK_NOSYSLOCK));

	surface->AddRef();

	hr = S_OK;

bail:
	UnlockDevice(d3dDevice);
	return hr;
}

HRESULT VideoSource::UnlockSurface(IDirect3DSurface9* surface)
{
	// UnlockRectƂɃfoCXbNĂKv邩ǂsB
	// LockRect̂Ƃ̓foCXbNKv̂ŁAłꉞbNĂB
	// ܂Asurface->Release(); ŎQƃJEg[ɂȂƂ̓foCX̃bNKvB

	HRESULT hr;
	IDirect3DDevice9* d3dDevice = NULL;

	BAIL_IF_FAILED(bail,
		hr = LockDevice(&d3dDevice));

	BAIL_IF_FAILED(bail,
		hr = surface->UnlockRect());

	surface->Release();

	hr = S_OK;

bail:
	UnlockDevice(d3dDevice);
	return hr;
}

HRESULT VideoSource::ReleaseSurface(IDirect3DSurface9* surface)
{
	HRESULT hr;
	IDirect3DDevice9* d3dDevice = NULL;

	BAIL_IF_FAILED(bail,
		hr = LockDevice(&d3dDevice));

	surface->Release();

	hr = S_OK;

bail:
	UnlockDevice(d3dDevice);
	return hr;
}

HRESULT VideoSource::LockDevice(IDirect3DDevice9** outDevice)
{
	HRESULT hr;

	if (mDevHandle != 0) {
		hr = sD3DManager->TestDevice(mDevHandle);
		if (hr == DXVA2_E_NEW_VIDEO_DEVICE) {
			sD3DManager->CloseDeviceHandle(mDevHandle);
			mDevHandle = 0;
		}
	}

	if (mDevHandle == 0) {
		BAIL_IF_FAILED(bail,
			hr = sD3DManager->OpenDeviceHandle(&mDevHandle));
	}

	BAIL_IF_FAILED(bail,
		hr = sD3DManager->LockDevice(mDevHandle, outDevice, TRUE));

	hr = S_OK;

bail:
	return hr;
}

HRESULT VideoSource::UnlockDevice(IDirect3DDevice9* device)
{
	if (device) {
		return sD3DManager->UnlockDevice(mDevHandle, FALSE);
	} else {
		return S_FALSE;
	}
}

AudioSource::AudioSource()
	: mDuration(0), mSampleRate(0), mSampleSizeInBits(0),
	  mChannels(0), mFloat(FALSE), mReader(NULL)
{
}

AudioSource::~AudioSource()
{
	RELEASE(mReader);
}

HRESULT AudioSource::InitWithFile(LPCWSTR file)
{
	HRESULT hr;

	IMFSourceReader* reader = NULL;
	IMFMediaType* mediaType = NULL;

	PROPVARIANT var;
	PropVariantInit(&var);


	// SourceReader̍쐬
	BAIL_IF_FAILED(bail,
		hr = MFCreateSourceReaderFromURL(file, NULL, &reader));


	// V[N\ǂ
	BAIL_IF_FAILED(bail,
		hr = reader->GetPresentationAttribute(MF_SOURCE_READER_MEDIASOURCE,
							MF_SOURCE_READER_MEDIASOURCE_CHARACTERISTICS, &var));
	ULONG flags;
	BAIL_IF_FAILED(bail,
		hr = PropVariantToUInt32(var, &flags));
	if (!(flags & MFMEDIASOURCE_CAN_SEEK) || (flags & MFMEDIASOURCE_HAS_SLOW_SEEK)) {
		hr = E_FAIL;
		goto bail;
	}


	// ŏ̃I[fBIXg[݂̂I
	BAIL_IF_FAILED(bail,
		hr = reader->SetStreamSelection(MF_SOURCE_READER_ALL_STREAMS, FALSE));

	BAIL_IF_FAILED(bail,
		hr = reader->SetStreamSelection(MF_SOURCE_READER_FIRST_AUDIO_STREAM, TRUE));


	// IEEE float ǂ̃`FbN
	BAIL_IF_FAILED(bail,
		hr = reader->GetCurrentMediaType(MF_SOURCE_READER_FIRST_AUDIO_STREAM, &mediaType));

	GUID tmpGUID;
	hr = mediaType->GetGUID(MF_MT_MAJOR_TYPE, &tmpGUID);
	if (SUCCEEDED(hr) && IsEqualGUID(tmpGUID, MFMediaType_Audio)) {
		hr = mediaType->GetGUID(MF_MT_SUBTYPE, &tmpGUID);
		if (SUCCEEDED(hr) && IsEqualGUID(tmpGUID, MFAudioFormat_Float)) {
			mFloat = TRUE;
		}
	}

	// fBA^Cv̐ݒ
	RELEASE(mediaType);
	BAIL_IF_FAILED(bail,
		hr = MFCreateMediaType(&mediaType));

	BAIL_IF_FAILED(bail,
		hr = mediaType->SetGUID(MF_MT_MAJOR_TYPE, MFMediaType_Audio));

	BAIL_IF_FAILED(bail,
		hr = mediaType->SetGUID(MF_MT_SUBTYPE, mFloat ? MFAudioFormat_Float : MFAudioFormat_PCM));

	BAIL_IF_FAILED(bail,
		hr = reader->SetCurrentMediaType(MF_SOURCE_READER_FIRST_AUDIO_STREAM, NULL, mediaType));


	RELEASE(mediaType);
	BAIL_IF_FAILED(bail,
		hr = reader->GetCurrentMediaType(MF_SOURCE_READER_FIRST_AUDIO_STREAM, &mediaType));

	// Tv[g
	BAIL_IF_FAILED(bail,
		hr = mediaType->GetUINT32(MF_MT_AUDIO_SAMPLES_PER_SECOND, &mSampleRate));

	// TvTCY
	BAIL_IF_FAILED(bail,
		hr = mediaType->GetUINT32(MF_MT_AUDIO_BITS_PER_SAMPLE, &mSampleSizeInBits));

	// `l
	BAIL_IF_FAILED(bail,
		hr = mediaType->GetUINT32(MF_MT_AUDIO_NUM_CHANNELS, &mChannels));

	// f[V
	BAIL_IF_FAILED(bail,
		hr = reader->GetPresentationAttribute(MF_SOURCE_READER_MEDIASOURCE, MF_PD_DURATION, &var));
	if (var.vt != VT_UI8) {
		hr = E_FAIL;
		goto bail;
	}
	mDuration = var.hVal.QuadPart;


	// ŏ̃I[fBIXg[xI
	// iTv\[Xł͂̂悤ɂĂ邪A̕Kv͂悭킩Ȃj
	BAIL_IF_FAILED(bail,
		hr = reader->SetStreamSelection(MF_SOURCE_READER_FIRST_AUDIO_STREAM, TRUE));


	reader->AddRef();
	mReader = reader;

	hr = S_OK;

bail:
	PropVariantClear(&var);
	RELEASE(mediaType);
	RELEASE(reader);
	return hr;
}

HRESULT AudioSource::ReadBufferAtTime(LONGLONG* ioTime, IMFMediaBuffer** outMedbuf, DWORD* outLen)
{
	HRESULT hr;

	IMFSample* sample = NULL;
	IMFMediaBuffer* medbuf = NULL;

	if (*ioTime != 0x8000000000000000LL) {
		PROPVARIANT var;
		PropVariantInit(&var);

		var.vt = VT_I8;
		var.hVal.QuadPart = *ioTime;
		hr = mReader->SetCurrentPosition(GUID_NULL, var);

		PropVariantClear(&var);
		BAIL_IF_FAILED(bail, hr);
	}

	DWORD flags;
	LONGLONG actualTime;
	BAIL_IF_FAILED(bail,
		hr = mReader->ReadSample(MF_SOURCE_READER_FIRST_AUDIO_STREAM, 
								0, NULL, &flags, &actualTime, &sample));
	if (flags & MF_SOURCE_READERF_ENDOFSTREAM) {
		hr = E_FAIL;
		goto bail;
	}

	BAIL_IF_FAILED(bail,
		hr = sample->ConvertToContiguousBuffer(&medbuf));

	DWORD len;
	BAIL_IF_FAILED(bail,
		hr = sample->GetTotalLength(&len));

	medbuf->AddRef();
	*ioTime = actualTime;
	*outMedbuf = medbuf;
	*outLen = len;

	hr = S_OK;

bail:
	RELEASE(medbuf);
	RELEASE(sample);
	return hr;
}
