/*
 * Copyright (c) 2010,2011 Yoshikazu Kuramochi
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package ch.kuramo.javie.effects.stylize;

import java.util.HashSet;
import java.util.Set;

import javax.media.opengl.GL2;
import javax.media.opengl.GLUniformData;

import ch.kuramo.javie.api.BlendMode;
import ch.kuramo.javie.api.Color;
import ch.kuramo.javie.api.ColorMode;
import ch.kuramo.javie.api.IAnimatableBoolean;
import ch.kuramo.javie.api.IAnimatableColor;
import ch.kuramo.javie.api.IAnimatableDouble;
import ch.kuramo.javie.api.IAnimatableEnum;
import ch.kuramo.javie.api.IArray;
import ch.kuramo.javie.api.IShaderProgram;
import ch.kuramo.javie.api.IVideoBuffer;
import ch.kuramo.javie.api.Quality;
import ch.kuramo.javie.api.Resolution;
import ch.kuramo.javie.api.VideoBounds;
import ch.kuramo.javie.api.annotations.Effect;
import ch.kuramo.javie.api.annotations.Property;
import ch.kuramo.javie.api.annotations.ShaderSource;
import ch.kuramo.javie.api.annotations.Effect.Categories;
import ch.kuramo.javie.api.services.IArrayPools;
import ch.kuramo.javie.api.services.IBlendSupport;
import ch.kuramo.javie.api.services.IBlurSupport;
import ch.kuramo.javie.api.services.IShaderRegistry;
import ch.kuramo.javie.api.services.ITexture1DSupport;
import ch.kuramo.javie.api.services.IVideoEffectContext;
import ch.kuramo.javie.api.services.IVideoRenderSupport;
import ch.kuramo.javie.api.services.IBlurSupport.BlurDimensions;

import com.google.inject.Inject;

@Effect(id="ch.kuramo.javie.Glow", category=Categories.STYLIZE)
public class Glow {

	public enum GlowColors { ORIGINAL_COLORS, A_AND_B_COLORS, A_AND_B_COLORS_ALPHA_BASE }

	public enum GlowComposite { ON_TOP, BEHIND, NONE }

	public enum GlowLooping { SAWTOOTH_AB, SAWTOOTH_BA, TRIANGLE_ABA, TRIANGLE_BAB }


	@Property("ORIGINAL_COLORS")
	private IAnimatableEnum<GlowColors> colors;

	@Property(value="60", min="0", max="100")
	private IAnimatableDouble threshold;

	@Property(value="10", min="0", max="500")
	private IAnimatableDouble radius;

	@Property(value="1", min="0", max="255")
	private IAnimatableDouble intensity;

	@Property("NONE")
	private IAnimatableEnum<GlowComposite> composite;

	@Property("ADD")
	private IAnimatableEnum<BlendMode> operation;

	@Property("TRIANGLE_ABA")
	private IAnimatableEnum<GlowLooping> colorLooping;

	@Property(value="1", min="1", max="127")
	private IAnimatableDouble colorLoops;

	@Property
	private IAnimatableDouble colorPhase;

	@Property(value="50", min="0", max="100")
	private IAnimatableDouble abMidpoint;

	@Property("1,1,1")
	private IAnimatableColor colorA;

	@Property("0,0,0")
	private IAnimatableColor colorB;

	@Property
	private IAnimatableEnum<BlurDimensions> dimensions;

	@Property("true")
	private IAnimatableBoolean fast;


	private final IVideoEffectContext context;

	private final IVideoRenderSupport support;

	private final IBlurSupport blurSupport;

	private final IArrayPools arrayPools;

	private final ITexture1DSupport tex1DSupport;

	private final IShaderProgram oColorsThresholdProgram;

	private final IShaderProgram abColorsThresholdProgram;

	private final IShaderProgram alphaBaseThresholdProgram;

	private final IShaderProgram sawtoothProgram;

	private final IShaderProgram triangleProgram;

	private final IShaderProgram oColorsIntensityProgram;

	private final IShaderProgram oColorsIntensityFloatProgram;

	private final IShaderProgram abColorsIntensityProgram;

	private final IBlendSupport[] blendSupports;

	@Inject
	public Glow(IVideoEffectContext context, IVideoRenderSupport support, IBlurSupport blurSupport,
			IArrayPools arrayPools, ITexture1DSupport tex1DSupport, IShaderRegistry shaders,
			IBlendSupport blendSupport0, IBlendSupport blendSupport1, IBlendSupport blendSupport2) {

		this.context = context;
		this.support = support;
		this.blurSupport = blurSupport;
		this.arrayPools = arrayPools;
		this.tex1DSupport = tex1DSupport;

		oColorsThresholdProgram = shaders.getProgram(Glow.class, "OCOLORS_THRESHOLD");
		abColorsThresholdProgram = shaders.getProgram(Glow.class, "ABCOLORS_THRESHOLD");
		alphaBaseThresholdProgram = shaders.getProgram(Glow.class, "ALPHA_BASE_THRESHOLD");
		sawtoothProgram = shaders.getProgram(Glow.class, "SAWTOOTH");
		triangleProgram = shaders.getProgram(Glow.class, "TRIANGLE");
		oColorsIntensityProgram = shaders.getProgram(Glow.class, "OCOLORS_INTENSITY");
		oColorsIntensityFloatProgram = shaders.getProgram(Glow.class, "OCOLORS_INTENSITY_FLOAT");
		abColorsIntensityProgram = shaders.getProgram(Glow.class, "ABCOLORS_INTENSITY");

		blendSupports = new IBlendSupport[] { blendSupport0, blendSupport1, blendSupport2 };
		blendSupport0.setProgramsClass(CompositeOnTop.class);
		blendSupport1.setProgramsClass(CompositeBehind.class);
	}

	public VideoBounds getVideoBounds() {
		// GaussianBlurとほぼ同じ。

		VideoBounds bounds = context.getPreviousBounds();
		if (bounds.isEmpty()) {
			return bounds;
		}

		double radius = context.value(this.radius);
		if (radius == 0) {
			return bounds;
		}

		// blurが50より大きい場合はfastプロパティの値にかかわらず高速モード
		boolean fast = (radius > 50) || context.value(this.fast);

		// 解像度に合わせてradiusの値を変換
		radius = context.getVideoResolution().scale(radius);

		BlurDimensions dimensions = context.value(this.dimensions);

		return blurSupport.calcGaussianBlurredBounds(bounds, radius, dimensions, fast);
	}

	public IVideoBuffer doVideoEffect() {
		IVideoBuffer input = context.doPreviousEffect();
		if (input.getBounds().isEmpty()) {
			return input;
		}

		Set<IVideoBuffer> tmpBuffers = new HashSet<IVideoBuffer>();
		tmpBuffers.add(input);

		try {
			GlowColors colors = context.value(this.colors);

			IVideoBuffer buffer = threshold(input, colors);
			tmpBuffers.add(buffer);

			buffer = blur(buffer);
			tmpBuffers.add(buffer);

			if (colors != GlowColors.ORIGINAL_COLORS) {
				buffer = abColors(buffer);
				tmpBuffers.add(buffer);
			}

			buffer = intensity(buffer, colors);
			tmpBuffers.add(buffer);

			buffer = composite(buffer, input);

			// composite が返す値は１つ目の引数と同じものの場合があるので remove しておく。
			tmpBuffers.remove(buffer);

			return buffer;

		} finally {
			for (IVideoBuffer vb : tmpBuffers) {
				vb.dispose();
			}
		}
	}

	@ShaderSource
	public static final String[] OCOLORS_THRESHOLD = {
		"uniform sampler2D texture;",
		"uniform float upper;",
		"uniform float lower;",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2D(texture, gl_TexCoord[0].st);",
		"	float a = color.a;",
		"	vec3 rgb = (a != 0.0) ? color.rgb/a : vec3(0.0);",
		"",
		"	rgb = clamp((rgb - lower) / (upper - lower), 0.0, 1.0);",
		"",
		"	gl_FragColor = vec4(rgb*a, a);",
		"}"
	};

	@ShaderSource
	public static final String[] ABCOLORS_THRESHOLD = {
		"uniform sampler2D texture;",
		"uniform float upper;",
		"uniform float lower;",
		"",
		"const vec3 lumaVec = vec3(0.299, 0.587, 0.114);",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2D(texture, gl_TexCoord[0].st);",
		"",
		"	float luma = dot(color.rgb, lumaVec);",
		"	luma = clamp((luma - lower) / (upper - lower), 0.0, 1.0);",
		"",
		"	gl_FragColor = vec4(luma);",
		"}"
	};

	@ShaderSource
	public static final String[] ALPHA_BASE_THRESHOLD = {
		"uniform sampler2D texture;",
		"uniform float upper;",
		"uniform float lower;",
		"",
		"void main(void)",
		"{",
		"	float a = texture2D(texture, gl_TexCoord[0].st).a;",
		"",
		"	a = clamp((a - lower) / (upper - lower), 0.0, 1.0);",
		"",
		"	gl_FragColor = vec4(a);",
		"}"
	};

	private IVideoBuffer threshold(IVideoBuffer input, GlowColors colors) {
		IShaderProgram program =
				(colors == GlowColors.ORIGINAL_COLORS) ? oColorsThresholdProgram :
				(colors == GlowColors.A_AND_B_COLORS) ? abColorsThresholdProgram : alphaBaseThresholdProgram;

		double threshold = context.value(this.threshold) / 100;
		double upper = Math.min(threshold + 0.0625, 1);
		double lower = Math.max(threshold - 0.0625, 0);

		Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
		uniforms.add(new GLUniformData("texture", 0));
		uniforms.add(new GLUniformData("upper", (float)upper));
		uniforms.add(new GLUniformData("lower", (float)lower));

		return support.useShaderProgram(program, uniforms, null, input);
	}

	private IVideoBuffer blur(IVideoBuffer input) {
		double radius = context.value(this.radius);
		if (radius == 0) {
			return input;
		}

		// blurが50より大きい場合はfastプロパティの値にかかわらず高速モード
		boolean fast = (radius > 50) || context.value(this.fast);

		// 解像度に合わせてradiusの値を変換
		radius = context.getVideoResolution().scale(radius);

		BlurDimensions dimensions = context.value(this.dimensions);

		return blurSupport.gaussianBlur(input, radius, dimensions, false, fast);
	}

	@ShaderSource
	public static final String[] SAWTOOTH = {
		"uniform sampler2D texture;",
		"uniform sampler1D abColors;",
		"uniform float loops;",
		"uniform float phase;",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2D(texture, gl_TexCoord[0].st);",
		"	float a = color.a;",
		"	float t = phase + a * loops;",
		"	gl_FragColor = vec4(texture1D(abColors, t).rgb, 1.0)*a;",
		"}"
	};

	@ShaderSource
	public static final String[] TRIANGLE = {
		"uniform sampler2D texture;",
		"uniform sampler1D abColors;",
		"uniform float loops;",
		"uniform float phase;",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2D(texture, gl_TexCoord[0].st);",
		"	float a = color.a;",
		"	float t = (phase + a * loops) * 2.0;",	// SAWTOOTHとの違いは *2.0 になっていることだけ。
		"	gl_FragColor = vec4(texture1D(abColors, t).rgb, 1.0)*a;",
		"}"
	};

	private IVideoBuffer abColors(IVideoBuffer input) {
		IArray<float[]> colorData = null;
		final int[] colorTex = new int[1];
		final GL2 gl = context.getGL().getGL2();
		try {
			GlowLooping looping = context.value(colorLooping);
			double loops = context.value(colorLoops);
			double phase = context.value(colorPhase) / 360;
			double midpoint = 1 - context.value(abMidpoint) / 100;

			IShaderProgram program;
			int wrapMode;
			if (looping == GlowLooping.SAWTOOTH_AB || looping == GlowLooping.SAWTOOTH_BA) {
				program = sawtoothProgram;
				wrapMode = GL2.GL_REPEAT;
			} else {
				program = triangleProgram;
				wrapMode = GL2.GL_MIRRORED_REPEAT;
			}

			Color colorA;
			Color colorB;
			if (looping == GlowLooping.SAWTOOTH_AB || looping == GlowLooping.TRIANGLE_ABA) {
				colorA = context.value(this.colorA);
				colorB = context.value(this.colorB);
			} else {
				colorA = context.value(this.colorB);
				colorB = context.value(this.colorA);
			}
			Color colorM = new Color((colorA.r+colorB.r)/2, (colorA.g+colorB.g)/2, (colorA.b+colorB.b)/2);

			colorData = arrayPools.getFloatArray(4*256);
			float[] colorArray = colorData.getArray();
			for (int i = 0; i < 256; ++i) {
				double t = i/255.0;
				if (t > midpoint) {
					double u = (t-midpoint)/(1.0-midpoint);
					colorArray[i*4  ] = (float)(colorA.r*u + colorM.r*(1-u));
					colorArray[i*4+1] = (float)(colorA.g*u + colorM.g*(1-u));
					colorArray[i*4+2] = (float)(colorA.b*u + colorM.b*(1-u));
				} else {
					double u = t/midpoint;
					colorArray[i*4  ] = (float)(colorB.r*(1-u) + colorM.r*u);
					colorArray[i*4+1] = (float)(colorB.g*(1-u) + colorM.g*u);
					colorArray[i*4+2] = (float)(colorB.b*(1-u) + colorM.b*u);
				}
				colorArray[i*4+3] = 1;
			}

			Quality quality = context.getQuality();
			Resolution resolution = context.getVideoResolution();
			ColorMode colorMode = context.getColorMode();
			int internalFormat = (quality == Quality.DRAFT || resolution.scale < 1) ? GL2.GL_RGBA8
								: colorMode == ColorMode.RGBA32_FLOAT ? GL2.GL_RGBA32F
								: colorMode == ColorMode.RGBA16_FLOAT ? GL2.GL_RGBA16F
								: quality == Quality.BEST ? GL2.GL_RGBA16F : GL2.GL_RGBA8;

			colorTex[0] = tex1DSupport.texture1DFromArray(
					colorData, GL2.GL_RGBA, internalFormat, GL2.GL_LINEAR_MIPMAP_LINEAR, wrapMode);

			Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
			uniforms.add(new GLUniformData("texture", 0));
			uniforms.add(new GLUniformData("abColors", 1));
			uniforms.add(new GLUniformData("loops", (float)loops));
			uniforms.add(new GLUniformData("phase", (float)phase));

			final VideoBounds bounds = input.getBounds();
			Runnable operation = new Runnable() {
				public void run() {
					gl.glActiveTexture(GL2.GL_TEXTURE1);
					gl.glBindTexture(GL2.GL_TEXTURE_1D, colorTex[0]);
					support.ortho2D(bounds);
					support.quad2D(bounds, new double[][] {{0,0},{1,0},{1,1},{0,1}});
				}
			};

			int pushAttribs = GL2.GL_TEXTURE_BIT;
			return support.useShaderProgram(program, uniforms, operation, pushAttribs, null, input);

		} finally {
			if (colorData != null) colorData.release();
			if (colorTex[0] != 0) gl.glDeleteTextures(1, colorTex, 0);
		}
	}

	@ShaderSource
	public static final String[] OCOLORS_INTENSITY = {
		"uniform sampler2D texture;",
		"uniform float intensity;",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2D(texture, gl_TexCoord[0].st);",
		"	gl_FragColor = vec4(min(color.rgb*intensity, color.a), color.a);",
		"}"
	};

	@ShaderSource
	public static final String[] OCOLORS_INTENSITY_FLOAT = {
		"uniform sampler2D texture;",
		"uniform float intensity;",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2D(texture, gl_TexCoord[0].st);",
		"	gl_FragColor = vec4(color.rgb*intensity, color.a);",
		"}"
	};

	@ShaderSource
	public static final String[] ABCOLORS_INTENSITY = {
		"uniform sampler2D texture;",
		"uniform float intensity;",
		"",
		"void main(void)",
		"{",
		"	vec4 color = texture2D(texture, gl_TexCoord[0].st);",
		"	gl_FragColor = vec4(color.rgb*intensity, min(color.a*intensity, 1.0));",
		"}"
	};

	private IVideoBuffer intensity(IVideoBuffer input, GlowColors colors) {
		double intensity = context.value(this.intensity);
		if (intensity == 1) {
			return input;
		}

		// A_AND_B_COLORS と A_AND_B_COLORS_ALPHA_BASE のシェーダは同じ
		IShaderProgram program
				= (colors != GlowColors.ORIGINAL_COLORS) ? abColorsIntensityProgram
				: input.getColorMode().isFloat() ? oColorsIntensityFloatProgram : oColorsIntensityProgram;

		Set<GLUniformData> uniforms = new HashSet<GLUniformData>();
		uniforms.add(new GLUniformData("texture", 0));
		uniforms.add(new GLUniformData("intensity", (float)intensity));

		return support.useShaderProgram(program, uniforms, null, input);
	}

	private IVideoBuffer composite(IVideoBuffer glow, IVideoBuffer orig) {
		GlowComposite composite = context.value(this.composite);
		BlendMode operation = context.value(this.operation);

		if (composite == GlowComposite.NONE && operation == BlendMode.NONE) {
			return glow;
		}

		return blendSupports[composite.ordinal()].blend(orig, glow, null, operation, 1.0, context);
	}

	private static String[] createCompositeSource(boolean onTop, String name, boolean dissolve) {
		boolean normal = name.equals("normal");

		return new String[] {
								"uniform sampler2D texDst;",	// glow
								"uniform sampler2D texSrc;",	// orig
								"uniform float opacity;",
				dissolve ?		"uniform float dissolveSeed;" : "",
								"",
				String.format(	"vec4 blend_%s(vec4 pDst, vec4 pSrc, float opacity%s);", name, dissolve ? ", float dissolveSeed" : ""),
				!normal ?		"vec4 blend_normal(vec4 pDst, vec4 pSrc, float opacity);" : "",
								"",
								"void main(void)",
								"{",
								"	vec4 glow = texture2D(texDst, gl_TexCoord[0].st);",
								"	vec4 orig = texture2D(texSrc, gl_TexCoord[1].st);",
				String.format(	"	glow = blend_%s(glow, orig, opacity%s);", name, dissolve ? ", dissolveSeed" : ""),
				String.format(	"	gl_FragColor = blend_normal(%s, 1.0);", onTop ? "glow, orig" : "orig, glow"),
								"}"
		};
	}

	private static final String BLEND_FUNCTIONS = "ch.kuramo.javie.core.shaders.BlendModeShaders.blend_functions";

	public static class CompositeOnTop {

		private static String[] createCompositeSource(String name, boolean dissolve) {
			return Glow.createCompositeSource(true, name, dissolve);
		}

		private static String[] createCompositeSource(String name) {
			return Glow.createCompositeSource(true, name, false);
		}

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] NONE = createCompositeSource("none");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] NORMAL = createCompositeSource("normal");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DISSOLVE = createCompositeSource("dissolve", true);

//		@ShaderSource(attach=BLEND_FUNCTIONS)
//		public static final String[] DANCING_DISSOLVE = DISSOLVE;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DARKEN = createCompositeSource("darken");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] MULTIPLY = createCompositeSource("multiply");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR_BURN = createCompositeSource("color_burn");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_BURN = createCompositeSource("linear_burn");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DARKER_COLOR = createCompositeSource("darker_color");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] ADD = createCompositeSource("add");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LIGHTEN = createCompositeSource("lighten");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SCREEN = createCompositeSource("screen");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR_DODGE = createCompositeSource("color_dodge");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_DODGE = createCompositeSource("linear_dodge");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LIGHTER_COLOR = createCompositeSource("lighter_color");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] OVERLAY = createCompositeSource("overlay");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SOFT_LIGHT = createCompositeSource("soft_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HARD_LIGHT = createCompositeSource("hard_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_LIGHT = createCompositeSource("linear_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] VIVID_LIGHT = createCompositeSource("vivid_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] PIN_LIGHT = createCompositeSource("pin_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HARD_MIX = createCompositeSource("hard_mix");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DIFFERENCE = createCompositeSource("difference");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] EXCLUSION = createCompositeSource("exclusion");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HUE = createCompositeSource("hue");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SATURATION = createCompositeSource("saturation");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR = createCompositeSource("color");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LUMINOSITY = createCompositeSource("luminosity");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] STENCIL_ALPHA = createCompositeSource("stencil_alpha");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] STENCIL_LUMA = createCompositeSource("stencil_luma");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SILHOUETTE_ALPHA = createCompositeSource("silhouette_alpha");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SILHOUETTE_LUMA = createCompositeSource("silhouette_luma");

	}

	public static class CompositeBehind {

		private static String[] createCompositeSource(String name, boolean dissolve) {
			return Glow.createCompositeSource(false, name, dissolve);
		}

		private static String[] createCompositeSource(String name) {
			return Glow.createCompositeSource(false, name, false);
		}

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] NONE = createCompositeSource("none");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] NORMAL = createCompositeSource("normal");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DISSOLVE = createCompositeSource("dissolve", true);

//		@ShaderSource(attach=BLEND_FUNCTIONS)
//		public static final String[] DANCING_DISSOLVE = DISSOLVE;

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DARKEN = createCompositeSource("darken");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] MULTIPLY = createCompositeSource("multiply");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR_BURN = createCompositeSource("color_burn");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_BURN = createCompositeSource("linear_burn");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DARKER_COLOR = createCompositeSource("darker_color");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] ADD = createCompositeSource("add");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LIGHTEN = createCompositeSource("lighten");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SCREEN = createCompositeSource("screen");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR_DODGE = createCompositeSource("color_dodge");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_DODGE = createCompositeSource("linear_dodge");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LIGHTER_COLOR = createCompositeSource("lighter_color");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] OVERLAY = createCompositeSource("overlay");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SOFT_LIGHT = createCompositeSource("soft_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HARD_LIGHT = createCompositeSource("hard_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LINEAR_LIGHT = createCompositeSource("linear_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] VIVID_LIGHT = createCompositeSource("vivid_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] PIN_LIGHT = createCompositeSource("pin_light");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HARD_MIX = createCompositeSource("hard_mix");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] DIFFERENCE = createCompositeSource("difference");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] EXCLUSION = createCompositeSource("exclusion");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] HUE = createCompositeSource("hue");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SATURATION = createCompositeSource("saturation");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] COLOR = createCompositeSource("color");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] LUMINOSITY = createCompositeSource("luminosity");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] STENCIL_ALPHA = createCompositeSource("stencil_alpha");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] STENCIL_LUMA = createCompositeSource("stencil_luma");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SILHOUETTE_ALPHA = createCompositeSource("silhouette_alpha");

		@ShaderSource(attach=BLEND_FUNCTIONS)
		public static final String[] SILHOUETTE_LUMA = createCompositeSource("silhouette_luma");

	}

}
