/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.cpupdate/src/jp/sourceforge/jeextension/cpupdate/actions/CpUpdateRunner.java,v 1.2 2004/12/17 07:11:22 kohnosuke Exp $
 * $Revision: 1.2 $
 * $Date: 2004/12/17 07:11:22 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.cpupdate.actions;

import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;

import jp.sourceforge.jeextension.common.xml.XMLException;
import jp.sourceforge.jeextension.common.xml.XMLFile;
import jp.sourceforge.jeextension.common.xml.XMLNode;
import jp.sourceforge.jeextension.common.xml.XMLNodeList;
import jp.sourceforge.jeextension.cpupdate.CpUpdatePlugin;

import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jface.operation.IRunnableWithProgress;

/**
 * CPUpdateRunner
 */
class CpUpdateRunner implements IRunnableWithProgress {

    /**
     * Selection elements.
     */
    private Object[] elements;
    
    /**
     * Current processing target's project.
     */
    private IProject currentProject;
    
    /**
     * Current thread's ProgressMonitor.
     */
    private IProgressMonitor currentMonitor;
    
    /**
     * Initiate CpUpdateRunner with selection elements.
     * @param selection Selection elements.
     */
    public CpUpdateRunner(Object[] selection) {
        super();
        this.elements = selection;
    }
    
    /**
     * {@inheritDoc}
     * @see org.eclipse.jface.operation.IRunnableWithProgress#run(org.eclipse.core.runtime.IProgressMonitor)
     */
    public void run(IProgressMonitor monitor)
        throws InvocationTargetException, InterruptedException {
        
        currentMonitor = monitor;
        
        for (int i = 0; i < elements.length; i++) {
            Object elem = elements[i];
            
            // selection element check.
            if (elem instanceof IProject) {
                currentProject = (IProject) elem;
            }
            else if (elem instanceof IJavaProject) {
                currentProject = ((IJavaProject) elem).getProject();
            }
            else {
                return;
            }
            // get define file.
            File f = CpUpdatePlugin.getReferenceFile(currentProject);
            if (f == null) {
                return;
            }
            //
            // try read define file.
            //
            XMLFile xmlFile = new XMLFile(f);
            XMLNode rootNode;
            try {
                rootNode = xmlFile.readNode();
            }
            catch (XMLException e) {
                CpUpdatePlugin.log(e);
                return;
            }
            //
            // try create entries.
            //
            try {
                XMLNodeList nodeList = rootNode.getChildNodes();
                for (int j = 0; j < nodeList.size(); j++) {
                    XMLNode node = nodeList.getNode(j);
                
                    if (node.getNodeName().equals("link-resource")) {
                        createLinkResource(node);
                    }
                
                    if (node.getNodeName().equals("output-folder")) {
                        createOutputFolder(node);
                    }
                
                    if (node.getNodeName().equals("classpath-entry")) {
                        createClasspath(node);
                    }
                }
            }
            catch (CoreException e) {
                CpUpdatePlugin.log(e);
                return;
            }
        }
    }
    
    /**
     * Create 'Link Resource'.
     * @param parent XMLNode object.
     * @throws CoreException throws folder process failed when create link.
     */
    private void createLinkResource(XMLNode parent) throws CoreException {
        XMLNodeList nodeList = parent.getChildNodes();
        for (int i = 0; i < nodeList.size(); i++) {
            XMLNode node = nodeList.getNode(i);
            String name = node.getAttribute("name");
            String path = node.getAttribute("path");
            // create link resource.
            IFolder folder = currentProject.getFolder(name);
            IPath fullPath = new Path(path);
            fullPath.toFile().mkdirs();
            if (folder.isLinked() == false) {
                folder.createLink(fullPath, IResource.NONE, currentMonitor);
            }
        }
    }
    
    /**
     * Create 'Output Folder'.
     * @param parent XMLNode object.
     * @throws CoreException throws folder process failed when create output.
     */
    private void createOutputFolder(XMLNode parent) throws CoreException {
        XMLNodeList nodeList = parent.getChildNodes();
        for (int i = 0; i < nodeList.size(); i++) {
            XMLNode node = nodeList.getNode(i);
            String name = node.getAttribute("name");
            // create output folder.
            IJavaProject javaProject = JavaCore.create(currentProject);
            IPath path = currentProject.getFullPath().append(name);
            javaProject.setOutputLocation(path, currentMonitor);
        }
    }
    
    /**
     * Create 'Classpath'.
     * @param parent XMLNode object.
     * @throws CoreException throws folder process failed when create classpath
     */
    private void createClasspath(XMLNode parent) throws CoreException {
        List entryList = new ArrayList();
        
        XMLNodeList nodeList = parent.getChildNodes();
        for (int i = 0; i < nodeList.size(); i++) {
            XMLNode node = nodeList.getNode(i);
            String kind = node.getAttribute("kind");
            if (kind.equals("src")) {
                String name = node.getAttribute("path");
                String output = node.getAttribute("output");
                addSourceFolder(name, output, entryList);
            }
            if (kind.equals("lib")) {
                String path = node.getAttribute("path");
                String src  = node.getAttribute("sourcepath");
                addLibraries(path, src, entryList);
            }
            if (kind.equals("con")) {
                String path = node.getAttribute("path");
                addContainer(path, entryList);
            }
        }
        
        // set order classpath entry.
        IJavaProject javaProject = JavaCore.create(currentProject);
        IClasspathEntry[] entries =
                (IClasspathEntry[]) entryList.toArray(
                        new IClasspathEntry[entryList.size()]);
        javaProject.setRawClasspath(entries, currentMonitor);
    }
    
    /**
     * Add 'SourceFolder' to entry list.
     * @param name 'SourceFolder' name.
     * @param out 'Sepecified Output Folder' name.
     * @param list entry list.
     * @throws CoreException throws process failed when create new folder.
     */
    protected void addSourceFolder(String name, String out, List list)
            throws CoreException {
        
        IPath path = currentProject.getFullPath().append(name);
        IPath output = null;
        if (out != null && out.equals("") == false) {
            output = currentProject.getFullPath().append(out);
        }
        ensureFolderExists(path);
        IClasspathEntry entry = JavaCore.newSourceEntry(
                path, new IPath[0], output);
        list.add(entry);
    }
    
    /**
     * Add 'Libraries' to entry list.
     * @param lib 'Libraries' path.
     * @param src 'Libraries' source path.
     * @param list entry list.
     */
    protected void addLibraries(String lib, String src, List list) {
        
        // get library path.
        IPath libPath;
        IResource libResource = currentProject.findMember(lib);
        if (libResource == null) {
            libPath = new Path(lib);
        }
        else {
            libPath = libResource.getFullPath();
        }
        
        // get source path.
        IPath srcPath;
        if (src == null || src.equals("")) {
            srcPath = null;
        }
        else {
            IResource srcResource = currentProject.findMember(lib);
            if (srcResource == null) {
                srcPath = new Path(lib);
            }
            else {
                srcPath = srcResource.getFullPath();
            }
        }

        IClasspathEntry entry =
                JavaCore.newLibraryEntry(libPath, srcPath, null, false);
        if (entry != null) {
            list.add(entry);
        }
    }

    /**
     * Add 'Container' to entry list.
     * @param name 'Container' name.
     * @param list entry list.
     */
    protected void addContainer(String name, List list) {
        list.add(JavaCore.newContainerEntry(new Path(name)));
    }

    /**
     * Check target's folder. if not exist, create one.
     * @param folderPath check target's folder name.
     * @throws CoreException throws process failed when create new folder.
     */
    private void ensureFolderExists(IPath folderPath)
            throws CoreException {
        
        IWorkspace workspace = currentProject.getWorkspace();

        for (int i = 1; i <= folderPath.segmentCount(); i++) {
            IPath partialPath = folderPath.uptoSegment(i);
            if (!workspace.getRoot().exists(partialPath)) {
                IFolder folder = workspace.getRoot().getFolder(partialPath);
                folder.create(true, true, currentMonitor);
            }
        }
    }
}
