/* ****************************************************************************
 * Copyright (c) 2002 Java Eclipse Extension Project.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.opensource.org/licenses/cpl.php
 * ============================================================================
 * $Header: /cvsroot/jeextension/jp.sourceforge.jeextension.styleeditor/src/jp/sourceforge/jeextension/styleeditor/core/DoubleClickStrategy.java,v 1.1 2005/01/29 03:24:19 kohnosuke Exp $
 * $Revision: 1.1 $
 * $Date: 2005/01/29 03:24:19 $
 * ============================================================================
 * ***************************************************************************/
package jp.sourceforge.jeextension.styleeditor.core;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextDoubleClickStrategy;
import org.eclipse.jface.text.ITextViewer;

/**
 * DoubleClickStrategy is handling double click operations.
 */
class DoubleClickStrategy implements ITextDoubleClickStrategy {

    /**
     * Text at DoubleClick target parts.
     */
    protected ITextViewer fText;

    /**
     * default constructor.
     */
    DoubleClickStrategy() {
        super();
    }

    /**
     * {@inheritDoc}
     * @see org.eclipse.jface.text.ITextDoubleClickStrategy#doubleClicked(org.eclipse.jface.text.ITextViewer)
     */
    public void doubleClicked(ITextViewer part) {
        int pos = part.getSelectedRange().x;
        
        if (pos < 0) {
            return;
        }
        
        fText = part;
        
        if (!selectComment(pos)) {
            selectWord(pos);
        }
    }

    /**
     * Set selection range.
     * @param startPos starts position at selection parts range.
     * @param stopPos end position at selection parts range.
     */
    private void selectRange(int startPos, int stopPos) {
        int offset = startPos + 1;
        int length = stopPos - offset;
        fText.setSelectedRange(offset, length);
    }

    /**
     * Returns that selection part is comment blocks.
     * @param caretPos selection parts positions.
     * @return if parts is comment then return true
     */
    protected boolean selectComment(int caretPos) {
        IDocument doc = fText.getDocument();
        int startPos, endPos;
        
        try {
            int pos = caretPos;
            char c = ' ';
            
            while (pos >= 0) {
                c = doc.getChar(pos);
                if (c == '\\') {
                    pos -= 2;
                    continue;
                }
                if (c == Character.LINE_SEPARATOR || c == '\"') {
                    break;
                }
                --pos;
            }
            
            if (c != '\"') {
                return false;
            }
            
            startPos = pos;
            
            pos = caretPos;
            int length = doc.getLength();
            c = ' ';
            
            while (pos < length) {
                c = doc.getChar(pos);
                if (c == Character.LINE_SEPARATOR || c == '\"') {
                    break;
                }
                ++pos;
            }
            if (c != '\"') {
                return false;
            }
            
            endPos = pos;
            
            int offset = startPos + 1;
            int len = endPos - offset;
            fText.setSelectedRange(offset, len);
            return true;
        }
        catch (BadLocationException e) {
            // none handling.
        }
        
        return false;
    }

    /**
     * Returns that selection part isn't comment blocks.
     * @param caretPos selection parts positions.
     * @return if parts isn't comment then return true
     */
    protected boolean selectWord(int caretPos) {
        
        IDocument doc = fText.getDocument();
        int startPos, endPos;
        
        try {
            
            int pos = caretPos;
            char c;
            
            while (pos >= 0) {
                c = doc.getChar(pos);
                if (!Character.isJavaIdentifierPart(c)) {
                    break;
                }
                --pos;
            }
            
            startPos = pos;
            
            pos = caretPos;
            int length = doc.getLength();
            
            while (pos < length) {
                c = doc.getChar(pos);
                if (!Character.isJavaIdentifierPart(c)) {
                    break;
                }
                ++pos;
            }
            
            endPos = pos;
            selectRange(startPos, endPos);
            return true;
        }
        catch (BadLocationException e) {
            // none handling.
        }
        return false;
    }
}