/*
 * Copyright 2000-2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.graffito.portlets;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.PortletConfig;
import javax.portlet.PortletException;
import javax.portlet.PortletMode;
import javax.portlet.PortletPreferences;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;
import javax.portlet.WindowState;

import org.apache.portals.bridges.velocity.GenericVelocityPortlet;
import org.apache.portals.graffito.ContentModelService;
import org.apache.portals.graffito.exception.ContentManagementException;
import org.apache.portals.graffito.model.Document;
import org.apache.portals.messaging.PortletMessaging;
import org.apache.velocity.context.Context;

import org.apache.portals.graffito.portlets.util.ServiceAccessor;


/**
 * CMS Document Displayer
 * 
 * @author <a href="mailto:taylor@apache.org">David Sean Taylor</a>
 * @version $Id: DocumentPortlet.java,v 1.1 2005/04/13 23:17:40 shinsuke Exp $
 */
public class DocumentPortlet extends GenericVelocityPortlet
{
    public final static String CURRENT_ISSUE = "current_issue";
    public final static String CURRENT_FOLDER = "current_folder";
    public final static String CURRENT_PAGE = "current_page";
    public final static String CURRENT_USER = "current_user";
    public final static String REQUEST_NODE = "node";
    /** the selected leaf node in the tree view */
    public final static String REQUEST_SELECT_NODE = "select_node";
    
    public final static String JAVASCRIPT = "<script language='JavaScript'>" +
    	"function setAnchor(anchor) { location.hash = anchor; } </script>";
    
    private ContentModelService cms;
    
    public void doView(RenderRequest request, RenderResponse response) throws PortletException, IOException
    {
    	response.setContentType("text/html");
        
        String currentFile = (String) PortletMessaging.receive(request, "FileDisplayer", "current_file");
        
        String fileName = null;
        PortletPreferences prefs = request.getPreferences();
        String uri = prefs.getValue("document-uri", null);
        if (uri != null && uri.length() > 0)
        {
            currentFile = uri;
        }
        if(currentFile != null)
        {
        	try 
            {
				Document doc = cms.getDocument(currentFile);
				if ((doc.getContentType().indexOf("html") != -1)  || (doc.getContentType().indexOf("text") != -1))
				{
                    String title = doc.getProperty("title");
                    if (title == null)
                        title = doc.getName();
					String currentDocument = "<div>Current Document: <b>" + title + "</b> - "+ doc.getUri() + "</div>";
	                String anchors = "<hr/>"; //createAnchors(doc);
					response.getPortletOutputStream().write(currentDocument.getBytes());
	                response.getPortletOutputStream().write(anchors.getBytes());                
					response.getPortletOutputStream().write(DIV_START.getBytes());
					drain(doc.getContent().getContentStream(), response.getPortletOutputStream());
					response.getPortletOutputStream().write(DIV_END.getBytes());
					
					int hash = currentFile.indexOf('#');
					if(hash != -1)
					{
						String anchor = currentFile.substring(hash+1);
						response.getPortletOutputStream().write(JAVASCRIPT.getBytes());
						response.getPortletOutputStream().write("<script>setAnchor('".getBytes());
						response.getPortletOutputStream().write(anchor.getBytes());
						response.getPortletOutputStream().write("');</script>".getBytes());
					}
				}
                else if (doc.getContentType().indexOf("gif") != -1)
                {
                    String title = doc.getProperty("title");
                    if (title == null)
                        title = doc.getName();
                    String currentDocument = "<div>Current Document: <b>" + title + "</b> - "+ doc.getUri() + "</div>";
                    String anchors = "<hr/>"; //createAnchors(doc);
                    response.getPortletOutputStream().write(currentDocument.getBytes());
                    response.getPortletOutputStream().write(anchors.getBytes());                
                    response.getPortletOutputStream().write(DIV_START.getBytes());
                    String gif = "<IMG SRC='" + request.getContextPath() + "/FileServlet?uri=" + doc.getUri() + "'/>";
                    response.getPortletOutputStream().write(gif.getBytes());
                    response.getPortletOutputStream().write(DIV_END.getBytes());                    
                }
				
				else 
				{
                    renderBinaryDoc(request, response, currentFile, doc);
				}
			} 
            catch (ContentManagementException e) 
            {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}           
        }
    }

    private String createAnchors(Document doc)
    {
        return "<hr/><a href='#'>Top</a>&nbsp;<a href='#010'>Sample Anchor 1</a>&nbsp;<a href='#014'>Sample Anchor 2</a>&nbsp;<a href='#020'>Sample Anchor 3</a><hr/>";
    }
    
    private void renderBinaryDoc(RenderRequest request, RenderResponse response, String currentFile, Document doc)
    throws IOException
    {
        String title = doc.getProperty("title");
        if (title == null)
            title = doc.getName();
        
        String contextPath = request.getContextPath();
        String linkage = "<a target='_new' href='" + 
            contextPath + "/FileServlet?uri=" + currentFile + "'>View in New Window</a>";        
        String currentDocument = "<div>Document: <b>" + title + "</b>&nbsp" + linkage;
        response.getPortletOutputStream().write(currentDocument.getBytes());
        
        String iframe = "<iframe src=\"" + contextPath + "/FileServlet?uri=" + currentFile + "\"";
        if (request.getWindowState().equals(WindowState.MAXIMIZED))
            iframe += " height=\"800\" width=\"100%\"";
        else
            iframe += " height=\"700\" width=\"100%\"";
        iframe += "></iframe>";
        response.getPortletOutputStream().write(iframe.getBytes());        
    }
    
    public void doEdit(RenderRequest request, RenderResponse response) 
    throws PortletException, IOException
    {
        response.setContentType("text/html");
        String currentArticle = (String) PortletMessaging.receive(request, "FileDisplayer", "current_file");
        if (currentArticle == null)
        {
            return;
        }
        ContentModelService cms = (ContentModelService)ServiceAccessor.getService("cmsModelImpl");
        try 
        {
            Document document = cms.getDocument(currentArticle);
            Context context = super.getContext(request);
            context.put("reference", (String)document.getProperty("reference"));
            context.put("title", (String)document.getProperty("title"));
            context.put("displayTitle", (String)document.getProperty("displayTitle"));
            context.put("year", (String)document.getProperty("year"));
            context.put("month", (String)document.getProperty("month"));
            if (document.getContentType().indexOf("pdf") != -1)                
                context.put("issue", (String)document.getProperty("articleRefs"));
            else
                context.put("issue", (String)document.getProperty("issue"));
                
            context.put("keywords", "example keywords, feature, not implemented, yet");            
        }
        catch (ContentManagementException e) 
        {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }           
        
        super.doEdit(request, response);
    }

    public void processAction(ActionRequest actionRequest, ActionResponse actionResponse) 
    throws PortletException,
          IOException
    {
        if (actionRequest.getPortletMode() == PortletMode.EDIT)
        {
            String title = (String)actionRequest.getParameter("title");
            // TODO: oops no API to save document
        }
        actionResponse.setPortletMode(PortletMode.VIEW);
        super.processAction(actionRequest, actionResponse);
        
    }
    
    // todo: make the scroll area and width configurable by prefs
    private static final String DIV_START = "<DIV id='itdbscroll' style='OVERFLOW: auto; WIDTH: 100%; HEIGHT: 600px;'>";
    private static final String DIV_END = "</DIV>";

    private static final int BLOCK_SIZE = 4096;

    public static void drain(InputStream r,OutputStream w) throws IOException
    {
        byte[] bytes=new byte[BLOCK_SIZE];
        try
        {
          int length=r.read(bytes);
          while(length!=-1)
          {
              if(length!=0)
                  {
                      w.write(bytes,0,length);
                  }
              length=r.read(bytes);
          }
      }
      finally
      {
        bytes=null;
      }

    }

    public void init(PortletConfig config) throws PortletException
    {
        super.init(config);
        
        cms = ServiceAccessor.getModelService();
        if (cms == null)
        {
            throw new PortletException("Failed to load CMS service");
        }
    }
        
}
