/*
 * Copyright 2000-2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.graffito.portlets;

import java.io.IOException;
import java.io.NotSerializableException;
import java.util.List;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.PortletConfig;
import javax.portlet.PortletException;
import javax.portlet.PortletRequest;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;

import org.apache.portals.bridges.common.GenericServletPortlet;
import org.apache.portals.graffito.ContentModelService;
import org.apache.portals.graffito.ContentSearchService;
import org.apache.portals.graffito.exception.ContentManagementException;
import org.apache.portals.graffito.model.Document;
import org.apache.portals.graffito.search.Filter;
import org.apache.portals.graffito.search.SearchResults;
import org.apache.portals.messaging.PortletMessaging;
import org.apache.webapp.admin.TreeControl;
import org.apache.webapp.admin.TreeControlNode;

import org.apache.portals.graffito.portlets.tree.CMSTreeLoader;
import org.apache.portals.graffito.portlets.util.ServiceAccessor;


/**
 * CMS Search
 * 
 * @author <a href="mailto:taylor@apache.org">David Sean Taylor</a>
 * @version $Id: SearchPortlet.java,v 1.1 2005/04/13 23:17:40 shinsuke Exp $
 */
public class SearchPortlet extends GenericServletPortlet
{
    public final static String PORTLET_URL = "portlet_url";
    
    /** the selected leaf node in the tree view */
    public final static String REQUEST_SELECT_NODE = "select_node";
    public final static String REQUEST_NODE = "node";
    
    private static final String SEARCH_MESSAGE = "search_message";
    private static final String SEARCH_STRING = "search_string";
    //TODO: localize this message    
    private static final String NOT_FOUND = "No matching documents found for search criteria.";

    private ContentSearchService search = null;
    private ContentModelService cms = null;
    private CMSTreeLoader loader ;
        
    //TODO: config via portlet.xml
    private int perPage = 10;
    
    public void doView(RenderRequest request, RenderResponse response) throws PortletException, IOException
    {
        response.setContentType("text/html");

        TreeControl control = (TreeControl) request.getPortletSession().getAttribute("search_tree");
        if (control != null)
        {
        	request.setAttribute("search_tree", control);
        }
        
        Object anchor = request.getPortletSession().getAttribute("tree_anchor");
        if(anchor != null)
        {
            request.setAttribute("tree_anchor", anchor);
            //request.getPortletSession().removeAttribute("tree_anchor");
        }
        
        String next = (String) request.getPortletSession().getAttribute("next");
        if(next != null)
        {
        	request.setAttribute("next", next);
        	request.getPortletSession().removeAttribute("next");
        }
        
        String prev = (String) request.getPortletSession().getAttribute("prev");
        if(prev != null)
        {
        	request.setAttribute("prev", prev);
        	request.getPortletSession().removeAttribute("prev");
        }
        
        Object refresh = request.getPortletSession().getAttribute("refresh_page");
        if(refresh != null)
        {
            request.getPortletSession().removeAttribute("refresh_page");
            request.setAttribute("refresh_page", refresh);
        }

        String searchMessage = (String) PortletMessaging.consume(request, SEARCH_MESSAGE);
        request.setAttribute(SEARCH_MESSAGE, searchMessage);
        
        String searchString = (String) PortletMessaging.receive(request, "Search", SEARCH_STRING);
        request.setAttribute(SEARCH_STRING, searchString);        
        
        super.doView(request, response);
    } 
 
    public void processAction(ActionRequest actionRequest, ActionResponse actionResponse) 
    throws PortletException,
          IOException
    {
        String searchString = actionRequest.getParameter("search");
        TreeControl control = (TreeControl) actionRequest.getPortletSession().getAttribute("search_tree");        
        if (searchString != null && searchString.trim().length() > 0)
        {
            control = doSearch(actionRequest, searchString);
            if(control == null)
            {
                PortletMessaging.publish(actionRequest, SEARCH_MESSAGE, NOT_FOUND);
            }
            else
            {
                List results  = (List) actionRequest.getPortletSession().getAttribute("search_results");
                if(perPage < results.size())
                {
                	actionRequest.getPortletSession().setAttribute("next", Integer.toString(perPage));
                }
                if (results.size() == 0)
                {
                    control = null;                    
                    PortletMessaging.publish(actionRequest, SEARCH_MESSAGE, NOT_FOUND);                    
                }                
            }
            PortletMessaging.publish(actionRequest, "Search", SEARCH_STRING, searchString);                        
        }
        //assert control != null
        
        String start = actionRequest.getParameter("start");
        if(start != null)
        {
        	int startInt = Integer.parseInt(start);
        	List results  = (List) actionRequest.getPortletSession().getAttribute("search_results");
        	
        	int end = startInt + perPage;
        	if(end > results.size())
        	{
        		end = results.size();
        	}
        	List subList = results.subList(startInt, end);
        	try {
        		control = this.buildCmsTree(subList);
        		actionRequest.getPortletSession().setAttribute("search_tree", control);
			} catch (ContentManagementException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
        	
        	if(startInt > 0)
        	{
        		int prev = startInt - perPage;
        		actionRequest.getPortletSession().setAttribute("prev", Integer.toString(prev));
        	}
        	int next = startInt + perPage;
        	if(next < results.size())
        	{                
        		actionRequest.getPortletSession().setAttribute("next", Integer.toString(next));
        	}
        }

        if (control == null)
        {
            actionRequest.getPortletSession().setAttribute("search_tree", null);
        }
        else
        {
            // expand or contact non-leaf nodes
            String node = actionRequest.getParameter(REQUEST_NODE);
            if (node != null)
            {
                TreeControlNode controlNode = control.findNode(node);
                if (controlNode != null)
                {
                    if(controlNode.isLazy() && !controlNode.isLoaded()) 
                    {
                        try {
							loader.loadChildren(controlNode);
						} catch (ContentManagementException e) {
							// TODO Auto-generated catch block
							e.printStackTrace();
						}
                    }
                    controlNode.setExpanded(!controlNode.isExpanded());
                    actionRequest.getPortletSession().setAttribute("tree_anchor", node);
                    
                    if(controlNode.isExpanded() && controlNode != control.getRoot())
                    {
                        TreeControlNode[] siblings = controlNode.getParent().findChildren();
                        for(int i=0; i<siblings.length; i++)
                        {
                            TreeControlNode sibling = siblings[i];
                            if(sibling != controlNode)
                            {
                                sibling.setExpanded(false);
                            }
                        }
                    }
                }
            }

            // select a node
            String selectedNode = actionRequest.getParameter(REQUEST_SELECT_NODE);
            if (selectedNode != null)
            {
                selectNode(actionRequest, control, selectedNode);
                actionRequest.getPortletSession().setAttribute("tree_anchor", selectedNode);
            }
        }
    }
    
    private TreeControl doSearch(ActionRequest request, String searchString)
    {
        Filter filter = search.newFilter();
        filter.setScope(this.getInitParameter("scope"));
        filter.addFullTextSearch(searchString);
        SearchResults results = null;
        try
        {
            results = search.searchCmsObjects(filter);
            if (results.size() > 0)
            {
            	List resultList = results.getResults();
                int end = perPage;
                if(perPage > resultList.size())
                {
                    end = resultList.size();
                }
            	List subList = resultList.subList(0, end);
                TreeControl control = buildCmsTree(subList);
                request.getPortletSession().setAttribute("search_tree", control);
                request.getPortletSession().setAttribute("search_results", resultList);
                return control;
            }
        }
        catch (ContentManagementException e)
        {
            System.err.println("Failed to search on search string: " + searchString + ", " + e.toString());
        }
        // handle case where nothing found
        if (results == null)
            results = new SearchResults();
        List resultList = results.getResults();
        //List subList = resultList.subList(0,perPage);
        TreeControlNode root = loader.createRootNode("Search Results", "SEARCH_DOMAIN");
        TreeControl control = new TreeControl(root);            
        request.getPortletSession().setAttribute("search_tree", control);
        request.getPortletSession().setAttribute("search_results", resultList);
        return control;        
    }
    
    private TreeControl buildCmsTree(List results) throws ContentManagementException
    {
        TreeControlNode root = loader.createRootNode("Search Results", "SEARCH_DOMAIN");
        TreeControl treeControl = new TreeControl(root);
        for (int ix = 0; ix < results.size(); ix++)
        {
            Document document = (Document)results.get(ix);
            /*
            String key = document.getProperty("reference");
            String dt = document.getProperty("displayTitle");
            TreeControlNode docNode = 
                new TreeControlNode(
                        key, 
                        "document.gif", 
                        document.getProperty("displayTitle"), 
                        PORTLET_URL, 
                        null, 
                        false, 
                        "DOC_DOMAIN",
                        "Document");
                        */
            // refToURIMap.put(document.getProperty("reference"), document.getUri());
            TreeControlNode docNode = loader.createDocumentNode(document);
            root.addChild(docNode);
        }
                 
        return treeControl;
    }
    
    private void selectNode(PortletRequest actionRequest, TreeControl control, String selectedNode)
    {
        System.out.println("selected node = " + selectedNode);
        control.selectNode(selectedNode);
        TreeControlNode child = control.findNode(selectedNode);
        if (child != null)
        {
            String domain = child.getDomain();
            /*
            if (domain.equals("FOLDER_DOMAIN"))
            {
                if (selectedNode != null)
                {
                    System.out.println("posting selected node = " + selectedNode);
                    
                    actionRequest.getPortletSession().setAttribute(
                            "current_folder", selectedNode,
                            PortletSession.APPLICATION_SCOPE);
                }
            }
            */
            if(domain.equals("DOC_DOMAIN") || domain.equals("ANCHOR_DOMAIN"))
            {
            	String uri = selectedNode; //(String) this.refToURIMap.get(selectedNode);
            	//String uri = loader.getURIFromNodeName(selectedNode);
                try
                {
                    PortletMessaging.publish(actionRequest, "FileDisplayer", "current_file", uri);
                }
                catch (NotSerializableException e)
                {}
                
                actionRequest.getPortletSession().setAttribute("refresh_page", Boolean.TRUE);
            }
        }
    }
    
    
    
    public void init(PortletConfig config) throws PortletException
    {
        super.init(config);
        
        search = (ContentSearchService)ServiceAccessor.getSearchService();
        if (search == null)
        {
            throw new PortletException("Failed to load CMS Search service");
        }
        cms = ServiceAccessor.getModelService();
        if (cms == null)
        {
            throw new PortletException("Failed to load CMS service");
        }
        
        loader = new CMSTreeLoader(cms);
    }
    
}
