/*
 * Copyright 2000-2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.graffito.portlets;

import java.io.IOException;
import java.io.NotSerializableException;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.PortletConfig;
import javax.portlet.PortletException;
import javax.portlet.PortletRequest;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;

import org.apache.portals.bridges.common.GenericServletPortlet;
import org.apache.portals.graffito.ContentModelService;
import org.apache.portals.graffito.exception.ContentManagementException;
import org.apache.portals.graffito.model.Document;
import org.apache.portals.graffito.model.Folder;
import org.apache.portals.messaging.PortletMessaging;
import org.apache.webapp.admin.TreeControl;
import org.apache.webapp.admin.TreeControlNode;

import org.apache.portals.graffito.portlets.tree.CMSTreeLoader;
import org.apache.portals.graffito.portlets.util.ServiceAccessor;



/**
 * CMS Tree Portlet
 * 
 * @author <a href="mailto:taylor@apache.org">David Sean Taylor</a>
 * @version $Id: TreePortlet.java,v 1.2 2005/04/17 22:50:39 shinsuke Exp $
 */
public class TreePortlet extends GenericServletPortlet
{
    public final static String PORTLET_ACTION = "portlet_action";
    public final static String PORTLET_URL = "portlet_url";
    public final static String CURRENT_FOLDER = "current_folder";
    public final static String CURRENT_PAGE = "current_page";
    public final static String CURRENT_USER = "current_user";
    public final static String REQUEST_NODE = "node";
    /** the selected leaf node in the tree view */
    public final static String REQUEST_SELECT_NODE = "select_node";
    
    protected final static String TARGET = "_parent";
    
    protected String xmlIndex;
    
    protected CMSTreeLoader loader;
    protected ContentModelService cms;
    
    public void doView(RenderRequest request, RenderResponse response) throws PortletException, IOException
    {
        response.setContentType("text/html");

        TreeControl control = (TreeControl) request.getPortletSession().getAttribute("j2_tree");
        if (control == null)
        {
            control = buildCmsTree(this.getInitParameter("scope"), "Content", "DTB_DOMAIN");
            request.getPortletSession().setAttribute("j2_tree", control);
        }
        
        Object anchor = request.getPortletSession().getAttribute("tree_anchor");
        if(anchor != null)
        {
            request.setAttribute("tree_anchor", anchor);
            //request.getPortletSession().removeAttribute("tree_anchor");
        }
        
       
        request.setAttribute("j2_tree", control);

        super.doView(request, response);
    } 
 
    public void processAction(ActionRequest actionRequest, ActionResponse actionResponse) 
    throws PortletException,
          IOException
    {
    	
    	Object refresh = actionRequest.getParameter("refresh_page");
        if(refresh != null)
        {        	
        	actionRequest.getPortletSession().removeAttribute("j2_tree");
        	return;
        }
        
        TreeControl control = (TreeControl) actionRequest.getPortletSession().getAttribute("j2_tree");
        //assert control != null
        if (control != null)
        {
            // expand or contact non-leaf nodes
            String node = actionRequest.getParameter(REQUEST_NODE);
            if (node != null)
            {
                TreeControlNode controlNode = control.findNode(node);
                if (controlNode != null)
                {
                    if(controlNode.isLazy() && !controlNode.isLoaded())	
                    {
                        try {
							loader.loadChildren(controlNode);
						} catch (ContentManagementException e) {
							// TODO Auto-generated catch block
							e.printStackTrace();
						}
                    }
                    controlNode.setExpanded(!controlNode.isExpanded());
                    actionRequest.getPortletSession().setAttribute("tree_anchor", node);
                    
                    if(controlNode.isExpanded() && controlNode != control.getRoot())
                    {
                    	TreeControlNode[] siblings = controlNode.getParent().findChildren();
                    	for(int i=0; i<siblings.length; i++)
                    	{
                    		TreeControlNode sibling = siblings[i];
                    		if(sibling != controlNode)
                    		{
                    			sibling.setExpanded(false);
                    		}
                    	}
                    }
                }
            }

            // select a node
            String selectedNode = actionRequest.getParameter(REQUEST_SELECT_NODE);
            if (selectedNode != null)
            {
                selectNode(actionRequest, control, selectedNode);
                actionRequest.getPortletSession().setAttribute("tree_anchor", selectedNode);
            }
        }
    }
   
    private void selectNode(PortletRequest actionRequest, TreeControl control, String selectedNode)
    {
        System.out.println("selected node = " + selectedNode);
        control.selectNode(selectedNode);
        TreeControlNode child = control.findNode(selectedNode);
        if (child != null)
        {
            String domain = child.getDomain();
            /*
            if (domain.equals("FOLDER_DOMAIN"))
            {
                if (selectedNode != null)
                {
                    System.out.println("posting selected node = " + selectedNode);
                    
                    actionRequest.getPortletSession().setAttribute(
                            "current_folder", selectedNode,
                            PortletSession.APPLICATION_SCOPE);
                }
            }
            */
            if(domain.equals("DOC_DOMAIN") || domain.equals("ANCHOR_DOMAIN"))
            {
            	//String uri = loader.getURIFromNodeName(selectedNode);
            	String uri = selectedNode; //(String) refToURIMap.get(selectedNode);
                try
                {
                    PortletMessaging.publish(actionRequest, "FileDisplayer", "current_file", uri);
                }
                catch (NotSerializableException e)
                {}
                
                actionRequest.getPortletSession().setAttribute("refresh_page", Boolean.TRUE);
            }
        }
    }
    
    public TreeControl buildCmsTree(String folderRoot, String label, String rootDomain)
    {
    	TreeControlNode root = loader.createRootNode(label, rootDomain);
    	TreeControl treeControl = new TreeControl(root);
        
        if (null == cms)
            return treeControl;
    
        // load from CMS
        try
        {            
            System.out.println("TEST: buildCmsTree: folderRoot="+folderRoot);
            Collection topLevel = cms.getFolders(folderRoot);
            Collections.sort((List)topLevel, CMSTreeLoader.getComparator());
            Iterator top = topLevel.iterator();
            while (top.hasNext())
            {
                Folder topFolder = (Folder)top.next();
                TreeControlNode node = loader.createFolderNode(topFolder);
                root.addChild(node);
                if(cms.getFolders(topFolder.getUri()).size() > 0)
                {
                	node.setLeaf(false);
                }
            }

            // then get documents
            Collection docLevel = cms.getDocuments(folderRoot);
            Collections.sort((List)docLevel, CMSTreeLoader.getComparator());
            Iterator docs = docLevel.iterator();
            while (docs.hasNext())
            {
                Document doc = (Document)docs.next();
                TreeControlNode issueNode = loader.createDocumentNode(doc);
                root.addChild(issueNode);
            }
            
        }
        catch (ContentManagementException cme)
        {
            // TODO: logging
            System.err.println("exception building cms list " + cme);
        }
                 
        return treeControl;
    }
    
    public void init(PortletConfig config) throws PortletException
    {
        super.init(config);
                
        cms = ServiceAccessor.getModelService();
        if (cms == null)
        {
            throw new PortletException("Failed to load CMS service");
        }
        loader = new CMSTreeLoader(cms);

    }

                 
}
