/*
 * Copyright 2000-2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.graffito.portlets.tree;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

import org.apache.portals.graffito.ContentModelService;
import org.apache.portals.graffito.exception.ContentManagementException;
import org.apache.portals.graffito.model.CmsObject;
import org.apache.portals.graffito.model.Document;
import org.apache.portals.graffito.model.Folder;
import org.apache.portals.graffito.model.Link;
import org.apache.webapp.admin.TreeControlNode;
import org.apache.portals.graffito.portlets.util.ServiceAccessor;

public class CMSTreeLoader 
{
	
	public final static String PORTLET_URL = "portlet_url";
		
	private ContentModelService cms; 
        
    public CMSTreeLoader(ContentModelService cms)
    {
        this.cms = cms;
    }
    
    public TreeControlNode createFolderNode(Folder folder) throws ContentManagementException
    {
        TreeControlNode node = 
            new TreeControlNode(
                    folder.getUri(), 
                    "folder.gif", 
                    folder.getName(), 
                    null, 
                    null, 
                    false, 
                    "FOLDER_DOMAIN",
                    "Folder",
                    true);

        if(cms.getDocuments(folder.getUri()).size() > 0)
        {
            node.setLeaf(false);
        }
        return node;
    }
    	
	public TreeControlNode createDocumentNode(Document document) throws ContentManagementException
    {
        String title = document.getProperty("displayTitle");
        if (title == null)
            title = document.getName();
        
    	TreeControlNode issueNode = 
            new TreeControlNode(
                    document.getUri(),
                    "document.gif", 
                    title, 
                    PORTLET_URL, 
                    null, 
                    false, 
                    "DOC_DOMAIN",
					"Document",
					true);
    	
    	//if(cms.getLinks(document.getUri()).size() > 0)
    	{
    		issueNode.setLeaf(true);
    	}
    	
    	return issueNode;
    }
    
    public TreeControlNode createLinkNode(Link link) throws ContentManagementException
    {
    	TreeControlNode linkNode = 
            new TreeControlNode(
                    link.getProperty("reference") + link.getProperty("anchor"),
                    "anchorlink.gif", 
                    link.getProperty("displayTitle"), 
                    PORTLET_URL, 
                    null, 
                    false, 
                    "ANCHOR_DOMAIN",
					null,
                    true);
    	
        if(cms.getLinks(link.getUri()).size() > 0)
        {
            linkNode.setLeaf(false);
        }
    	
    	return linkNode;
    }
    
    public TreeControlNode createRootNode(String label, String rootDomain)
    {
    	TreeControlNode root = new TreeControlNode("ROOT-NODE", 
                "folder.gif", 
                label, // label 
                null,       // action
                null,       // target
                true,       // expanded?
                rootDomain,
				null);
    	
    	return root;
    }    
    
    public void loadChildren(TreeControlNode node) throws ContentManagementException
    {
    	String domain = node.getDomain();
    	if(domain.equals("FOLDER_DOMAIN"))
    	{
            // first get folders
	    	Collection subFolders = cms.getFolders(node.getName());
	        Collections.sort((List)subFolders, cmsComparator);
	        Iterator folders = subFolders.iterator();
	        while (folders.hasNext())
	        {
	            Folder folder = (Folder)folders.next();
	            TreeControlNode monthNode = createFolderNode(folder);
	            node.addChild(monthNode);
	        }

            // then get documents
            Collection issueLevel = cms.getDocuments(node.getName());
            Collections.sort((List)issueLevel, cmsComparator);
            Iterator issues = issueLevel.iterator();
            while (issues.hasNext())
            {
                Document doc = (Document)issues.next();
                TreeControlNode issueNode = createDocumentNode(doc);
                node.addChild(issueNode);
                    
                //refToURIMap.put(doc.getProperty("reference"), doc.getUri());
            }
            node.setLoaded(true);
            
    	}
//    	else if(domain.equals("DOC_DOMAIN"))
//    	{
//    		//String uri = (String) refToURIMap.get(node.getName());
//            String uri = node.getName();
//    		Collection links = cms.getLinks(uri);
//    		Iterator iter = links.iterator();
//    		while (iter.hasNext()) {
//				Link link = (Link) iter.next();
//				TreeControlNode linkNode = createLinkNode(link);
//				node.addChild(linkNode);
//                
//                //refToURIMap.put(link.getProperty("reference") + link.getProperty("anchor"), link.getUri());
//			}
//    		node.setLoaded(true);
//    	}
//        else if(domain.equals("ANCHOR_DOMAIN"))
//        {
//            //String uri = (String) refToURIMap.get(node.getName());
//            String uri = node.getName();
//            
//            Collection links = cms.getLinks(uri);
//            Iterator iter = links.iterator();
//            while (iter.hasNext()) {
//                Link link = (Link) iter.next();
//                TreeControlNode linkNode = createLinkNode(link);
//                node.addChild(linkNode);
//                
//              //  refToURIMap.put(link.getProperty("reference") + link.getProperty("anchor"), link.getUri());                
//            }
//            node.setLoaded(true);
//        }
        
    }
	
	
	public static Comparator cmsComparator = new CmsObjectComparator();
	
	public static Comparator getComparator() {
		return cmsComparator;
	}
    
    private static class CmsObjectComparator implements Comparator
    {
        public int compare(Object o1, Object o2)
        {
            CmsObject lhs = (CmsObject)o1;
            CmsObject rhs = (CmsObject)o2;
            
            return rhs.getName().compareTo(lhs.getName());
        }
    }
}
