/*
 * Copyright 2000-2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.graffito.portlets.tree;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.webapp.admin.TreeControl;
import org.apache.webapp.admin.TreeControlNode;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

public class TreeLoaderHandler extends DefaultHandler
{
    public final static String PORTLET_URL = "portlet_url";
    
    private Map yearToTreeNode = new HashMap();
    private Map yearToChildren = new HashMap();
    private Map yearmonthToTreeNode = new HashMap();
    private Map yearmonthToChildren = new HashMap();
    
    private Comparator treeNodeComparator = new TreeNodeComparator();
    
    private TreeControl treeControl;
    private TreeControlNode root = null;
    
    //private int anchorId = 0;
    
    public TreeLoaderHandler()
    {
    }
    
    public void startDocument ()
	throws SAXException
    {
        root = new TreeControlNode("ROOT-NODE", "folder.gif", 
                    "Articles", null, null, true, "ROOT_DOMAIN");
        
        treeControl = new TreeControl(root);
    }
    
    public void endDocument ()
	throws SAXException
    {
        addYearNodesToRoot();
    }
    
    public TreeControl getTreeControl()
    {
        return treeControl;
    }
    
    private void addYearNodesToRoot()
    {
        List yearList = new ArrayList(yearToTreeNode.values());
        Collections.sort(yearList, treeNodeComparator);
        
        Iterator yearIter = yearList.iterator();
        while (yearIter.hasNext())
        {
            TreeControlNode yearNode = (TreeControlNode) yearIter.next();
            root.addChild(yearNode);
            addMonthNodesToYearNode(yearNode);
        }
    }
    
    private void addMonthNodesToYearNode(TreeControlNode yearNode)
    {
        List monthList = (List) yearToChildren.get(yearNode.getLabel());
        Collections.sort(monthList, treeNodeComparator);        
        
        Iterator monthIter = monthList.iterator();
        while (monthIter.hasNext())
        {
            TreeControlNode monthNode = (TreeControlNode)monthIter.next();
            yearNode.addChild(monthNode);
            addIssuesToMonth(yearNode, monthNode);
        }
    }
    
    private void addIssuesToMonth(TreeControlNode yearNode, TreeControlNode monthNode)
    {
        String ymId = yearNode.getLabel() + ":" + monthNode.getLabel();
        List issueList = (List) this.yearmonthToChildren.get(ymId);
        Collections.sort(issueList, treeNodeComparator);
        
        Iterator issueIter = issueList.iterator();
        while (issueIter.hasNext())
        {
            TreeControlNode issueNode = (TreeControlNode)issueIter.next();
            TreeControlNode existingIssueNode = treeControl.findNode(issueNode.getName());
            if(existingIssueNode == null)
            {
                monthNode.addChild(issueNode);
            }
        }
    }
    
    private String getElementName(String localName, String qName)
    {
        String result = null;
        
        if(localName == null || localName.length() == 0)
        {
            result = qName;
        }
        else
        {
            result = localName;
        }
        
        return result;
    }
    
    public void startElement (String uri, String localName,
		      String qName, Attributes attributes)
	throws SAXException
	{
        String element = getElementName(localName, qName);
        if(element.equals("Article"))
        {
            String year = attributes.getValue("year");
            String month = attributes.getValue("month");
            String issue = attributes.getValue("issue");
            String title = attributes.getValue("title");
            String displaytitle = attributes.getValue("displayTitle");
            String reference = attributes.getValue("reference");
            
            String uid = createArticleId(year, month, issue, title);
            
            TreeControlNode yearNode = (TreeControlNode) yearToTreeNode.get(year);
            if(yearNode == null)
            {
                yearNode = new TreeControlNode(year, "folder.gif", 
                        year, null, null, false, "YEAR_DOMAIN");
                
                yearToTreeNode.put(year, yearNode);
            }
            
            String ymId = createArticleId(year, month, null, null);
            
            TreeControlNode monthNode = (TreeControlNode) yearmonthToTreeNode.get(ymId);
            if(monthNode == null)
            {
                monthNode = new TreeControlNode(ymId, "folder.gif", 
                        month, null, null, false, "MONTH_DOMAIN");
                
                yearmonthToTreeNode.put(ymId, monthNode);
                List yearChildren = (List) yearToChildren.get(year);
                if(yearChildren == null)
                {
                    yearChildren = new ArrayList();
                    yearToChildren.put(year, yearChildren);
                }
                yearChildren.add(monthNode);
            }
            
            TreeControlNode issueNode = new TreeControlNode(uid, "document.gif", 
                    title, PORTLET_URL, null, false, "ISSUE_DOMAIN");
            
            //issueNode.setAnchor("" + anchorId);
            //++anchorId;
            
            List children = (List) yearmonthToChildren.get(ymId);
            if(children == null)
            {
                children = new ArrayList();
                yearmonthToChildren.put(ymId, children);
            }
            children.add(issueNode);
        }
	}
    private String createArticleId(String year, String month, String issue, String title)
    {
        StringBuffer buffer = new StringBuffer();
        
        buffer.append(year);
        if(month != null)
        {
            buffer.append(":");
            buffer.append(month);
            if(issue != null)
            {
                buffer.append(":");
                buffer.append(issue);
                if(title != null)
                {
                    buffer.append(":");
                    buffer.append(title);
                }
            }
        }
        
        return buffer.toString();
    }
    
    private class TreeNodeComparator implements Comparator
    {
	    public int compare(Object o1, Object o2)
	    {
	        TreeControlNode lhs = (TreeControlNode)o1;
	        TreeControlNode rhs = (TreeControlNode)o2;
	        
	        return rhs.getLabel().compareTo(lhs.getLabel());
	    }
    }
    
}
